#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import json
import os
import threading
import time
import traceback

from shinken.configuration_incarnation import PartConfigurationIncarnationContainer
from shinken.ipc.share_item import ShareItem
from shinken.load import WindowsValue
from shinken.log import PART_INITIALISATION, LoggerFactory
from shinken.message import Message
from shinken.misc.os_utils import safe_write_file, clean_file_name
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import BaseModule, ModuleState
from shinken.util import DelayedForceMemoryTrimManagerByCounter
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, SeparatorFormat, ConfigurationFormat, TypeConfiguration
from shinkensolutions.locking.fair_lock_ordonnancer import FairLockGroupOrdonnancer

if TYPE_CHECKING:
    from shinken.misc.monitoring_item_manager.regenerator import Regenerator
    from shinken.misc.type_hint import Optional, List, Callable, Any
    from shinken.log import PartLogger
    from shinken.misc.monitoring_item_manager.regenerator import Regenerator
    from shinken.configuration_incarnation import PartConfigurationIncarnation, ConfigurationIncarnation

BROKS_MANAGEMENT_EARLY_LOCK = False
BROKS_MANAGEMENT_ENABLE_CATCHUP = True
BROKS_MANAGEMENT_ALLOWED_LATE_SETS = 10
BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP = 200000
BROKS_MANAGEMENT_CATCHUP_LOOPS = True

BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_THRESHOLD_BEFORE_GC_COLLECT = 20
BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_COUNTER_RESET_PERIOD = 10 * 60  # 10min

# noinspection SpellCheckingInspection
BROKS_GETTER__INCLUDE_DESERIALISATION_AND_CATCHUP_IN_LOCK = '%s__broks_getter__include_deserialisation_and_catchup_in_lock'
BROKS_GETTER__ACTIVATE_LATE_SET_CATCHUP = '%s__broks_getter__activate_late_set_catchup'
BROKS_GETTER__NB_LATE_SET_ALLOWED_BEFORE_CATCHUP = '%s__broks_getter__nb_late_set_allowed_before_catchup'
BROKS_GETTER__CATCHUP_BROKS_MANAGED_BY_MODULE_IN_A_CATCHUP_LOOP = '%s__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop'
BROKS_GETTER__CATCHUP_RUN_ENDLESS_UNTIL_NB_LATE_SET_ALLOWED_REACHED = '%s__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached'

UNAVAILABILITY_MARGIN = 500  # in sec
BASE_RETENTION_FILE_PATH = '/var/lib/shinken/broker_module_retention_stats_%s_%s.json'
MAX_UNAVAILABILITY_TIME_WHEN_NEW_CONFIGURATION_KEEP = 20


class ContextClass:
    def __init__(
            self,
            acquire: 'Optional[Callable[[], None]]' = None,
            release: 'Optional[Callable[[], None]]' = None,
            *,
            has_lock: 'Optional[Callable[[], bool]]' = None
    ) -> None:
        self.acquire = acquire
        self.release = release
        self._has_lock = has_lock
    
    
    def __enter__(self):
        if self.acquire:
            self.acquire()
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.release:
            self.release()
    
    
    def has_lock(self) -> bool:
        if self._has_lock:
            return self._has_lock()
        else:
            raise NotImplementedError('has_lock() callback has not been set')


class BrokerBaseModule(BaseModule, ConfigurationReaderMixin):
    def __init__(self, module_configuration):
        BaseModule.__init__(self, module_configuration)
        
        self._fair_lock_ordonnancer = None  # type: Optional[FairLockGroupOrdonnancer]
        self.consumer_lock = None  # type: Optional[ContextClass]
        self.producer_lock = None  # type: Optional[ContextClass]
        self.rg = None  # type: Optional[Regenerator]
        self._manage_broks_functions = []  # type: List[Callable[[Any], None]]
        self._part_configuration_loaded_hooks = []  # type: List[Callable[[str], None]]
        self._complete_configuration_loaded_hooks = []  # type: List[Callable[[str], None]]
        self._writer_early_lock = BROKS_MANAGEMENT_EARLY_LOCK
        self._writer_catchup_enabled = BROKS_MANAGEMENT_ENABLE_CATCHUP
        self._writer_late_sets_allowed = BROKS_MANAGEMENT_ALLOWED_LATE_SETS
        self._writer_late_broks_max = BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP
        self._writer_catchup_loops = BROKS_MANAGEMENT_CATCHUP_LOOPS
        self.unavailability_duration = WindowsValue(values_ttl=60)
        self.unavailability_time_when_new_configuration = 0
        self.unavailability_start_time_when_new_configuration = None
        self.unavailability_end_time_when_new_configuration = 0
        self.configuration_id_for_last_unavailability = None
        self.known_config_uuid = None
        self.daemon_configuration = ShareItem(key_name='shinken_broker_daemon_configuration_for_modules', reinit=False)
        self.rg = None  # type: Optional[Regenerator]
        self.monitoring_configuration_part_process = PartConfigurationIncarnationContainer()
        self._last_unavailability_time_when_new_configuration = {}
        self._last_unavailability_time_when_new_configuration_lock = threading.RLock()
        self._unavailability_duration_lock = threading.RLock()
        
        self.time_before_warning_when_reloading_new_configuration = int(getattr(module_configuration, 'broker__module_webui__loading_new_configuration__time_before_warning_if_load_is_long', '10'))
        self.is_configuration_fully_load = True
        self.is_configuration_loading = True
        self._total_nb_part_to_load = 0
        self._nb_part_loaded = 0
        
        if hasattr(self, 'module_type'):
            module_type = self.module_type
        else:
            module_type = getattr(self.myconf, 'module_type', None)
        
        log_init = self.logger.get_sub_part(PART_INITIALISATION, len(PART_INITIALISATION))
        
        if not module_type:
            log_init.warning('unknown module type, broks management will run with default values')
            return
        
        if module_type == 'broker_module_livedata':
            module_type_cfg_format = 'broker__module_livedata'
        else:
            module_type_cfg_format = 'broker__module_%s' % module_type
        
        configuration_format = [
            SeparatorFormat('BROKS GETTER'),
            ConfigurationFormat([BROKS_GETTER__INCLUDE_DESERIALISATION_AND_CATCHUP_IN_LOCK % module_type_cfg_format], BROKS_MANAGEMENT_EARLY_LOCK, TypeConfiguration.BOOL, '_writer_early_lock'),
            ConfigurationFormat([BROKS_GETTER__ACTIVATE_LATE_SET_CATCHUP % module_type_cfg_format], BROKS_MANAGEMENT_ENABLE_CATCHUP, TypeConfiguration.BOOL, '_writer_catchup_enabled'),
            ConfigurationFormat([BROKS_GETTER__NB_LATE_SET_ALLOWED_BEFORE_CATCHUP % module_type_cfg_format], BROKS_MANAGEMENT_ALLOWED_LATE_SETS, TypeConfiguration.INT, '_writer_late_sets_allowed'),
            ConfigurationFormat([BROKS_GETTER__CATCHUP_BROKS_MANAGED_BY_MODULE_IN_A_CATCHUP_LOOP % module_type_cfg_format], BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP, TypeConfiguration.INT, '_writer_late_broks_max'),
            ConfigurationFormat([BROKS_GETTER__CATCHUP_RUN_ENDLESS_UNTIL_NB_LATE_SET_ALLOWED_REACHED % module_type_cfg_format], BROKS_MANAGEMENT_CATCHUP_LOOPS, TypeConfiguration.BOOL, '_writer_catchup_loops'),
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, self.myconf, log_init)
        self.read_configuration()
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
    
    
    def loop_turn(self):
        pass
    
    
    def lock_init(
            self,
            broks_eater,
            fair_lock_ordonnancer_name='ITEMS ACCESS ORDONNANCER',
            consumer_name='HTTP requests',
            producer_name='Broks management',
            consumer_max_switch_time=1,
            producer_max_switch_time=1,
            error_log_time=30,
            logger=None
    ):
        # type: (List[Callable[[Any], None]], str, str, str, int, int, int, Optional[PartLogger]) -> None
        
        self._manage_broks_functions = broks_eater
        
        if not logger:
            logger = self.logger
        
        self._fair_lock_ordonnancer = FairLockGroupOrdonnancer(
            name=fair_lock_ordonnancer_name,
            consumer_name=consumer_name,
            consumer_max_wish_switch_time=consumer_max_switch_time,
            producer_name=producer_name,
            producer_max_wish_switch_time=producer_max_switch_time,
            logger=logger,
            error_log_time=error_log_time
        )
        
        self.consumer_lock = ContextClass(acquire=self._fair_lock_ordonnancer.consumer_acquire, release=self._fair_lock_ordonnancer.consumer_release, has_lock=self._fair_lock_ordonnancer.consumer_has_lock)
        self.producer_lock = ContextClass(acquire=self._fair_lock_ordonnancer.producer_acquire, release=self._fair_lock_ordonnancer.producer_release, has_lock=self._fair_lock_ordonnancer.producer_has_lock)
    
    
    def init_configuration_load_hooks(self, part_configuration_loaded_hooks, complete_configuration_loaded_hooks):
        # type: (List[Callable[[str], None]], List[Callable[[str], None]]) -> None
        self._part_configuration_loaded_hooks = list(part_configuration_loaded_hooks)
        self._complete_configuration_loaded_hooks = list(complete_configuration_loaded_hooks)
    
    
    def do_loop_turn(self):
        raise NotImplementedError()
    
    
    def _die_with_strong_error(self, error, with_trace_back=True):
        crash_logger = LoggerFactory.get_logger('CRASH - INSIDE MODULE PROCESS')
        crash_logger.error(error)
        if isinstance(error, Exception):
            error = str(error)
        traceback_message = traceback.format_exc() if with_trace_back else error
        if with_trace_back:
            crash_logger.print_stack()
        msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': error, 'trace': traceback_message})
        self.from_module_to_main_daemon_queue.put(msg)
        # wait 2 sec, so we know that the broker got our message, and die
        time.sleep(2)
        os._exit(2)  # noqa : forced exit
    
    
    # The manage brok thread will consume broks, but if it dies, broks will stack in the queue manager
    # This is not acceptable, and such error means a strong die with log in healthcheck
    def manage_brok_thread(self):
        try:
            self._manage_brok_thread()
        except Exception as exp:  # noqa: catch all to die
            self._die_with_strong_error(str(exp))
    
    
    @staticmethod
    def _make_memory_trim_manager_for_brok_management(minimum_ask_number_before_force_trim):
        # type: (int) -> DelayedForceMemoryTrimManagerByCounter
        return DelayedForceMemoryTrimManagerByCounter(
            minimum_ask_number_before_force_trim=minimum_ask_number_before_force_trim,
            counter_reset_period=BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_COUNTER_RESET_PERIOD,
        )
    
    
    def _manage_brok_thread(self):
        have_lock = False
        logger_unavailability = LoggerFactory.get_unavailability_logger()
        logger_loading_configuration = LoggerFactory.get_loading_configuration_logger()
        logger_perf = self.logger.get_sub_part('MANAGE BROKS', len('MANAGE BROKS')).get_sub_part('PERF', len('PERF'))
        logger_late_broks_sets = self.logger.get_sub_part('MANAGE BROKS', len('MANAGE BROKS')).get_sub_part('LATE BROKS SETS', len('LATE BROKS SETS'))
        have_received_initial_brok_start_or_end = False
        have_loaded_initial_broks = False
        unavailability_start_time = time.time()
        delayed_force_memory_trim_by_counter = self._make_memory_trim_manager_for_brok_management(BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_THRESHOLD_BEFORE_GC_COLLECT)
        while True:
            start = time.time()
            broks = self.to_q.get()
            
            if self._writer_early_lock and not have_lock:
                self._fair_lock_ordonnancer.producer_acquire()
                have_lock = True
                unavailability_start_time = time.time()
            
            get_late_broks_set_time = time.time()
            nb_late_broks_sets_taken = 0
            if self._writer_catchup_enabled and (self.to_q.qsize() > self._writer_late_sets_allowed >= 0):
                t1 = time.time()
                while self.to_q.qsize() > 0:
                    t0 = time.time()
                    _late_broks_set = self.to_q.get()
                    _current_nb_late_broks = len(_late_broks_set)
                    broks.extend(_late_broks_set)
                    broks_to_process = len(broks)
                    logger_late_broks_sets.info('Getting brok set with %s broks in %.3fs [time for read queue size=%.3fs]. Total broks to process= %s/max:%s. Broks sets in queue: %s.' %
                                                (_current_nb_late_broks, time.time() - t0, t0 - t1, broks_to_process, self._writer_late_broks_max, self.to_q.qsize()))
                    nb_late_broks_sets_taken += 1
                    t1 = time.time()
                    
                    if broks_to_process > self._writer_late_broks_max:
                        logger_late_broks_sets.info('Late brok taken => limit reach : %s / limit: %s.' % (broks_to_process, self._writer_late_broks_max))
                        break
            
            after_get = time.time()
            # First unpickle broks, but outside the lock time
            for brok in broks:
                brok.prepare()
            after_prepare = time.time()
            
            # For updating, we cannot do it while answering queries, so wait for no readers
            if not self._writer_early_lock and not have_lock:
                self._fair_lock_ordonnancer.producer_acquire()
                have_lock = True
                unavailability_start_time = time.time()
            
            last_configuration_incarnation_received_by_arbiter = self.daemon_configuration.configuration_incarnation
            if last_configuration_incarnation_received_by_arbiter:
                config_incarnation_uuid = last_configuration_incarnation_received_by_arbiter.get_uuid()
                if self.known_config_uuid != config_incarnation_uuid:
                    total_nb_part_to_load = len(self.daemon_configuration.all_monitoring_configuration_part)
                    logger_loading_configuration.info('The configuration 〖%s〗 from the arbiter 〖%s〗 must be loaded with 〖%s〗 monitoring configuration parts.' %
                                                      (config_incarnation_uuid, last_configuration_incarnation_received_by_arbiter.get_author(), total_nb_part_to_load))
                    logger_loading_configuration.debug('Monitoring configuration part id to load:〖%s〗' % ', '.join(['%s-%s' % (i, config_incarnation_uuid) for i in self.daemon_configuration.all_monitoring_configuration_part]))
                    self.known_config_uuid = config_incarnation_uuid
                    self.is_configuration_loading = True
                    self.is_configuration_fully_load = False
                    self._total_nb_part_to_load = total_nb_part_to_load
                    self._nb_part_loaded = 0
                    if self.monitoring_configuration_part_process.get_configuration_loaded_uuid() == config_incarnation_uuid:
                        self._nb_part_loaded = len(self.monitoring_configuration_part_process.get_configuration_incarnation_part_from_loaded_configuration())
                        self.is_configuration_fully_load = self._total_nb_part_to_load == self._nb_part_loaded
                        self.is_configuration_loading = not self.is_configuration_fully_load
                    
                    minimum_ask_number_before_force_trim = self._total_nb_part_to_load - self._nb_part_loaded
                    if minimum_ask_number_before_force_trim <= 0:
                        minimum_ask_number_before_force_trim = BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_THRESHOLD_BEFORE_GC_COLLECT
                    else:
                        minimum_ask_number_before_force_trim = min(minimum_ask_number_before_force_trim, BROKS_MANAGEMENT_FORCE_MEMORY_TRIM_THRESHOLD_BEFORE_GC_COLLECT)
                    delayed_force_memory_trim_by_counter = self._make_memory_trim_manager_for_brok_management(minimum_ask_number_before_force_trim)
                    
                    if self._nb_part_loaded > 0:
                        # This condition is true when the schedulers have sent the shards before the broker received the configuration from the arbiter.
                        # The hooks call was postponed until the arbiter sends the configuration to the broker. Therefore, call the hooks now.
                        self._call_configuration_loaded_hooks(config_incarnation_uuid)
            
            after_wait_write_lock = time.time()
            broks_type_counter = {}
            broks_type_times = {}
            for brok in broks:
                brok_part_configuration_incarnation = brok.part_configuration_incarnation
                if brok.type == 'program_status':
                    part_log_message = brok_part_configuration_incarnation.build_log_message()
                    logger_loading_configuration.info('A new monitoring configuration part from the scheduler 〖%s〗 have been received. %s' % (brok_part_configuration_incarnation.scheduler_name, part_log_message))
                    have_received_initial_brok_start_or_end = True
                    logger_unavailability.info('The module may be unavailable while the new monitoring configuration part is being loaded. %s' % part_log_message)
                    self.is_configuration_loading = True
                    self.monitoring_configuration_part_process.add_part_received(brok_part_configuration_incarnation)
                
                try:
                    t0 = time.time()
                    for manage_broks_function in self._manage_broks_functions:
                        manage_broks_function(brok)
                    broks_type_counter[brok.type] = broks_type_counter.get(brok.type, 0) + 1
                    broks_type_times[brok.type] = broks_type_times.get(brok.type, 0.0) + (time.time() - t0)
                except Exception as exp:
                    self._die_with_strong_error(str(exp))  # no hope, cannot survive this
                
                if brok.type == 'initial_broks_done':
                    logger_loading_configuration.info(
                        'The monitoring configuration part from the scheduler 〖%s〗 have been processed. %s' % (brok_part_configuration_incarnation.scheduler_name, brok_part_configuration_incarnation.build_log_message()))
                    have_received_initial_brok_start_or_end = True
                    have_loaded_initial_broks = True
                    self.monitoring_configuration_part_process.add_part_loaded(brok_part_configuration_incarnation)
                    delayed_force_memory_trim_by_counter.ask_for_memory_trim(context='because the module >%s< have received a new monitoring configuration' % self.name)
            
            end = time.time()
            logger_perf.info('[ %4d broks ] [ wait and get first set on queue=%.3fs ] [ get %s late sets on=%.3fs ] [ deserialization=%.3fs ] [ wait write lock=%.3fs ] [ manage broks=%.3fs ] [ total=%.3fs ]' % (
                len(broks), get_late_broks_set_time - start, nb_late_broks_sets_taken, after_get - get_late_broks_set_time, after_prepare - after_get, after_wait_write_lock - after_prepare, end - after_wait_write_lock, end - start))
            
            message = []
            for b_type, count in broks_type_counter.items():
                message.append('[%s=%s]' % (b_type, count))
            logger_perf.info('                 => handled broks -> count by types : %s' % ' '.join(message))
            
            message = []
            for b_type, count in broks_type_times.items():
                message.append('[%s=%0.3f]' % (b_type, count))
            logger_perf.info('                 => handled broks -> time by types : %s' % ' '.join(message))
            
            try:
                for _detail_timer_name, _detail_timer in self.rg.detail_timer.items():
                    logger_detail_timer = logger_perf.get_sub_part(_detail_timer_name, register=False)
                    for sub_part_name, sub_part_time in _detail_timer.items():
                        logger_detail_timer.debug('sub_part : time for %s [ %0.3f ]' % (sub_part_name, sub_part_time))
            except:
                # the broker module cannot have any regenerator
                pass
            
            # if we are late we don't release the lock, and we continue to process broks
            if self._writer_catchup_enabled and self._writer_catchup_loops:
                qsize = self.to_q.qsize()
                if qsize > self._writer_late_sets_allowed >= 0:
                    logger_perf.info('Number of Broks sets still in queue after managing broks is %s. We keep the lock and continue the brok managing.' % qsize)
                    continue
            
            if have_lock:
                # We can release the lock as a writer
                self._fair_lock_ordonnancer.producer_release()
                have_lock = False
                now = time.time()
                unavailability_duration = now - unavailability_start_time
                with self._unavailability_duration_lock:
                    self.unavailability_duration.push_value(unavailability_duration)
                
                if now - self.unavailability_end_time_when_new_configuration > UNAVAILABILITY_MARGIN:
                    self.unavailability_start_time_when_new_configuration = unavailability_start_time
                    self.unavailability_time_when_new_configuration = 0
                
                configuration_loaded_uuid = self.monitoring_configuration_part_process.get_configuration_loaded_uuid()
                if self.configuration_id_for_last_unavailability != configuration_loaded_uuid:
                    self.unavailability_start_time_when_new_configuration = unavailability_start_time
                    self.unavailability_time_when_new_configuration = 0
                    self.configuration_id_for_last_unavailability = configuration_loaded_uuid
                
                if have_received_initial_brok_start_or_end:
                    self.unavailability_time_when_new_configuration += unavailability_duration
                    have_received_initial_brok_start_or_end = False
                    new_configuration_part_loaded = have_loaded_initial_broks
                    have_loaded_initial_broks = False
                    
                    last_configuration_incarnation_received_by_arbiter = self.daemon_configuration.configuration_incarnation  # type: ConfigurationIncarnation
                    last_monitoring_configuration_part_received_by_scheduler = self.monitoring_configuration_part_process.last_part_configuration_incarnation_received
                    monitoring_configuration_parts_id_to_load_received_by_arbiter = self.daemon_configuration.all_monitoring_configuration_part
                    
                    if last_configuration_incarnation_received_by_arbiter and last_monitoring_configuration_part_received_by_scheduler and last_configuration_incarnation_received_by_arbiter.get_uuid() == last_monitoring_configuration_part_received_by_scheduler.get_uuid():
                        monitoring_configuration_part_total_to_load = {(i, last_configuration_incarnation_received_by_arbiter.get_uuid()) for i in monitoring_configuration_parts_id_to_load_received_by_arbiter}
                    else:
                        monitoring_configuration_part_total_to_load = set()
                    
                    monitoring_configuration_part_loaded = {(i.get_part_id(), i.get_uuid()) for i in self.monitoring_configuration_part_process.get_configuration_incarnation_part_from_loaded_configuration()}
                    
                    conf_by_arbiter_log_message = last_configuration_incarnation_received_by_arbiter.build_log_message() if last_configuration_incarnation_received_by_arbiter else 'Configuration from Arbiter not received'
                    logger_loading_configuration.debug('Last configuration incarnation received by Arbiter : %s' % conf_by_arbiter_log_message)
                    if last_monitoring_configuration_part_received_by_scheduler:
                        conf_by_scheduler_log_message = last_monitoring_configuration_part_received_by_scheduler.build_log_message()
                        logger_loading_configuration.debug('Last monitoring configuration part received by Scheduler : %s' % conf_by_scheduler_log_message)
                    logger_loading_configuration.debug('Monitoring configuration part to load  :[%s]' % ', '.join(['%s-%s' % (i[0], i[1]) for i in monitoring_configuration_part_total_to_load]))
                    logger_loading_configuration.debug('Monitoring configuration part loaded   :[%s]' % ', '.join(['%s-%s' % (i[0], i[1]) for i in monitoring_configuration_part_loaded]))
                    
                    nb_part_loaded = len(monitoring_configuration_part_loaded)
                    total_nb_conf_to_load = len(monitoring_configuration_part_total_to_load)
                    self._nb_part_loaded = nb_part_loaded
                    if monitoring_configuration_part_total_to_load == monitoring_configuration_part_loaded:
                        logger_loading_configuration.info('All 〖%s〗 monitoring configuration parts from schedulers have been loaded' % nb_part_loaded)
                        self.is_configuration_fully_load = True
                        self.is_configuration_loading = False
                    elif monitoring_configuration_part_total_to_load:
                        logger_loading_configuration.info('〖%s〗 / 〖%s〗 monitoring configuration parts from schedulers have been loaded' % (nb_part_loaded, total_nb_conf_to_load))
                        self.is_configuration_fully_load = False
                        self.is_configuration_loading = False
                    else:
                        logger_loading_configuration.info('〖%s〗 monitoring configuration parts from schedulers have been loaded' % nb_part_loaded)
                    
                    conf_by_scheduler_no_part_log_message = 'unknown configuration'
                    if last_monitoring_configuration_part_received_by_scheduler:
                        self.save_last_unavailability(last_monitoring_configuration_part_received_by_scheduler, nb_part_loaded, total_nb_conf_to_load, unavailability_start_time)
                        conf_by_scheduler_no_part_log_message = last_monitoring_configuration_part_received_by_scheduler.get_configuration_incarnation().build_log_message()
                    
                    part_message = '〖%s〗/〖%s〗' % (nb_part_loaded, total_nb_conf_to_load) if monitoring_configuration_part_total_to_load else '〖%s〗' % nb_part_loaded
                    
                    logger_unavailability.info(
                        'The module %s is now available. Unavailability started at 〖%s〗 took:〖%0.3fs〗 to load %s part of configuration %s.' %
                        (
                            self.get_name(),
                            logger_unavailability.format_time(self.unavailability_start_time_when_new_configuration),
                            unavailability_duration,
                            part_message,
                            conf_by_scheduler_no_part_log_message
                        )
                    )
                    self.unavailability_end_time_when_new_configuration = now
                    
                    # The uuids are different when a scheduler sends a shard before the broker received the configuration from the arbiter.
                    # In this case, the hooks call is postponed.
                    if new_configuration_part_loaded and configuration_loaded_uuid == self.known_config_uuid:
                        self._call_configuration_loaded_hooks(configuration_loaded_uuid)
    
    
    def _call_configuration_loaded_hooks(self, configuration_loaded_uuid):
        # type: (str) -> None
        for configuration_loaded_hook in self._part_configuration_loaded_hooks:
            configuration_loaded_hook(configuration_loaded_uuid)
        
        if self.is_configuration_fully_load:
            for configuration_loaded_hook in self._complete_configuration_loaded_hooks:
                configuration_loaded_hook(configuration_loaded_uuid)
    
    
    def on_init(self):
        # type: () -> None
        with self._last_unavailability_time_when_new_configuration_lock:
            file_path = clean_file_name(BASE_RETENTION_FILE_PATH % (self.daemon_display_name, self.get_name()))
            if not os.path.exists(file_path):
                return
            with open(file_path, 'r') as fd:
                self._last_unavailability_time_when_new_configuration = json.load(fd)
    
    
    def save_last_unavailability(self, last_monitoring_configuration_part_received_by_scheduler, nb_conf_loaded, total_nb_conf_to_load, unavailability_start_time):
        # type: (PartConfigurationIncarnation, int, int, float) -> None
        with self._last_unavailability_time_when_new_configuration_lock:
            unavailability_id = '%s-%s' % (last_monitoring_configuration_part_received_by_scheduler.get_uuid(), self.unavailability_start_time_when_new_configuration)
            # unavailability_id is a monitoring configuration id and a counter because we can have multiple unavailability with same monitoring configuration id
            # like when Arbiter push a new configuration when a scheduler have been reboot (see #SESUP-1671)
            info_unavailability_time_when_new_configuration = last_monitoring_configuration_part_received_by_scheduler.get_configuration_incarnation().dump_as_json()
            info_unavailability_time_when_new_configuration['unavailability_time_when_new_configuration'] = self.unavailability_time_when_new_configuration
            info_unavailability_time_when_new_configuration['nb_conf_handle'] = nb_conf_loaded
            info_unavailability_time_when_new_configuration['nb_conf_to_handle'] = total_nb_conf_to_load
            info_unavailability_time_when_new_configuration['unavailability_start_time'] = self._last_unavailability_time_when_new_configuration.get(unavailability_id, {}).get('unavailability_start_time', unavailability_start_time)
            self._last_unavailability_time_when_new_configuration[unavailability_id] = info_unavailability_time_when_new_configuration
            
            to_del = [(k, v['creation_date']) for k, v in self._last_unavailability_time_when_new_configuration.items()]
            to_del.sort(key=lambda x: x[1], reverse=True)
            for uuid, _ in to_del[MAX_UNAVAILABILITY_TIME_WHEN_NEW_CONFIGURATION_KEEP:]:
                self._last_unavailability_time_when_new_configuration.pop(uuid)
            
            file_path = clean_file_name(BASE_RETENTION_FILE_PATH % (self.daemon_display_name, self.get_name()))
            safe_write_file(file_path, json.dumps(self._last_unavailability_time_when_new_configuration))
    
    
    def get_module_info(self):
        
        if self.monitoring_configuration_part_process.last_part_configuration_incarnation_loaded and not self.is_configuration_fully_load and not self.is_configuration_loading:
            with self._last_unavailability_time_when_new_configuration_lock:
                unavailability_id = '%s-%s' % (self.monitoring_configuration_part_process.last_part_configuration_incarnation_loaded.get_uuid(), self.unavailability_start_time_when_new_configuration)
                unavailability_start_time = self._last_unavailability_time_when_new_configuration.get(unavailability_id, {}).get('unavailability_start_time', 0)
                if time.time() - unavailability_start_time > (self.time_before_warning_when_reloading_new_configuration * 60):
                    return {
                        'status': ModuleState.WARNING,
                        'output': 'The configuration is not fully loaded, some parts are not loaded ( %s/%s )' % (self._nb_part_loaded, self._total_nb_part_to_load)
                    }
        return {
            'status': ModuleState.OK,
            'output': 'OK'
        }
    
    
    def _internal_get_raw_stats(self, param='', module_wanted=''):
        result = super(BrokerBaseModule, self)._internal_get_raw_stats(param, module_wanted)
        result['unavailability_time_when_new_configuration'] = self.unavailability_time_when_new_configuration
        with self._unavailability_duration_lock:
            result['unavailability_duration'] = self.unavailability_duration.get_sum_value()
        with self._last_unavailability_time_when_new_configuration_lock:
            result['last_unavailability_time_when_new_configuration'] = self._last_unavailability_time_when_new_configuration
        if self.rg:
            result['item_counter'] = self.rg.item_counter
        
        return result
