#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import copy
import functools
import hashlib
import os
import re
import threading
import traceback

from shinken.log import logger, LoggerFactory, LOG_SECTION_SIZE
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.inventory import ElementsInventory
from shinken.objects.item import Item
from shinken.objects.itemgroup import Itemgroup, Itemgroups
from shinken.objects.shard import Shard
from shinken.property import BoolProp, IntegerProp, StringProp, RawProp
from shinken.util import alive_then_spare_then_deads, to_bool
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List, Dict, Any, Union
    from shinken.schedulerlink import SchedulerLink
    from shinken.pollerlink import PollerLink
    from shinken.reactionnerlink import ReactionnerLink
    from shinken.brokerlink import BrokerLink
    from shinken.receiverlink import ReceiverLink
    from shinken.satellitelink import SatelliteLink
    from .pack import Packs

try:
    from texttable import Texttable
except ImportError:  # windows
    Texttable = None

MAX_REALMS_LEVEL = 10

SECTION_SPACE_SIZE = LOG_SECTION_SIZE + 3

INVALID_CHARACTERS_REALM = {
    'regex': re.compile(r'''["'<>=:,]'''),
    'value': u'''"'<>=:,'''
}

__SHINKEN_REALM_HAS_NO_NAME__ = '__shinken_realm_has_no_name__'

# If there is a problem with the new broker spare computation, the admin can fail back to the old
# computation with this flag
FLAG_DISABLE_NEW_BROKER_SPARE_COMPUTATION = os.environ.get('FLAG_DISABLE_NEW_BROKER_SPARE_COMPUTATION', '0') == '1'

# It change from hostgroup Class because there is no members
# properties, just the realm_members that we rewrite on it.

SATELLITE_KINDS = ('reactionner', 'poller', 'broker', 'receiver')
SATELLITE_KINDS_STR = ', '.join(SATELLITE_KINDS)


class Realm(Itemgroup):
    id = 1  # zero is always a little bit special... like in database
    my_type = 'realm'
    
    # Typing
    schedulers = []  # type: List[SchedulerLink]
    pollers = []  # type: List[PollerLink]
    reactionners = []  # type: List[ReactionnerLink]
    brokers = []  # type: List[BrokerLink]
    receivers = []  # type: List[ReceiverLink]
    shards = None  # type: Dict[int, Shard]
    to_satellites = None  # type: Dict[str, Dict[Any, Any]]
    configuration_errors = []  # type: List[str]
    higher_realms = []  # type: List[Realm]
    
    satellite_kinds = SATELLITE_KINDS
    satellite_kinds_str = SATELLITE_KINDS_STR
    
    properties = Itemgroup.properties.copy()
    properties.update({
        'id'                   : IntegerProp(default=0, fill_brok=['full_status']),
        'realm_name'           : StringProp(fill_brok=['full_status']),
        'realm_members'        : StringProp(default=''),  # No status_broker_name because it put hosts, not host_name
        'higher_realms'        : StringProp(default=''),
        'default'              : BoolProp(default='0'),
        'broker_complete_links': BoolProp(default='0'),
        'only_schedulers'      : BoolProp(default='0'),
    })
    
    running_properties = Item.running_properties.copy()
    running_properties.update({
        'shards'                : RawProp(default={}),
        'sharded_configurations': RawProp(default={}),
        'higher_realms'         : StringProp(default=[]),
        'original_realm_members': RawProp(default=[]),
    })
    
    macros = {
        'REALMNAME'   : 'realm_name',
        'REALMMEMBERS': 'members',
    }
    
    static_macros = set()
    
    
    def __init__(self, params=None, skip_useless_in_configuration=False):
        if TYPE_CHECKING:
            self.realm_name = None  # type: Optional[str]
            self.realm_members = None  # type: Optional[str]
            self.default = None  # type: Optional[bool]
            self.broker_complete_links = None  # type: Optional[bool]
            self.only_schedulers = None  # type: Optional[bool]
            self.packs = None  # type: Optional[Packs]
            self.id = 0
            self.higher_realms = None  # type: Optional[List[Realm]]
            self.default = False
            self.original_realm_members = None  # type: Optional[Union[List[str],List]]
            self._already_explode = False
            self.nb_reactionners = 0
            self.potential_reactionners = []
            self.nb_pollers = 0
            self.potential_pollers = []
            self.nb_brokers = 0
            self.potential_brokers = []
            self.nb_receivers = 0
            self.potential_receivers = []
            self.structure_lock = None  # type: Optional[threading.RLock]
        
        super(Realm, self).__init__(params=params, skip_useless_in_configuration=skip_useless_in_configuration)
        # We will keep a list of all the hosts and checks we have with basic information, mainly for receivers & brokers
        self._host_and_check_receiver_inventory = ElementsInventory()
        self._host_and_check_broker_inventory = ElementsInventory()
        self._got_loop = False
        self._realm_name_hash = None  # type: Optional[str]
        self._create_loggers()
    
    
    # VERY IMPORTANT: when adding a new logger, ADD IT to the getstate exclusion list
    def _create_loggers(self):
        self.logger = LoggerFactory.get_logger().get_sub_part('REALM: %s' % self.get_name())
        self.logger_dispatch = self.logger.get_sub_part('DISPATCH', part_name_size=8)
        self.logger_configuration = self.logger_dispatch.get_sub_part('CONFIGURATION')
    
    
    # Call by pickle for dataify the realm, but we do not want the lock (not picklelisable)
    # IMPORTANT: we don't want lock and loggers, as we do not allow them to be pickled
    def __getstate__(self):
        res = self.__dict__.copy()
        to_del = ('structure_lock', 'logger', 'logger_dispatch', 'logger_configuration')
        for prop in to_del:
            del res[prop]
        return res
    
    
    # Inverted function of getstate
    def __setstate__(self, state):
        self.__dict__.update(state)
        # Recreate the lock
        self.structure_lock = threading.RLock()
        # And the logger too, because pickle remove them
        self._create_loggers()
    
    
    def get_name(self):
        return getattr(self, 'realm_name', __SHINKEN_REALM_HAS_NO_NAME__)
    
    
    def get_name_hash(self):
        # type: () -> str
        if self._realm_name_hash is None:
            name = self.get_name()
            if name == __SHINKEN_REALM_HAS_NO_NAME__:
                self._realm_name_hash = ''
            else:
                if isinstance(name, str):
                    encode_string = name.encode('utf-8', 'ignore')
                else:
                    encode_string = name
                self._realm_name_hash = hashlib.sha256(encode_string).hexdigest()
        
        return self._realm_name_hash
    
    
    def get_realms(self):
        return self.realm_members
    
    
    def got_loop(self):
        return self._got_loop
    
    
    def add_string_member(self, member):
        if member:
            self.realm_members += ',' + member
    
    
    def remove_string_member(self, member):
        members_list = self.get_realm_members()
        if member in members_list:
            members_list.remove(member)
            self.realm_members = ','.join(members_list)
    
    
    def get_realm_members(self):
        if self.has('realm_members'):
            return [r.strip() for r in self.realm_members.split(',') if r.strip()]
        else:
            return []
    
    
    # This realm can have special issues:
    # * multiples active schedulers (potential retentions issues):
    #   * if no retention
    #   * if retention module that is not mongodb (distributed)  TODO: how to detect this?
    #   * if retention mongodb BUT not the same address
    #   * if retention mongodb BUT still localhost address (need public IP)
    # * a scheduler without a broker is an error
    def is_correct(self):
        # type: () -> bool
        _to_return = True
        
        self._is_name_correct()
        active_schedulers = [s for s in self.schedulers if not s.spare]
        nb_active_sched = len(active_schedulers)
        
        # Allowed retention modules are in the list:
        # * pickle flat file : pickle_retention_file (or pickle-retention-file)
        # * mongodb : mongodb_retention (or mongodb-retention)
        is_distributed = (nb_active_sched >= 2)
        
        all_mongodb_modules = set()
        for sched in self.schedulers:
            modules = sched.modules
            # if no modules at all, get out, because no retention is set
            if len(modules) == 0:
                self.configuration_errors.append("The scheduler %s does not have any retention module and so it won't be able to save and load its retention data. Please add a scheduler retention module." % sched.get_name())
                continue
            for mod in modules:
                if mod.module_type == 'mongodb_retention':
                    all_mongodb_modules.add(mod)
            mod_types = [mod.module_type.replace('-', '_') for mod in modules]  # beware: framework 2.4 module are with - instead of _
            nb_pickle = mod_types.count('pickle_retention_file')
            nb_mongo = mod_types.count('mongodb_retention')
            if is_distributed and nb_pickle >= 1:
                self.configuration_errors.append(
                    'The scheduler %s has a pickle_retention_file module but is in a distributed realm with several schedulers. This module is incompatible for those environments as the retention file is not shared across servers. Please use a mongodb_retention type module instead.' % (
                        sched.get_name()))
                continue
            if is_distributed and nb_mongo < 1:
                self.configuration_errors.append('The scheduler %s is in a distributed realm (%d active schedulers) but do not have any mongodb retention module. It will break the retention data.' % (sched.get_name(), nb_active_sched))
                continue
        
        # Now all individual cases for schedulers are done, look at if the mongodb retention modules conf is ok in a distributed env, should not be with localhost
        if is_distributed:
            for module in all_mongodb_modules:
                uri = getattr(module, 'mongodb_retention__database__uri', getattr(module, 'mongodb_uri', getattr(module, 'uri', None)))
                bypass_banning_localhost_uri = getattr(module, 'mongodb_retention__database__bypass_banning_localhost_uri', '0')
                if uri is None:
                    self.configuration_errors.append(
                        'The %s module is a mongodb retention module, but without a "mongodb_retention__database__uri" parameter. The default value "%s" cannot be used in a distributed realm ( %d actives schedulers ). Please add one.' % (
                            module.get_name(), 'mongodb://localhost/?w=1&fsync=false', nb_active_sched))
                    continue
                
                if bypass_banning_localhost_uri == '0' and ('localhost' in uri or '127.0.0.1' in uri):
                    self.configuration_errors.append(
                        'The %s module is a mongodb retention module, but configured with localhost URI. In a distributed realm ( %d active schedulers ), all retention modules in that realm must be set to the same server. Please specify the IP address of the mongodb retention server. If this is not an error you can add the option "mongodb_retention__database__bypass_banning_localhost_uri" to your module configuration.' % (
                            module.get_name(), nb_active_sched))
                    continue
        
        _to_return &= super(Realm, self).is_correct()
        return _to_return
    
    
    def _is_name_correct(self):
        _name = self.get_name()
        if _name == __SHINKEN_REALM_HAS_NO_NAME__:
            self.is_correct__append_in_configuration_errors('A realm definition in file %s has no name, please ensure the key "realm_name" is defined' % self.imported_from)
            return False
        if not self.is_name_valid(_name):
            self.is_correct__append_in_configuration_errors('''Forbidden characters %s found in the name of realm "%s" for "%s"''' % (ToolsBoxString.format_for_message(INVALID_CHARACTERS_REALM['value']), _name, self.imported_from))
            return False
        return True
    
    
    @staticmethod
    def is_name_valid(to_test):
        return not INVALID_CHARACTERS_REALM['regex'].search(to_test)
    
    
    # We want to see if non-void schedulers (with hosts) are reachable
    def is_correct_schedulers_with_hosts_and_satellites(self):
        if self.only_schedulers:
            return True
        
        nb_hosts_in_realm = 0
        for shard in self.shards.values():
            nb_hosts_in_realm += len(shard.hosts)
        
        _errors = []
        if nb_hosts_in_realm != 0:
            _no_daemon_err = "The scheduler %s (in the realm %s) have elements to monitor but has no %s in its realm (or upper realm with the manage_sub_realms option enabled)"
        else:
            _no_daemon_err = "The scheduler %s (in the realm %s) has no %s in its realm (or upper realm with the manage_sub_realms option enabled). This is just a warning because it does not have elements to monitor but it will be an error if you add hosts or clusters in this realm"
        # Look if we have at least a broker for schedulers
        for sched in self.schedulers:
            sched_name = sched.get_name()
            realm_name = self.get_name()
            if len(self.potential_brokers) == 0:
                _errors.append(_no_daemon_err % (sched_name, realm_name, 'brokers'))
            
            if len(self.potential_pollers) == 0:
                _errors.append(_no_daemon_err % (sched_name, realm_name, 'pollers'))
            
            if len(self.potential_reactionners) == 0:
                _errors.append(_no_daemon_err % (sched_name, realm_name, 'reactionners'))
        
        # All is great
        if len(_errors) == 0:
            return True
        
        # Oups, can be a realm error if the scheduler have hosts (because the realm have some), or
        # just a warning if the realm is without hosts (like the top level one when the admin
        # forget to remove the default scheduler)
        if nb_hosts_in_realm == 0:  # no hosts, WARNING
            for _err in _errors:
                logger.warning(_err)
            return True  # is valid
        else:  # have hosts, will be ERROR
            for _err in _errors:
                logger.error(_err)
            return False
    
    
    # Use to make python properties
    # TODO: change itemgroup function pythonize?
    def pythonize(self):
        cls = self.__class__
        for prop, tab in cls.properties.items():
            try:
                old_val = getattr(self, prop)
                new_val = tab.pythonize(old_val)
                # print "Changing ", old_val, "by", new_val
                setattr(self, prop, new_val)
            except AttributeError:
                pass  # Will be caught at the is_correct moment
    
    
    def add_shard(self, shard_id, serialized_shard, host_names):
        self.shards[shard_id] = Shard(serialized_shard, shard_id, host_names)
    
    
    # We are receiving a partial (one shard only) host & checks inventory (name, uuid, etc) so we are merging it
    # as they are host_uuid indexed, and one host can only be in one shard
    def update_inventory_from_sharded_inventories(self, sharded_inventories):
        for (receiver_sharded_inventory, broker_sharded_inventory) in sharded_inventories:
            self._host_and_check_receiver_inventory.update_from_sharded_inventory(receiver_sharded_inventory)
            self._host_and_check_broker_inventory.update_from_sharded_inventory(broker_sharded_inventory)
        self._host_and_check_receiver_inventory.compute_hash()
        self._host_and_check_broker_inventory.compute_hash()
        
        logger.info('[INVENTORY] [%s] The realm hosts & checks inventory for receivers is now updated from shard inventories. We now have an inventory of %d hosts. The new realm inventory hash is %s.' % (
            self.get_name(), self._host_and_check_receiver_inventory.get_len(), self._host_and_check_receiver_inventory.get_hash()))
        logger.info('[INVENTORY] [%s] The realm hosts & checks inventory for brokers is now updated from shard inventories. We now have an inventory of %d hosts. The new realm inventory hash is %s.' % (
            self.get_name(), self._host_and_check_broker_inventory.get_len(), self._host_and_check_broker_inventory.get_hash()))
    
    
    # We should give an inventory to the receiver, in a sharded way
    # * if we have only one shard, we can directly give our inventory
    # * if not, we should compute another inventory with only the needs hosts
    def _get_inventory_for_receiver(self, previous_elements_weight_sum, receiver_weight, sum_of_receiver_weights, receiver_index, receiver_weights):
        # One shard means that our weight is the sum
        if receiver_weight == sum_of_receiver_weights:
            return self._host_and_check_receiver_inventory
        receiver_inventory = self._host_and_check_receiver_inventory.get_new_inventory_from_us_and_take_only_a_shard_part(previous_elements_weight_sum, receiver_weight, sum_of_receiver_weights, receiver_index, receiver_weights)
        return receiver_inventory
    
    
    # We should give an inventory to the broker.
    # Not the same as receiver, as the broker need EVERY hosts/checks
    # so we can give the whole inventory
    def get_inventory_for_broker(self):
        return self._host_and_check_broker_inventory
    
    
    def get_realms_by_explosion(self, realms, level=0):
        
        if level > MAX_REALMS_LEVEL:
            err = "Error: %s got too many level for realms" % self.get_name()
            self.configuration_errors.append(err)
            return
        
        if getattr(self, '_already_explode', False):
            return
        
        self.original_realm_members = self.get_realm_members()
        self._already_explode = True
        
        all_my_sons_members = set()
        realm_mbrs = self.get_realm_members()
        
        for sub_realm_name in realm_mbrs:
            sub_realm = realms.find_by_name(sub_realm_name.strip())
            if sub_realm is None:
                err = "Error: the realm '%s' have the sub realm '%s' but it does not exists" % (self.get_name(), sub_realm_name)
                self.configuration_errors.append(err)
                continue
            # my son is one of my members
            all_my_sons_members.add(sub_realm_name)
            # Ask the sub realm to fill itself
            sub_realm.get_realms_by_explosion(realms, level + 1)
            for sub_realm_son in sub_realm.get_realm_members():
                if sub_realm_son == self.get_name():  # do not set ourselve as sons if ther is a loop
                    err = "Error: The realm %s got a loop with the other realm %s" % (self.get_name(), sub_realm_name)
                    self.configuration_errors.append(err)
                    self._got_loop = True
                    continue
                all_my_sons_members.add(sub_realm_son)
        self.realm_members = ','.join(all_my_sons_members)
    
    
    def get_all_subs_pollers(self):
        r = copy.copy(self.pollers)
        for p in self.realm_members:
            tmps = p.get_all_subs_pollers()
            for s in tmps:
                r.append(s)
        return r
    
    
    def get_all_subs_reactionners(self):
        r = copy.copy(self.reactionners)
        for p in self.realm_members:
            tmps = p.get_all_subs_reactionners()
            for s in tmps:
                r.append(s)
        return r
    
    
    def get_all_subs_receivers(self):
        r = copy.copy(self.receivers)
        for p in self.realm_members:
            tmps = p.get_all_subs_receivers()
            for s in tmps:
                r.append(s)
        return r
    
    
    def count_reactionners(self):
        self.nb_reactionners = 0
        for reactionner in self.reactionners:
            if not reactionner.spare:
                self.nb_reactionners += 1
        for realm in self.higher_realms:
            for reactionner in realm.reactionners:
                if not reactionner.spare and reactionner.manage_sub_realms:
                    self.nb_reactionners += 1
    
    
    def fill_potential_reactionners(self):
        self.potential_reactionners = []
        for reactionner in self.reactionners:
            self.potential_reactionners.append(reactionner)
        for realm in self.higher_realms:
            for reactionner in realm.reactionners:
                if reactionner.manage_sub_realms:
                    self.potential_reactionners.append(reactionner)
    
    
    def count_pollers(self):
        self.nb_pollers = 0
        for poller in self.pollers:
            if not poller.spare:
                self.nb_pollers += 1
        for realm in self.higher_realms:
            for poller in realm.pollers:
                if not poller.spare and poller.manage_sub_realms:
                    self.nb_pollers += 1
    
    
    def fill_potential_pollers(self):
        self.potential_pollers = []
        for poller in self.pollers:
            self.potential_pollers.append(poller)
        for realm in self.higher_realms:
            for poller in realm.pollers:
                if poller.manage_sub_realms:
                    self.potential_pollers.append(poller)
    
    
    def count_brokers(self):
        self.nb_brokers = 0
        for broker in self.brokers:
            if not broker.spare:
                self.nb_brokers += 1
        for realm in self.higher_realms:
            for broker in realm.brokers:
                if not broker.spare and broker.manage_sub_realms:
                    self.nb_brokers += 1
    
    
    def fill_potential_brokers(self):
        self.potential_brokers = []
        for broker in self.brokers:
            self.potential_brokers.append(broker)
        for realm in self.higher_realms:
            for broker in realm.brokers:
                if broker.manage_sub_realms:
                    self.potential_brokers.append(broker)
    
    
    def count_receivers(self):
        self.nb_receivers = 0
        for receiver in self.receivers:
            if not receiver.spare:
                self.nb_receivers += 1
        for realm in self.higher_realms:
            for receiver in realm.receivers:
                if not receiver.spare and receiver.manage_sub_realms:
                    self.nb_receivers += 1
    
    
    def fill_potential_receivers(self):
        self.potential_receivers = []
        for receiver in self.receivers:
            self.potential_receivers.append(receiver)
        for realm in self.higher_realms:
            for receiver in realm.receivers:
                if receiver.manage_sub_realms:
                    self.potential_receivers.append(receiver)
    
    
    # Return the list of satellites of a certain type
    # like reactionner -> self.reactionners
    def get_satellites_by_type(self, type):
        if hasattr(self, type + 's'):
            return getattr(self, type + 's')
        else:
            logger.debug("[realm] do not have this kind of satellites: %s" % type)
            return []
    
    
    # Return the list of potentials satellites of a certain type
    # like reactionner -> self.potential_reactionners
    def get_potential_satellites_by_type(self, type):
        if hasattr(self, 'potential_' + type + 's'):
            return getattr(self, 'potential_' + type + 's')
        else:
            logger.debug("[realm] do not have this kind of satellites: %s" % type)
            return []
    
    
    # Return the list of potentials satellites of a certain type
    # like reactionner -> self.nb_reactionners
    def get_nb_of_must_have_satellites(self, type):
        if hasattr(self, 'nb_' + type + 's'):
            return getattr(self, 'nb_' + type + 's')
        else:
            logger.debug("[realm] do not have this kind of satellites: %s" % type)
            return 0
    
    
    def _get_all_satellites(self):
        # type: () -> List[SatelliteLink]
        return self.pollers + self.reactionners + self.brokers + self.receivers  # noqa => we are ok with the type
    
    
    # Fill dict of realms for managing the satellites confs
    def prepare_for_satellites_conf(self):
        self.structure_lock = threading.RLock()
        
        self.count_reactionners()
        self.fill_potential_reactionners()
        self.count_pollers()
        self.fill_potential_pollers()
        self.count_brokers()
        self.fill_potential_brokers()
        self.count_receivers()
        self.fill_potential_receivers()
        
        s = "Realm %-15s: (in realm / in realm + higher realms) (schedulers:%d) (pollers:%d/%d) (reactionners:%d/%d) (brokers:%d/%d) (receivers:%d/%d)" % \
            (self.get_name(),
             len(self.schedulers),
             self.nb_pollers, len(self.potential_pollers),
             self.nb_reactionners, len(self.potential_reactionners),
             self.nb_brokers, len(self.potential_brokers),
             self.nb_receivers, len(self.potential_receivers)
             )
        logger.info(s)
    
    
    # Get a conf package of satellites links that can be useful for
    # a scheduler
    def get_satellites_links_for_scheduler(self):
        with self.structure_lock:
            cfg = {'pollers'     : {},
                   'reactionners': {},
                   'schedulers'  : {}}
            
            # First we create/void theses links
            
            # First our own level
            for p in self.potential_pollers:
                c = p.give_satellite_cfg()
                cfg['pollers'][p.id] = c
            
            for r in self.potential_reactionners:
                c = r.give_satellite_cfg()
                cfg['reactionners'][r.id] = c
            
            for r in self.get_scheduler_ordered_list():
                s = r.give_satellite_cfg()
                cfg['schedulers'][r.id] = s
            
            return cfg
    
    
    # Make an ORDERED list of schedulers so we can
    # send them conf in this order for a specific realm
    def get_scheduler_ordered_list(self):
        # get scheds, alive and no spare first
        scheds = set()
        for s in self.schedulers:
            scheds.add(s)
        
        # now the spare scheds of higher realms
        # they are after the sched of realm, so
        # they will be used after the spare of
        # the realm
        for higher_r in self.higher_realms:
            for s in higher_r.schedulers:
                if s.spare:
                    scheds.add(s)
        scheds = list(scheds)
        # Now we sort the scheds so we take master, then spare
        # the dead, but we do not care about them
        scheds.sort(key=functools.cmp_to_key(alive_then_spare_then_deads))
        
        return scheds
    
    
    def get_not_assigned_shards(self):
        with self.structure_lock:
            # Important: we want to always have the same order here so if nothing change, same scheduler
            #            will have the same shard id
            unassigned_shards = [shard for shard in self.shards.values() if not shard.is_assigned]
            unassigned_shards_sorted = sorted(unassigned_shards, key=lambda shd: shd.get_id())
            return unassigned_shards_sorted
    
    
    def _unset_scheduler_conf(self, sched):
        with self.structure_lock:
            self.dispatch_ok = False  # so we ask a new dispatching
            sched.unassign_shard()
    
    
    def compute_accessible_realms(self):
        for daemon_type in ('receiver', 'broker'):
            realm_accessible_satellites = self.get_potential_satellites_by_type(daemon_type)
            for satellite in realm_accessible_satellites:
                satellite.set_accessible_realm(self.get_name())
    
    
    def print_initial_listing(self):
        # Now we have all links between realms/satellite, we can show a summary
        if self.have_satellites():
            self.logger_dispatch.info('  ==== Daemons listing of Realm %s ====' % self.get_name())
            self.print_accessible_realms()
        self.get_potential_schedulers(-1, force_print=True, prefix='')  # -1 => show all schedulers
    
    
    def check_schedulers_dispatch(self, first_dispatch_done):
        # type: (bool) -> None
        potential_schedulers = set(self.get_scheduler_ordered_list())
        schedulers_with_valid_shard = set()
        realm_name = self.get_name()
        for shard_id, shard in self.shards.items():
            sched = shard.get_assigned_scheduler()  # type: Optional[SchedulerLink]
            if sched is None:
                # On the very first run, we do not raise an error, it's normal to not be assigned
                if first_dispatch_done:
                    self.logger_configuration.info('UNMANAGED SHARD   shard [%d] is unmanaged. We will sent it to a new scheduler.' % shard_id)
                continue
            
            if not sched.alive:
                sched.logger_configuration.error('DEAD SCHEDULER   Scheduler had the shard [%d] but is dead. We will sent its shard to another scheduler.' % shard_id)
                self._unset_scheduler_conf(sched)
                continue
            
            # Maybe the scheduler restarts, so is alive but without the conf we think it was managing
            # so ask it what it is really managing, and if not, put the conf unassigned
            if not sched.assert_manage_shard_id(realm_name, shard_id):
                # We unassigned expected shard but if we make it here there are multiple shards assign to the same scheduler, so it is a bug. (see SEF-9788)
                have_unassigned = shard.unassign_scheduler()
                if have_unassigned:
                    logger.error('ERROR the scheduler [%s] had multiple assign shard' % sched.get_name())
                continue
            
            # ok seem to be good but maybe there is an old/bad satellites
            allowed_brokers_for_this_scheduler, _, _, _ = self.get_dispatchable_satellite('broker')
            allowed_brokers_names_for_this_scheduler = [broker.get_name() for broker in allowed_brokers_for_this_scheduler]
            sched.logger_configuration.debug('Checking if scheduler brokers are in this allowed list: "%s"' % (', '.join(allowed_brokers_names_for_this_scheduler)))
            sched.assert_only_allowed_brokers(allowed_brokers_names_for_this_scheduler)
            schedulers_with_valid_shard.add(sched)
        
        schedulers_without_shard = potential_schedulers - schedulers_with_valid_shard
        schedulers_without_shard = [sched for sched in schedulers_without_shard if sched.alive]  # don't care about DEAD one
        master_schedulers_with_shard = [sched for sched in schedulers_without_shard if not sched.spare]
        spare_schedulers_with_shard = [sched for sched in schedulers_with_valid_shard if sched.spare]
        # if spare schedulers manage conf and non-spare are waiting, we need to ask redispatch for this conf
        if any(master_schedulers_with_shard) and any(spare_schedulers_with_shard):
            for index, sched in enumerate(spare_schedulers_with_shard):
                sched.logger_configuration.info('IDLE SCHEDULER WITH SHARD   The scheduler have a shard but should a master came back to life. We are asking it to go in sleep mode.')
                self._unset_scheduler_conf(sched)
                # LONG CALL
                sched.create_and_put_inactive_scheduler_configuration('The scheduler slave have a shard but a master came back to life. We are asking it to go in sleep mode.')
                # ask to redispatch only for the number of available non-spare schedulers. avoid redispatch every scheduler if only one master come up
                if index >= len(master_schedulers_with_shard):
                    break
        
        # Some schedulers are waiting for a new conf, but they all have been sent, send them a spare conf
        for sched in schedulers_without_shard:
            if not sched.is_managing_any_configuration():  # it is NEVER have any conf, set as idle
                sched.logger_configuration.debug('IDLE SATELLITE   The scheduler is currently idle.')
                self._unset_scheduler_conf(sched)
                # only send a real spare conf if it's not the very first loop, at least let a dispatch make all assignation
                if first_dispatch_done:
                    sched.logger_configuration.debug('SET AS SPARE   The scheduler do not have any shard to manage, so it will be set as a spare until it is assigned a shard.')
                    # LONG CALL
                    sched.create_and_put_inactive_scheduler_configuration('The scheduler do not have any shard to manage, so it will be set as idle until it is assigned a shard.')
        
        return
    
    
    @staticmethod
    def __stack_print_accessible_realms_table_row_for_satellite(table, satellite, _type):
        # type: (Texttable, SatelliteLink, str) -> None
        alive_str = 'ALIVE' if satellite.alive else 'DEAD'
        manage_sub_realms_str = '%s' % satellite.manage_sub_realms
        managed_realms = satellite.get_all_my_managed_realms()
        managed_realms_names = sorted([realm.get_name() for realm in managed_realms])
        managed_schedulers_str = satellite.get_string_of_managed_schedulers()
        table.add_row([_type, satellite.get_name(), alive_str, satellite.get_short_spare_display(), manage_sub_realms_str, ', '.join(managed_realms_names), managed_schedulers_str])
    
    
    def have_satellites(self):
        return len(self.pollers) != 0 or len(self.reactionners) != 0 or len(self.brokers) != 0 or len(self.receivers) != 0
    
    
    def print_accessible_realms(self, prefix=''):
        # Maybe we have nothing to show
        if not self.have_satellites():
            return
        table = Texttable(max_width=0)  # unlimited size
        table.set_cols_align(['l', 'l', 'l', 'l', 'l', 'l', 'l'])  # align left
        table.add_row(['Type', 'Name', 'Ping', 'Spare', 'Manage sub realms', 'Used in realms', 'Link to schedulers/shards'])
        
        for satellite in self.brokers:
            self.__stack_print_accessible_realms_table_row_for_satellite(table, satellite, 'Broker')
        for satellite in self.pollers:
            self.__stack_print_accessible_realms_table_row_for_satellite(table, satellite, 'Poller')
        for satellite in self.reactionners:
            self.__stack_print_accessible_realms_table_row_for_satellite(table, satellite, 'Reactionner')
        for satellite in self.receivers:
            self.__stack_print_accessible_realms_table_row_for_satellite(table, satellite, 'Receiver')
        
        table_draw = table.draw()
        for line in table_draw.splitlines():
            space_size = SECTION_SPACE_SIZE - len(prefix)  # SECTION_SPACE_SIZE => to align with the chapter end
            self.logger_dispatch.info((prefix + (' ' * space_size)) + line)
    
    
    # Imagine a world where... oh no, wait...
    # Imagine a master got the conf and the network is down a spare takes it (good :) ). Like the Empire, the master
    # strikes back! It was still alive! (like Elvis). It still got conf and is running! not good!
    # Bad dispatch: a link that has a conf but I do not allow this so I ask it to wait a new conf and stop kidding.
    def check_bad_dispatch(self, first_dispatch_done):
        for scheduler in self.schedulers:
            if not first_dispatch_done:
                scheduler.assert_no_previous_run_configuration()
            else:
                scheduler.assert_no_invalid_configuration()
    
    
    # We only want to have schedulers that are :
    # * alive
    # * not spare if possible (only if not enough not spare schedulers to serve)
    def get_potential_schedulers(self, nb_shards_to_dispatch, force_print=False, prefix=''):
        all_schedulers = self.get_scheduler_ordered_list()
        potential_schedulers = []
        already_assigned_schedulers = [scheduler for scheduler in all_schedulers if scheduler.is_managing_a_shard()]
        dead_schedulers = [scheduler for scheduler in all_schedulers if not scheduler.alive]
        unassigned_master_schedulers = [scheduler for scheduler in all_schedulers if not scheduler.spare and scheduler.alive and not scheduler.is_managing_a_shard()]
        unassigned_spare_schedulers = [scheduler for scheduler in all_schedulers if scheduler.spare and scheduler.alive and not scheduler.is_managing_a_shard()]
        if nb_shards_to_dispatch != -1:  # normal case
            for i in range(nb_shards_to_dispatch):
                # First look in master ones
                if unassigned_master_schedulers:
                    scheduler = unassigned_master_schedulers.pop(0)
                    potential_schedulers.append(scheduler)
                    continue
                # Only if master is gone, look at spare
                if unassigned_spare_schedulers:
                    scheduler = unassigned_spare_schedulers.pop(0)
                    potential_schedulers.append(scheduler)
                    continue
                # Oups, no more scheduler alive?
        else:  # just for display, get all schedulers
            potential_schedulers = unassigned_master_schedulers
        
        if nb_shards_to_dispatch > 0 or force_print:
            self._print_dispatchable_schedulers(nb_shards_to_dispatch, already_assigned_schedulers, potential_schedulers, dead_schedulers, unassigned_spare_schedulers, prefix)
        return potential_schedulers
    
    
    def dispatch_schedulers(self):
        try:
            self._do_dispatch_schedulers()
        except:  # noqa => catch in a thread
            self.logger_configuration.error('The scheduler dispatch thread did fail: %s' % traceback.format_exc())
    
    
    def _print_dispatchable_schedulers(self, nb_shards_to_dispatch, already_assigned_schedulers, potential_schedulers, dead_schedulers, unassigned_spare_schedulers, prefix):
        if len(already_assigned_schedulers) == 0 and len(potential_schedulers) == 0 and len(dead_schedulers) == 0 and len(unassigned_spare_schedulers) == 0:
            return
        
        if nb_shards_to_dispatch > 0:
            self.logger_dispatch.info("  Dispatching shards and satellites ")
            self.logger_dispatch.info('     Dispatching %d unassigned shards (for a total of %d shards in this realm) to schedulers' % (nb_shards_to_dispatch, len(self.shards)))
        
        table = Texttable(max_width=0)  # unlimited size
        table.set_cols_align(['l', 'l', 'l', 'l', 'l'])  # align left, center, center
        table.add_row(['State', 'Name', 'Ping', 'Spare', 'Managed shard'])
        
        for scheduler in already_assigned_schedulers:
            self.__stack_dispatch_table_row_for_scheduler(table, scheduler, 'WORKING')
        for scheduler in potential_schedulers:
            self.__stack_dispatch_table_row_for_scheduler(table, scheduler, 'AVAILABLE')
        for scheduler in dead_schedulers:
            self.__stack_dispatch_table_row_for_scheduler(table, scheduler, 'DEAD')
        for scheduler in unassigned_spare_schedulers:
            self.__stack_dispatch_table_row_for_scheduler(table, scheduler, 'IDLE SPARE')
        
        table_draw = table.draw()
        for line in table_draw.splitlines():
            space_size = SECTION_SPACE_SIZE - len(prefix)
            self.logger_dispatch.info(prefix + (' ' * space_size) + line)  # SECTION_SPACE_SIZE => to align with the chapter end
    
    
    def _do_dispatch_schedulers(self):
        did_work = False
        shards_to_dispatch = self.get_not_assigned_shards()
        nb_shards_to_dispatch = len(shards_to_dispatch)
        
        # Now we get in scheds all scheduler of this realm and upper so
        # we will send them conf (in this order)
        potential_schedulers = self.get_potential_schedulers(nb_shards_to_dispatch, prefix='BEFORE DISPATCH =>')
        
        # Now we do the real job
        for shard in shards_to_dispatch:
            shard_id = shard.id
            self.logger_configuration.debug('DISPATCH SHARD    Dispatching shard [%s]' % shard_id)
            
            # If there is no alive schedulers, not good...
            if len(potential_schedulers) == 0:
                self.logger_configuration.error('NO SCHEDULERS    but there a no alive schedulers in this realm.')
            
            # we need to loop until the conf is assigned or when there are no more schedulers available
            while True:
                # No more schedulers.. not good, no loop
                # => the conf does not need to be dispatch
                if len(potential_schedulers) == 0:
                    break
                
                # Ok there are still some schedulers
                scheduler = potential_schedulers.pop(0)
                
                # Must take this lock here to avoid deadlock with "big_lock" in this function ( SEF-11496 )
                with self.structure_lock:
                    is_sent = scheduler.create_and_put_active_scheduler_configuration(shard, self)
                # Maybe the scheduler do not even need a shard, or maybe the sending failed.In all cases, just loop to the next scheduler
                if not is_sent:
                    continue
                did_work = True
                # Ok, the conf is dispatched, no more loop for this configuration
                break
        
        if did_work:
            self.get_potential_schedulers(0, force_print=True, prefix='AFTER DISPATCH =>')
            
            nb_still = len(self.get_not_assigned_shards())
            if nb_still > 0:
                self.logger_configuration.error("MISSING SHARDS  All schedulers shards are not dispatched: %d are missing" % nb_still)
            elif nb_shards_to_dispatch > 0:  # there were shards to dispatch, and all are done
                self.logger_configuration.info("SHARD ALL SENT   All %d schedulers shards are dispatched." % len(self.shards))
    
    
    # Get a list of satellite that are available for this realm and
    # * are alive
    # * are reachable
    # * sorted by nearest and not spare first
    def get_dispatchable_satellite(self, kind):
        # Broker have a new spare computation, but maybe the admin ask to
        # do not use it, then use old way
        if kind == 'broker' and not FLAG_DISABLE_NEW_BROKER_SPARE_COMPUTATION:
            return self._get_dispatchable_satellite_for_brokers()
        must_have_cout = self.get_nb_of_must_have_satellites(kind)
        
        # make copies of potential_react list for sort
        valid_realm_satellites = []
        dead_satellites = []
        
        for satellite in self.get_potential_satellites_by_type(kind):
            # skip not reachable ones
            if not satellite.alive:
                dead_satellites.append(satellite)
                continue
            valid_realm_satellites.append(satellite)
        valid_realm_satellites.sort(key=functools.cmp_to_key(alive_then_spare_then_deads))  # always have the element sorted
        dispatchable_satellites = valid_realm_satellites[:must_have_cout]
        dispatchable_satellites.sort(key=functools.cmp_to_key(alive_then_spare_then_deads))
        overcount_satellites = valid_realm_satellites[must_have_cout:]
        overcount_satellites.sort(key=functools.cmp_to_key(alive_then_spare_then_deads))
        
        return dispatchable_satellites, dead_satellites, overcount_satellites, []  # currently useless spares is void on other than broker
    
    
    # Brokers are in the first line for the new way of selecting daemons:
    # Take masters, and only masters
    # if a master is dead, take its spare
    # deads are dead, spare with no master will NEVER be called, what ever happen
    def _get_dispatchable_satellite_for_brokers(self):
        # make copies of potential_react list for sort
        master_satellites = []  # List[BrokerLink]
        all_spare_satellites = []  # List[BrokerLink]
        dead_satellites = []  # List[BrokerLink]
        useless_spares = []  # List[BrokerLink]
        
        # Put all master, all spare and deads in lists
        for satellite in self.get_potential_satellites_by_type('broker'):
            # skip not reacheable ones
            if not satellite.spare:
                master_satellites.append(satellite)
            else:
                if satellite.master_daemon is None:
                    useless_spares.append(satellite)
                else:  # real spare
                    all_spare_satellites.append(satellite)
            if not satellite.alive:
                dead_satellites.append(satellite)
        
        # Now remove dad masters, and put alive spare if possible
        spare_that_take_over = []  # no need for set(), spares are uniq
        master_to_remove = []
        for master in master_satellites:
            if master.alive:  # master alive are ok
                continue
            # Oups, the master is DEAD
            master_to_remove.append(master)
            # Have a alive spare?
            spare = master.spare_daemon
            if spare is not None:
                if spare.alive:  # maybe the spare was dead too ^^
                    spare_that_take_over.append(spare)
        dispatchable_satellites = [master for master in master_satellites if master not in master_to_remove]
        dispatchable_satellites.extend(spare_that_take_over)
        
        # Overcount satellites are spare that are not allowed to have conf currently
        overcount_satellites = [satellite for satellite in all_spare_satellites if satellite not in dispatchable_satellites]
        
        return dispatchable_satellites, dead_satellites, overcount_satellites, useless_spares
    
    
    def __stack_dispatch_table_row(self, table, satellite, state):
        alive_str = 'ALIVE' if satellite.alive else 'DEAD'
        realm_str = satellite.realm.get_name() if satellite.realm != self else ''
        managed_schedulers_str = satellite.get_string_of_managed_schedulers()
        table.add_row([state, satellite.get_name(), alive_str, satellite.get_short_spare_display(), realm_str, managed_schedulers_str])
    
    
    @staticmethod
    def __stack_dispatch_table_row_for_scheduler(table, satellite, state):
        alive_str = 'ALIVE' if satellite.alive else 'DEAD'
        managed_shard = satellite.get_managed_shard_id()
        managed_shard_str = '%s' % managed_shard if managed_shard is not None else 'no shard'
        table.add_row([state, satellite.get_name(), alive_str, satellite.get_short_spare_display(), managed_shard_str])
    
    
    def _print_dispatchable_satellite(self, kind, dispatchable_satellites, dead_satellites, overcount_satellites, useless_spares, prefix=''):
        must_have_cout = self.get_nb_of_must_have_satellites(kind)
        
        print_string = "This realm need %d %ss:" % (must_have_cout, kind)
        self.logger_dispatch.info(print_string)
        
        table = Texttable(max_width=0)  # unlimited size
        table.set_cols_align(['l', 'l', 'l', 'l', 'l', 'l'])  # align left, center, center
        table.add_row(['State', 'Name', 'Ping', 'Spare', 'Realm if different', 'Link to schedulers/shards'])
        for satellite in dispatchable_satellites:
            self.__stack_dispatch_table_row(table, satellite, 'AVAILABLE')
        for satellite in dead_satellites:
            self.__stack_dispatch_table_row(table, satellite, 'DEAD')
        for satellite in overcount_satellites:
            self.__stack_dispatch_table_row(table, satellite, 'AVAILABLE SPARE')
        for satellite in useless_spares:
            self.__stack_dispatch_table_row(table, satellite, 'UNUSED')
        
        table_draw = table.draw()
        for line in table_draw.splitlines():
            space_size = SECTION_SPACE_SIZE - len(prefix)
            self.logger_dispatch.info(prefix + (' ' * space_size) + line)  # SECTION_SPACE_SIZE => to align with the chapter end
    
    
    @staticmethod
    def _dispatch_shard_to_satellite(satellite, kind, shard_id, shard_for_satellite_part, shard, arbiters_cfg):
        scheduler_managing_the_shard_name = shard.get_current_scheduler_name()
        
        is_managing_the_valid_config_incarnation = satellite.is_managing_the_valid_configuration_incarnation()
        is_managing_this_scheduler = satellite.do_i_manage_this_scheduler(shard_id, scheduler_managing_the_shard_name)
        
        # Maybe this satellite already got this configuration, so skip it
        if is_managing_the_valid_config_incarnation and is_managing_this_scheduler:
            satellite.logger_configuration.debug('ALREADY MANAGED SHARD Skipping shard [%d] send to the %s: it already manage it' % (shard_id, kind))
            is_sent = True
        else:  # ok, it really need it :)
            satellite.logger_configuration.debug('SATELLITE SENT START Trying to send shard assignation [%s=>%s] to the %s (have valid incarnation=%s, already manage scheduler=%s)' % (
                shard_id, scheduler_managing_the_shard_name, kind, is_managing_the_valid_config_incarnation, is_managing_this_scheduler))
            
            satellite.push_shard_in_cfg(shard_id, shard_for_satellite_part, arbiters_cfg)
            # see bottom of page for extended_conf explanation
            extended_conf = satellite.cfg
            extended_conf['arbiter_trace'] = satellite.get_my_arbiter_trace()
            
            # LONG CALL
            is_sent = satellite.put_conf(extended_conf)
            if is_sent:
                # There are two cases we are here:
                # * we need to update the daemon with the new configuration incarnation
                # * the scheduler was not present
                #  -> it's important to do NOT log we did send a new scheduler, where in fact the daemon won't display it in the log
                if not is_managing_the_valid_config_incarnation:
                    satellite.logger_configuration.info('SATELLITE SENT OK  The daemon is now updated to the new configuration incarnation (%s)' % satellite.configuration_incarnation)
                if not is_managing_this_scheduler:
                    satellite.logger_configuration.info('SATELLITE SENT OK  Dispatch OK of shard assignation [%-4d=> %-20s] to the %s' % (
                        shard_id, scheduler_managing_the_shard_name, kind))
        
        if is_sent:
            # We change the satellite configuration, update our data
            satellite.known_conf_managed_push(shard_id, scheduler_managing_the_shard_name)
        
        else:
            satellite.logger_configuration.warning('SATELLITE SENT FAIL  Dispatch failed of shard assignation [%s=>%s] to the %s' % (shard_id, scheduler_managing_the_shard_name, kind))
        
        return is_sent
    
    
    def assert_receivers_inventories(self):
        try:
            self._proxy_assert_receivers_inventories()
        except:
            self.logger.error('The thread for checking receivers inventories did crash')
            self.logger.print_stack()
            os._exit(2)  # noqa
    
    
    # Assert that the receivers have the good inventories
    def _proxy_assert_receivers_inventories(self):
        # make copies of potential_react list for sort
        all_valid_realm_receivers, _, _, _ = self.get_dispatchable_satellite('receiver')
        # Only push inventory to the receiver that need it
        only_shardable_receivers = [receiver for receiver in all_valid_realm_receivers if receiver.elements_sharding_enabled]
        
        # must be sorted to be sure we always have the same dispatch
        sorted_shardable_receivers = sorted(only_shardable_receivers, key=lambda x: x.get_name())
        
        receiver_weights = [receiver.elements_sharding_weight for receiver in sorted_shardable_receivers]
        sum_of_receiver_weights = sum(receiver_weights)
        previous_elements_weight_sum = 0
        
        for (receiver_index, receiver) in enumerate(sorted_shardable_receivers):
            # realm inventory
            inventory_for_receiver = self._get_inventory_for_receiver(previous_elements_weight_sum, receiver.elements_sharding_weight, sum_of_receiver_weights, receiver_index, receiver_weights)
            
            # Be sure that this receiver have this realm inventory up-to-date
            receiver.assert_have_realm_inventory(self.get_name(), inventory_for_receiver)
            
            # Increase the weight sum for the next receiver
            previous_elements_weight_sum += receiver.elements_sharding_weight
    
    
    def assert_brokers_inventories(self):
        try:
            self._proxy_assert_brokers_inventories()
        except:
            self.logger.error('The thread for checking brokers inventories did crash')
            self.logger.print_stack()
            os._exit(2)  # noqa
    
    
    def _proxy_assert_brokers_inventories(self):
        # make copies of potential_react list for sort
        all_valid_realm_brokers, _, _, _ = self.get_dispatchable_satellite('broker')
        
        # All brokers are taking the same inventory (full with all hosts/checks)
        inventory_for_broker = self.get_inventory_for_broker()
        for broker in all_valid_realm_brokers:
            # Be sure that this receiver have this realm inventory up-to-date
            broker.assert_have_realm_inventory(self.get_name(), inventory_for_broker)
    
    
    def assert_receivers_host_mapping(self):
        try:
            for receiver in self.receivers:
                receiver.assert_valid_host_mapping()
        except:  # noqa  => Catch all, to display the thread crash
            self.logger.error('The thread for checking host mapping in receivers did crash')
            self.logger.print_stack()
            os._exit(2)  # noqa
    
    
    def dispatch_satellites(self, arbiters_cfg, valid_shard_ids):
        try:
            self._do_dispatch_satellites(arbiters_cfg, valid_shard_ids)
        except:  # noqa => ok with ALL exception
            logger.error('ERROR on the dispatch_satellites thread: %s. Exiting' % traceback.format_exc())
            os._exit(2)  # noqa => we need to call this to kill threads
    
    
    def _do_dispatch_satellites(self, arbiters_cfg, valid_shard_ids):
        did_work = False
        # General dispatch: let each satellite know about a shard place (on which scheduler) and other nodes too
        for shard in self.shards.values():
            shard_id = shard.id
            scheduler_managing_the_shard = shard.get_assigned_scheduler()
            scheduler_managing_the_shard_name = shard.get_current_scheduler_name()
            for kind in self.satellite_kinds:
                shard_for_satellite_part = None
                
                dead_satellites = []
                overcount_satellites = []
                useless_spares = []
                # - If assigned, only the good number of satellites, and alive must have it
                #   and not all the others
                # - If not assigned, ALL satellites must NOT have this shard
                if shard.is_assigned:
                    # make copies of potential_react list for sort
                    valid_realm_satellites, dead_satellites, overcount_satellites, useless_spares = self.get_dispatchable_satellite(kind)
                    should_not_have_satellites = overcount_satellites  # set(self.get_potential_satellites_by_type(kind)) - set(valid_realm_satellites)
                    shard_for_satellite_part = scheduler_managing_the_shard.give_satellite_cfg()
                else:
                    valid_realm_satellites = []
                    should_not_have_satellites = self.get_potential_satellites_by_type(kind)
                
                # Set shard to the satellites that NEED it
                satellites_not_managing_the_shard = []
                # Now we dispatch cfg to every one ask for it
                for satellite in valid_realm_satellites:
                    # Maybe this satellite already got this configuration, so skip it
                    if satellite.is_managing_the_valid_configuration_incarnation() and satellite.do_i_manage_this_scheduler(shard_id, scheduler_managing_the_shard_name):
                        satellite.logger_configuration.debug('ALREADY MANAGED SHARD Skipping shard [%d] send to the %s: it already manage it' % (shard_id, kind))
                        continue
                    if not satellite.is_managing_the_valid_configuration_incarnation():
                        satellite.logger_configuration.debug('NEED CONFIGURATION UPDATE the %s is not up to date for the configuration incarnation, sending it a configuration update' % kind)
                    satellites_not_managing_the_shard.append(satellite)
                
                if len(satellites_not_managing_the_shard) > 0:
                    did_work = True
                    self._print_dispatchable_satellite(kind, valid_realm_satellites, dead_satellites, overcount_satellites, useless_spares, 'BEFORE DISPATCH =>')
                    # Now really check that they do have it
                    for satellite in satellites_not_managing_the_shard:
                        self._dispatch_shard_to_satellite(satellite, kind, shard_id, shard_for_satellite_part, shard, arbiters_cfg)
                
                # UNSET where it's useless
                for satellite in should_not_have_satellites:
                    satellite.assert_do_not_manage_shard(shard)
        
        # Now for satellites that are spare, make them go sleep when they are void
        for kind in self.satellite_kinds:
            for satellite in self.get_satellites_by_type(kind):  # only MY satellites
                satellite.assert_spare_sleeping_if_void()
        
        # Now look if satellites are not managing schedulers that are no more here (delete/disabled)
        for kind in self.satellite_kinds:
            for satellite in self.get_satellites_by_type(kind):  # only MY satellites
                satellite.assert_only_known_shard_ids(valid_shard_ids)
        
        if did_work:
            self.print_accessible_realms('AFTER DISPATCH =>')
    
    
    # We will fill all higher_realms from our sons with ourselve
    def fill_sub_realms_with_myself_as_higher_realm(self):
        for realm in self.realm_members:
            realm.add_higher_realm(self)
    
    
    def get_all_my_lower_realms(self, level=0):
        r = set()
        if level > MAX_REALMS_LEVEL:  # Loop?
            return r
        for realm in self.realm_members:
            r.add(realm)
            sub_sub_realms = realm.get_all_my_lower_realms(level + 1)
            r.union(sub_sub_realms)
        return r
    
    
    def add_higher_realm(self, higher_realm, level=0):
        if level > MAX_REALMS_LEVEL:  # Loop?
            return
        if higher_realm not in self.higher_realms:
            self.higher_realms.append(higher_realm)
        
        for realm in self.realm_members:
            realm.add_higher_realm(higher_realm, level + 1)


class Realms(Itemgroups):
    name_property = "realm_name"  # is used for finding hostgroups
    inner_class = Realm
    
    if TYPE_CHECKING:
        def __init__(self, items):
            super(Realms, self).__init__(items)
            self.conf_is_correct = False
    
    
    def get_members_by_name(self, pname):
        realm = self.find_by_name(pname)
        if realm is None:
            return []
        return realm.get_realms()
    
    
    def linkify(self, only_schedulers=False):
        self.linkify_realm_by_realm()
        
        # prepare list of satellites and confs
        for realm in self:
            realm.pollers = []
            realm.schedulers = []
            realm.reactionners = []
            realm.brokers = []
            realm.providers = []
            realm.receivers = []
            realm.packs = []
            realm.only_schedulers = only_schedulers
    
    
    # We just search for each realm the others realms
    # and replace the name by the realm
    def linkify_realm_by_realm(self):
        for p in self.items.values():
            mbrs = p.get_realm_members()
            # The new member list, in id
            new_mbrs = []
            for mbr in mbrs:
                new_mbr = self.find_by_name(mbr)
                if new_mbr is not None:
                    new_mbrs.append(new_mbr)
            # We find the id, we replace the names
            p.realm_members = new_mbrs
        
        # Also put in each realm the list of its higher realms
        # Need to set higher_realms as we are not in the pythonize part
        for r in self.items.values():
            r.higher_realms = []
        
        # Only fill higher realms if there is no loop in th realm definition
        realm_with_loops = [realm for realm in self if realm.got_loop()]
        if realm_with_loops:
            logger.error('There are realms with loop, cannot compute realm tree')
        else:
            for realm in self.items.values():
                realm.fill_sub_realms_with_myself_as_higher_realm()  # DO NOT CALL THIS ONE IF THERE ARE LOOPS!
    
    
    # Use to fill members with hostgroup_members
    def explode(self):
        default_realm = self.get_default()
        if not default_realm:
            return None
        
        for p in self:
            p.get_realms_by_explosion(self)
        
        # We clean the tags
        for tmp_p in self.items.values():
            # noinspection PyProtectedMember
            del tmp_p._already_explode
    
    
    def get_parent_realm_for(self, realm_name):
        """Returns the parent realm corresponding to the string realm_name, or None if not
        found. Returns a Realm class or None."""
        for r in self:
            members_lst = r.get_realm_members()
            if realm_name in members_lst:
                return r
    
    
    def get_path_to_default(self, realm_name):
        """Return a list of nested realms to traverse to join the default (root) realm"""
        realms = [realm_name]
        current_realm = realm_name
        iter_max = 15  # catch loop in realm config
        iter_current = 0
        while iter_current < iter_max:
            parent_realm = self.get_parent_realm_for(current_realm)
            if parent_realm is None:
                break
            parent_realm_name = parent_realm.get_name()
            # detect loop
            if parent_realm_name == current_realm:
                break
            realms.append(parent_realm_name)
            current_realm = parent_realm_name
            iter_current += 1
        
        return reversed(realms)
    
    
    def get_default(self):
        # type: () -> Optional[Realm]
        
        default_realms = [realm for realm in self if to_bool(getattr(realm, 'default', False))]
        # There can be only one default realm
        if len(default_realms) > 1:
            return None
        return default_realms[0]
    
    
    def prepare_for_satellites_conf(self):
        for r in self:
            r.prepare_for_satellites_conf()
    
    
    def _check_default_realm(self):
        errors = self.compute_default_realm_errors()
        
        if not errors:
            return True
        self.configuration_errors.extend(errors)
        return False
    
    
    def compute_default_realm_errors(self):
        to_return = []
        default_realms = [realm.get_name() for realm in self if to_bool(getattr(realm, 'default', False))]
        
        if len(default_realms) == 1:
            pass
        elif len(default_realms) == 0:
            to_return.append('''There must be one default realm defined. Please set one by setting the 'default' property to '1' for one realm.''')
        elif len(default_realms) > 1:
            to_return.append('''There must be one default realm defined. Please set one by setting the 'default' property to '1' for one realm.''')
        return to_return
    
    
    def is_correct(self):
        # type: () -> bool
        is_correct = super(Realms, self).is_correct()
        is_default_value = self._check_default_realm()
        self.conf_is_correct = is_correct and is_default_value
        return self.conf_is_correct
