#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2020:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


import gc
import os
import sys
import tempfile

import psutil

from shinken.log import logger


class RedirectStdErrStream:
    def __init__(self, stderr=None):
        self._stderr = stderr or sys.stderr
    
    
    def __enter__(self):
        self.old_stderr = sys.stderr
        self.old_stderr.flush()
        sys.stderr = self._stderr
    
    
    def __exit__(self, exc_type, exc_value, traceback):
        self._stderr.flush()
        sys.stderr = self.old_stderr


class MemoryStats:
    
    def query_memory_stats(self, expanded=False):
        
        process = psutil.Process(os.getpid())
        process_memory = process.memory_info()[0]
        
        process_python_gc_items = sum(sys.getsizeof(i) for i in gc.get_objects())
        
        mem_stats = {
            'process_memory'         : process_memory,
            'process_python_gc_items': process_python_gc_items,
        }
        
        if expanded:
            try:
                # noinspection PyPackageRequirements
                from guppy import hpy
            except ImportError:
                logger.error('(support-only) MEMORY DUMP: FAIL check if guppy lib is installed')
                mem_stats['guppy_install'] = False
                return mem_stats
            
            hp = hpy()
            h = hp.heap()
            
            heap_dump = str(h)
            logger.debug('heap_dump:[%s]' % heap_dump)
            heap_dump = heap_dump.splitlines()
            global_info_line = heap_dump[0].split()
            total_nb_item = global_info_line[5]
            process_python_heap_size = global_info_line[10]
            
            by_items_info = []
            self._parse_heap_lines(by_items_info, heap_dump[2:-1])
            heap_dump = str(h.more)
            logger.debug('heap_dump:[%s]' % heap_dump)
            heap_dump = heap_dump.splitlines()
            self._parse_heap_lines(by_items_info, heap_dump[1:-1])
            
            mem_stats['total_nb_item'] = total_nb_item
            mem_stats['process_python_heap_size'] = process_python_heap_size
            mem_stats['by_items_info'] = by_items_info
            mem_stats['guppy_install'] = True
        
        return mem_stats
    
    
    @staticmethod
    def _parse_heap_lines(by_items_info, heap_dump):
        for item_info_line in heap_dump:
            if item_info_line.startswith('              '):
                by_items_info[-1]['item_type'] += item_info_line.strip()
                continue
            item_info_line = item_info_line.split(None, 7)
            index = item_info_line[0]
            nb = item_info_line[1]
            nb_percent = item_info_line[2]
            size = item_info_line[3]
            size_percent = item_info_line[4]
            cumulative = item_info_line[5]
            cumulative_percent = item_info_line[6]
            try:
                item_type = item_info_line[7]
            except:
                item_type = ''
            by_items_info.append({
                'index'             : index,
                'nb'                : nb,
                'nb_percent'        : nb_percent,
                'size'              : size,
                'size_percent'      : size_percent,
                'cumulative'        : cumulative,
                'cumulative_percent': cumulative_percent,
                'item_type'         : item_type,
            })
    
    
    @staticmethod
    def dump_memory_full_memory_dump(daemon_name):
        if sys.version_info[1] == 6:  # python 2.6
            try:
                # noinspection PyPackageRequirements
                from guppy import hpy
                hp = hpy()
                logger.error('(support-only) MEMORY DUMP (to be sent to the support):\n%s' % hp.heap())
                return
            except ImportError:
                # noinspection PyUnusedLocal
                hpy = None
                logger.error('(support-only) MEMORY DUMP: FAIL check if guppy lib is installed')
        if sys.version_info[1] == 7:  # python 2.7
            try:
                import six
                # noinspection PyPackageRequirements
                from meliae import scanner as meliae_scanner
                # noinspection PyPackageRequirements,  PyProtectedMember
                from meliae import (_intset, _scanner)
                
                # Monkey patch (OverflowError on Alma & Redhat 8)
                # Add a try/except clause to discard unsuitable object and avoid a crash (uncaught OverflowError exception). Remaining data should be enough for memory analysis
                # noinspection SpellCheckingInspection
                def dump_all_referenced_monkey_patched(outf, obj, is_pending=False):
                    """Recursively dump everything that is referenced from obj."""
                    if isinstance(outf, six.string_types):
                        outf = open(outf, 'wb')
                    if is_pending:
                        pending = obj
                    else:
                        pending = [obj]
                    last_offset = len(pending) - 1
                    # TODO: Instead of using an IDSet, we could use a BloomFilter. It would
                    #       mean some objects may not get dumped (blooms say "yes you
                    #       definitely are not present", but only "you might already be
                    #       present", collisions cause false positives.)
                    #       However, you can get by with 8-10bits for a 1% FPR, rather than
                    #       using 32/64-bit pointers + overhead for avoiding hash collisions.
                    #       So on 64-bit we drop from 16bytes/object to 1...
                    seen = _intset.IDSet()
                    if is_pending:
                        seen.add(id(pending))
                    while last_offset >= 0:
                        next = pending[last_offset]
                        last_offset -= 1
                        id_next = id(next)
                        if id_next in seen:
                            continue
                        seen.add(id_next)
                        # We will recurse here, so tell dump_object_info to not recurse
                        _scanner.dump_object_info(outf, next, recurse_depth=0)
                        
                        # monkey patch ( try / except ) : discard unsuitable object(s)
                        try:
                            for ref in _scanner.get_referents(next):
                                if id(ref) not in seen:
                                    last_offset += 1
                                    if len(pending) > last_offset:
                                        pending[last_offset] = ref
                                    else:
                                        pending.append(ref)
                        except OverflowError:
                            pass
                
                
                meliae_scanner.dump_all_referenced = dump_all_referenced_monkey_patched
                # noinspection SpellCheckingInspection
                _f = "/tmp/memorydump-%s.json" % daemon_name
                meliae_scanner.dump_all_objects(_f)
                logger.error('(support-only) Memory information dumped to file %s (to be sent to the support)' % _f)
            except ImportError:
                logger.error('(support-only) MEMORY DUMP: FAIL check if meliae lib is installed')
    
    
    @staticmethod
    def print_memory_stats():
        # noinspection SpellCheckingInspection
        if not hasattr(sys, '_debugmallocstats'):
            logger.warn('Cannot print memory stats on this OS')
            return
        
        with tempfile.TemporaryFile() as fp:
            with RedirectStdErrStream(stderr=fp):
                # noinspection PyUnresolvedReferences, PyProtectedMember
                sys._debugmallocstats()
            fp.seek(0)
            _memory_stats = fp.read()
        logger.info(_memory_stats)


memory_stats = MemoryStats()
