#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import base64
import json
import os
import threading
import time
import traceback
import zlib
from abc import ABC

from shinken.http_client import HTTPClient, HTTPExceptions
from shinken.log import logger, LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.item import Item, Items
from shinken.property import BoolProp, IntegerProp, StringProp, DictProp, AddrProp, FloatProp, EditableListProp
from shinken.util import get_obj_name_two_args_and_void, get_obj_name, transform_str_key_dict_into_int_dict
from .configuration_incarnation import ConfigurationIncarnation
from .safepickle import SafeUnpickler

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Dict, Optional, List
    from shinken.log import PartLogger
    from shinken.inter_daemon_message import InterDaemonMessage
    from shinken.objects.shard import Shard
    from shinken.objects.realm import Realm

logger = LoggerFactory.get_logger()


def with_big_lock(func):
    
    def wrapper(self, *args, **kwargs):
        with self.big_lock:
            return func(self, *args, **kwargs)
    
    
    return wrapper


class SatelliteLink(Item, ABC):
    # SatelliteLink is a common Class for link to satellite for    Arbiter with Conf Dispatcher.
    
    id = 0  # each Class will have its own id
    
    con: 'Optional[HTTPClient]'
    uri: str
    alive: bool
    attempt: int
    configuration_incarnation: 'Optional[ConfigurationIncarnation]'
    arbiter_trace: 'Optional[Dict]'
    my_type: str
    logger_name: 'PartLogger'
    logger_ping: 'PartLogger'
    logger_dispatch: 'PartLogger'
    logger_configuration: 'PartLogger'
    logger_initial_check: 'PartLogger'
    already_have_conf: bool
    broks: 'Optional[List]'
    last_check: float
    
    port: int
    address: 'Optional[str]'
    timeout: 'Optional[int]'
    data_timeout: 'Optional[int]'
    check_interval: 'Optional[int]'
    max_check_attempts: 'Optional[int]'
    spare: 'Optional[bool]'
    manage_sub_realms: 'Optional[bool]'
    modules: 'Optional[List]'
    polling_interval: 'Optional[int]'
    use_timezone: 'Optional[str]'
    realm: 'Optional[Realm]'
    satellitemap: 'Optional[Dict]'
    use_ssl: 'Optional[bool]'
    hard_ssl_name_check: 'Optional[bool]'
    enabled: 'Optional[bool]'
    reachable: 'Optional[bool]'
    managed_configuration: 'Optional[Dict]'
    already_have_conf: 'Optional[bool]'
    diff_time_with_arbiter: 'Optional[float]'
    daemon_version: 'Optional[Any]'
    
    properties = Item.properties.copy()
    properties.update({
        'address'            : StringProp(fill_brok=['full_status']),
        'timeout'            : IntegerProp(default='3', fill_brok=['full_status'], to_send=True),
        'data_timeout'       : IntegerProp(default='120', fill_brok=['full_status'], to_send=True),
        'check_interval'     : IntegerProp(default='60', fill_brok=['full_status']),
        'max_check_attempts' : IntegerProp(default='3', fill_brok=['full_status']),
        'spare'              : BoolProp(default='0', fill_brok=['full_status'], to_send=True),
        'manage_sub_realms'  : BoolProp(default='1', fill_brok=['full_status']),
        'modules'            : EditableListProp(default='', to_send=True),  # editable because we can need to add modules dynamically
        'polling_interval'   : IntegerProp(default='1', fill_brok=['full_status'], to_send=True),
        'use_timezone'       : StringProp(default='NOTSET', to_send=True),
        'realm'              : StringProp(default='', fill_brok=['full_status'], brok_transformation=get_obj_name_two_args_and_void, conf_send_preparation=get_obj_name, to_send=True),
        'satellitemap'       : DictProp(default=None, elts_prop=AddrProp, to_send=True, override=True),
        'use_ssl'            : BoolProp(default='0', fill_brok=['full_status']),
        'hard_ssl_name_check': BoolProp(default='0', fill_brok=['full_status']),
        'enabled'            : BoolProp(default='1'),
    })
    
    running_properties = Item.running_properties.copy()
    running_properties.update({
        'con'                      : StringProp(default=None),
        'alive'                    : StringProp(default=True, fill_brok=['full_status']),
        'broks'                    : StringProp(default=[]),
        'attempt'                  : StringProp(default=0, fill_brok=['full_status']),  # the number of failed attempt
        'reachable'                : StringProp(default=False, fill_brok=['full_status']),  # can be network ask or not (dead or check in timeout or error)
        'last_check'               : IntegerProp(default=0, fill_brok=['full_status']),
        'managed_configuration'    : StringProp(default={'schedulers': {}, 'configuration_incarnation_dump': {}, 'activated': False}),
        'already_have_conf'        : StringProp(default=False),
        'diff_time_with_arbiter'   : FloatProp(default=0),
        'daemon_version'           : StringProp(default=None),
        'arbiter_version'          : StringProp(default=None),
        'configuration_incarnation': StringProp(default=None),
        'arbiter_trace'            : StringProp(default=None),
    })
    
    
    def __init__(self, *args, **kwargs):
        super(SatelliteLink, self).__init__(*args, **kwargs)
        
        self.arbiter_satellite_map = {'address': '0.0.0.0', 'port': 0}
        if hasattr(self, 'address'):
            self.arbiter_satellite_map['address'] = self.address
        if hasattr(self, 'port'):
            try:
                self.arbiter_satellite_map['port'] = int(self.port)
            except ValueError:
                pass
        self.big_lock = threading.RLock()
        self.cfg = {}
        self._generate_loggers()
    
    
    def _generate_loggers(self):
        self.logger_name = logger.get_sub_part(self.get_name())
        self.logger_ping = self.logger_name.get_sub_part('IS ALIVE CHECK')
        self.logger_dispatch = self.logger_name.get_sub_part('DISPATCH TO %s' % self.my_type.upper())
        self.logger_configuration = self.logger_dispatch.get_sub_part('CONFIGURATION')
        self.logger_initial_check = self.logger_name.get_sub_part('INITIAL DAEMONS CHECK')
    
    
    def get_name(self):
        raise NotImplementedError()
    
    
    def get_short_spare_display(self):
        return 'SPARE' if self.spare else ''
    
    
    def _reset_managed_configuration(self):
        self.logger_configuration.debug('RESETTING managed configuration')
        self.managed_configuration = {'schedulers': {}, 'configuration_incarnation_dump': {}, 'activated': False}
    
    
    def set_configuration_incarnation(self, configuration_incarnation):
        # type (ConfigurationIncarnation) -> None
        self.configuration_incarnation = configuration_incarnation
    
    
    def set_arbiter_trace(self, arbiter_trace: 'Dict') -> None:
        self.arbiter_trace = arbiter_trace
    
    
    def get_current_managed_configuration(self):
        if not self.managed_configuration['configuration_incarnation_dump']:
            return None
        current_configuration_incarnation = ConfigurationIncarnation.create_from_json(self.managed_configuration['configuration_incarnation_dump'])
        return current_configuration_incarnation
    
    
    def is_managing_the_valid_configuration_incarnation(self):
        current_configuration_incarnation = self.get_current_managed_configuration()
        do_manage_same_configuration_incarnation = self.configuration_incarnation.is_equal(current_configuration_incarnation)
        return do_manage_same_configuration_incarnation
    
    
    def set_arbiter_satellitemap(self, satellitemap):
        # arbiter_satellite_map is the satellitemap in current context:
        #     - A SatelliteLink is owned by an Arbiter
        #     - satellitemap attribute of SatelliteLink is the map defined IN THE satellite configuration
        #       but for creating connections, we need they have the satellite map from the Arbiter
        with self.big_lock:
            self.arbiter_satellite_map = {'address': self.address, 'port': self.port, 'use_ssl': self.use_ssl, 'hard_ssl_name_check': self.hard_ssl_name_check}
            self.arbiter_satellite_map.update(satellitemap)
    
    
    def reset_connection(self):
        with self.big_lock:
            self.con = None
            self.uri = ''
    
    
    def create_connection(self, do_log=False):
        with self.big_lock:
            self.con = HTTPClient(
                address=self.arbiter_satellite_map['address'],
                port=self.arbiter_satellite_map['port'],
                timeout=self.timeout,
                data_timeout=self.data_timeout,
                use_ssl=self.use_ssl,
                strong_ssl=self.hard_ssl_name_check
            )
            self.uri = self.con.uri
            if do_log:
                self.logger_ping.info('Creating new connection to the %s (uri="%s", timeout=%ss, data_timeout=%ss)' % (self.my_type, self.uri, self.timeout, self.data_timeout))
    
    
    def _get_my_realm_name(self):
        if hasattr(self.realm, 'get_name'):
            return self.realm.get_name()
        else:
            return self.realm
    
    
    def put_conf(self, conf):
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # Maybe the connection was not ok, bail out
            if not self.con:
                return False
            
            try:
                t_before = time.time()
                self.con.get('ping')
                self.con.post('put_conf', {'conf': conf}, wait='long')
                self.logger_configuration.debug('Successfully pushed configuration to [%s:%s] in %.3fs' % (self.get_my_type(), self.get_name(), time.time() - t_before))
                # mark conf as sent avoiding sending loop
                self.already_have_conf = True
                
                return True
            except HTTPExceptions as exp:
                self.con = None
                self.logger_configuration.error('Failed to send configuration to %s: %s' % (self.get_name(), str(exp)))
                return False
    
    
    # Get and clean all of our broks
    def get_all_broks(self):
        with self.big_lock:
            res = self.broks
            self.broks = []
            return res
    
    
    def is_alive(self):
        return self.alive
    
    
    def __set_alive_state(self) -> bool:
        was_alive = self.alive
        self.alive = True
        self.attempt = 0
        self.reachable = True
        return was_alive
    
    
    # Set alive, reachable, and reset attempts.
    # If we change state, raise a status brok update
    def set_alive(self):
        with self.big_lock:
            was_alive = self.__set_alive_state()
            
            # We came from dead to alive, so we must add a brok update
            if not was_alive:
                self.logger_ping.info('ALIVE The %s is back alive.' % (self.get_my_type()))
                b = self.get_update_status_brok()
                self.broks.append(b)
    
    
    def __set_dead_state(self) -> bool:
        with self.big_lock:
            was_alive = self.alive
            self.alive = False
            self.con = None
            self._reset_managed_configuration()
            return was_alive
    
    
    def set_dead(self, reason):
        with self.big_lock:
            was_alive = self.__set_dead_state()
            
            # We are dead now. Must raise
            # a brok to say it
            if was_alive:
                self.logger_ping.error('The %s is considered as dead after %d/%d successive FAILED ping attempts. Last HTTP(s) ping error: "%s"' % (self.get_my_type(), self.attempt, self.max_check_attempts, reason))
                b = self.get_update_status_brok()
                self.broks.append(b)
            else:  # Still dead
                self.logger_ping.info('HTTP(s) ping check did fail. Still DEAD (fail more than %s max attempts). HTTP(s) error: "%s"' % (
                    self.max_check_attempts, reason))
    
    
    # Go in reachable=False and add a failed attempt
    # if we reach the max, go dead
    def _add_failed_check_attempt(self, reason=''):
        with self.big_lock:
            self.reachable = False
            self.attempt += 1
            self.attempt = min(self.attempt, self.max_check_attempts)
            
            # check when we just go HARD (dead)
            if self.attempt >= self.max_check_attempts:
                self.set_dead(reason)
            else:
                # Don't need to warn again and again if the satellite is already dead
                if self.alive:
                    self.logger_ping.warning('HTTP(s) ping check did fail. Current failed attempts: %d/%d (max) before going DEAD. HTTP(s) error: "%s"' % (
                        self.attempt, self.max_check_attempts, reason))
    
    
    # Update satellite info each self.check_interval seconds,
    # so we smooth arbiter actions for just useful actions
    # and not cry for a little timeout
    def update_infos(self):
        with self.big_lock:
            # First look if it's not too early to ping
            now = time.time()
            
            since_last_check = now - self.last_check
            if since_last_check < self.check_interval:
                return
            
            self.last_check = now
            
            # We ping and update the managed list
            is_ok = self.ping()
            if is_ok:  # don't try to do a managed configuration call if the ping did fail
                self._notify_presence()
                self._update_managed_configuration()
            
            # Update the state of this element
            b = self.get_update_status_brok()
            self.broks.append(b)
    
    
    # The elements just got a new conf_id, we put it in our list
    # because maybe the satellite is too busy to answer now
    def known_conf_managed_push(self, shard_id, scheduler_name):
        with self.big_lock:
            self.managed_configuration['schedulers'][shard_id] = scheduler_name
            # We are also sure that now the daemon is up-to-date
            self.managed_configuration['configuration_incarnation_dump'] = self.configuration_incarnation.dump_as_json()
    
    
    def get_string_of_managed_schedulers(self):
        with self.big_lock:
            d = {}
            for (shard_id, scheduler_name) in self.managed_configuration['schedulers'].items():
                d[scheduler_name] = shard_id
        # using only locals, no more need lock
        scheduler_names = sorted(d.keys())
        lst = ['%s/%s' % (scheduler_name, d[scheduler_name]) for scheduler_name in scheduler_names]
        if len(lst) == 0:
            return 'No schedulers'
        return ', '.join(lst)
    
    
    # We just ask a shard to be removed from the daemon, and it was done by the distant daemon,
    # so update our view of its configuration in memory
    def _remove_shard_assignation_in_memory(self, shard_id):
        with self.big_lock:
            if shard_id in self.managed_configuration['schedulers']:
                del self.managed_configuration['schedulers'][shard_id]
                self.cfg['schedulers'].pop(shard_id, None)
    
    
    def is_managing_any_shard(self):
        with self.big_lock:
            return len(self.managed_configuration['schedulers']) != 0
    
    
    @with_big_lock
    def set_as_inactive(self) -> None:
        self.cfg['schedulers'] = {}
        self.cfg['activated'] = False
        self.cfg['arbiter_trace'] = self.get_my_arbiter_trace()
        
        is_managing_the_valid_config_incarnation = self.is_managing_the_valid_configuration_incarnation()
        
        # LONG CALL
        is_sent = self.put_conf(self.cfg)
        if is_sent:
            # Log if we did send it a new incarnation
            if not is_managing_the_valid_config_incarnation:
                self.logger_configuration.info('SATELLITE SENT OK  The daemon is now updated to the new configuration incarnation (%s)' % self.configuration_incarnation)
            # Let the memory info be up-to-date
            self.managed_configuration['schedulers'].clear()
            self.managed_configuration['activated'] = False
            self.managed_configuration['configuration_incarnation_dump'] = self.configuration_incarnation.dump_as_json()
    
    
    @with_big_lock
    def push_shard_in_cfg(self, shard_id: int, scheduler_info: 'Dict', arbiters_info: 'Dict') -> None:
        self.cfg['schedulers'][shard_id] = scheduler_info
        self.cfg['arbiters'] = arbiters_info
        self.cfg['activated'] = True
    
    
    @with_big_lock
    def assert_do_not_manage_shard(self, shard: 'Shard') -> None:
        shard_id = shard.get_id()
        
        if shard.is_assigned:
            assign_msg = '(managed by %s)' % shard.get_current_scheduler_name()
        else:
            assign_msg = '(not managed by any schedulers)'
        
        # Maybe this satellite already got this configuration, so skip it
        if self.do_i_manage_any_scheduler(shard_id):
            was_removed = self.remove_from_conf(shard_id, 'no more useful for this daemon')
            if was_removed:
                self.logger_configuration.info('[SPARE/IDLE] The shard %s %s was removed as it is no more useful' % (shard_id, assign_msg))
                if not self.is_managing_any_shard():
                    self.set_as_inactive()
            else:
                self.logger_configuration.warning('[SPARE/IDLE] The shard %s %s was not removed, retrying later' % (shard_id, assign_msg))
        return
    
    
    def get_known_shard_ids(self):
        with self.big_lock:
            return list(self.managed_configuration['schedulers'].keys())
    
    
    def get_all_my_managed_realms(self):
        managed_realms = [self.realm]  # we are always managing our own realm
        if not self.manage_sub_realms:  # no sub realms, finish
            return managed_realms
        # so ask our realm about its sub realms
        managed_realms.extend(self.realm.get_all_my_lower_realms())
        return managed_realms
    
    
    # We need an arbiter trace, but with a specific expire period time based on OUR check_interval
    def get_my_arbiter_trace(self):
        if not self.arbiter_trace:
            return None
        
        my_arbiter_trace = self.arbiter_trace.copy()  # copy as we modify it
        my_arbiter_trace['expire_period'] = self.check_interval * self.max_check_attempts
        my_arbiter_trace['arbiter_time'] = time.time()  # let the daemon know which time this arbiter is
        
        return my_arbiter_trace
    
    
    ##
    # Called by arbiters to notify satellites of the main arbiter.
    ##
    def _notify_presence(self):
        if not self.arbiter_trace:
            return
        
        with self.big_lock:
            try:
                if self.con is None:
                    self.create_connection()
                
                # If the connection failed to initialize, bail out
                if self.con is None:
                    return
                my_arbiter_trace = self.get_my_arbiter_trace()
                update_trace = self.con.post('arbiter_traces_register', {'arbiter_trace': my_arbiter_trace}, wait='long')
                update_trace = json.loads(update_trace)
                try:
                    self.diff_time_with_arbiter = update_trace['diff_time_with_arbiter']
                    self.daemon_version = update_trace.get('daemon_version', None)
                except TypeError:
                    self.logger_ping.warning("The distant daemon [%s] at [%s] don't have time information please update it." % (self.get_name(), self.uri))
            
            except HTTPExceptions as exp:
                self.logger_ping.info('Extended ping [%s] at [%s] KO (%s)' % (self.get_name(), self.uri, exp))
    
    
    def ping(self) -> bool:
        with self.big_lock:
            try:
                if self.con is None:
                    self.create_connection(do_log=True)
                
                # If the connection failed to initialize, bail out
                if self.con is None:
                    self._add_failed_check_attempt()
                    return False
                
                r = self.con.get('ping')
                
                # Should return us pong string
                if r == 'pong':
                    self.set_alive()
                    self.logger_ping.debug('Ping of [%s] [uri=%s] => OK' % (self.get_name(), self.uri))
                    return True
                else:
                    self.logger_ping.debug('Ping of [%s] [uri=%s] => FAIL' % (self.get_name(), self.uri))
                    self._add_failed_check_attempt()
                    return False
            except HTTPExceptions as exp:
                self.logger_ping.debug('Ping of [%s] [uri=%s] => FAIL' % (self.get_name(), self.uri))
                self._add_failed_check_attempt(reason=str(exp))
                return False
    
    
    def initial_daemon_check(self, max_initial_daemons_check_time):
        try:
            self._do_initial_daemon_check(max_initial_daemons_check_time)
        except Exception:  # noqa => catch al of a thread
            logger.error('The %s initial daemon check thread did crash: %s. Exiting daemon' % (self.get_name(), traceback.format_exc()))
            os._exit(2)  # noqa => raw kill
    
    
    def _do_initial_daemon_check(self, max_initial_daemons_check_time):
        self.logger_initial_check.debug('The initial check of the daemon is starting (max=%ss)' % max_initial_daemons_check_time)
        start = time.time()
        
        last_err = ''
        while True:
            # Timeout?
            if abs(time.time() - start) >= max_initial_daemons_check_time:
                self.__set_dead_state()  # set dead, but no log
                self.logger_initial_check.error(
                    'The daemon is set to DEAD after initial daemon check, fail to contact it after %ds, last try error: "%s"' % (max_initial_daemons_check_time, last_err))
                return
            
            with self.big_lock:
                try:
                    if self.con is None:
                        self.create_connection(do_log=False)
                    
                    # If the connection failed to initialize, bail out
                    if self.con is None:
                        time.sleep(0.1)  # don't hammer CPU
                        continue
                    
                    r = self.con.get('ping')
                    
                    # If at least one response: ALIVE
                    if r == 'pong':
                        self.__set_alive_state()
                        self.logger_initial_check.info('The daemon is ALIVE after initial daemon check, connection was OK')
                        return
                    time.sleep(0.1)  # don't hammer CPU
                except HTTPExceptions as exp:
                    last_err = str(exp)  # do NOT keep exception object, only the string
                    if 'Connection refused' in last_err:
                        self.__set_dead_state()  # set dead, but no log
                        self.logger_initial_check.error('The daemon is set to DEAD after initial daemon check, port is closed: "%s"' % last_err)
                        return
                    self.logger_initial_check.debug('Ping of [uri=%s] => FAIL %s' % (self.uri, last_err))
                    time.sleep(0.1)  # don't hammer CPU
        
        self.logger_initial_check.debug('The initial check of the daemon finished in %.2fs' % (time.time() - start))
    
    
    def wait_new_conf(self):
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            try:
                self.con.get('wait_new_conf')
                return True
            except HTTPExceptions:
                self.con = None
                return False
    
    
    def remove_from_conf(self, shard_id, reason):
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            try:
                self.con.get('remove_from_conf', {'shard_id': shard_id, 'reason': reason})
                self._remove_shard_assignation_in_memory(shard_id)
                return True
            except HTTPExceptions:
                self.con = None
                return False
    
    
    def assert_only_known_shard_ids(self, valid_shard_ids):
        with self.big_lock:
            current_shard_ids = set(self.managed_configuration['schedulers'].keys())
            valid_shard_ids = set(valid_shard_ids)
            unknown_shard_ids = current_shard_ids - valid_shard_ids
            # Now unknown shard_ids, all is OK
            if len(unknown_shard_ids) == 0:
                self.logger_configuration.debug('All our %s known shards are valid' % len(current_shard_ids))
                return
            # There are some unknown shard ids that must be deleted (maybe old/removed schedulers)
            for shard_id in unknown_shard_ids:
                scheduler_name = self.managed_configuration['schedulers'][shard_id]
                _reason = 'The scheduler "%s" is removed as it is no more used ( old scheduler managing the shard=%s that was either deleted or disabled ).' % (scheduler_name, shard_id)
                was_deleted = self.remove_from_conf(shard_id, _reason)
                if not was_deleted:
                    self.logger_configuration.warning('Cannot remove the unknown shard_id %s (from delete/disabled scheduler %s) from the daemon, we will retry the next turn' % (shard_id, scheduler_name))
                else:
                    self.logger_configuration.info('[ DELETE / DISABLED SCHEDULER ] %s' % _reason)
    
    
    def _update_managed_configuration(self):
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            try:
                managed_configuration = self.con.get('get_currently_managed_configuration')
            except HTTPExceptions as exp:
                self.logger_configuration.warning('Call to know what the daemon is currently managing did fail: %s' % exp)
                return
            
            if managed_configuration is None:
                self.already_have_conf = False
                self._reset_managed_configuration()
                return
            
            # Ok protect against json that is changing keys as string instead of int
            schedulers = transform_str_key_dict_into_int_dict(managed_configuration['schedulers'])
            if schedulers is None:
                self.logger_configuration.error('Call to know what the daemon is currently managing did fail: the return is malformed %s' % managed_configuration)
                self.con = None
                self._reset_managed_configuration()
                return
            managed_configuration['schedulers'] = schedulers
            
            self.already_have_conf = True
            self.managed_configuration = managed_configuration
    
    
    # Return True if the satellite said to manage a shard with this scheduler
    def do_i_manage_this_scheduler(self, shard_id, scheduler_name):
        with self.big_lock:
            managed_scheduler_name = self.managed_configuration['schedulers'].get(shard_id, None)
            return scheduler_name == managed_scheduler_name
    
    
    # Return True if the satellite said to manage a shard, whatever the scheduler is
    def do_i_manage_any_scheduler(self, shard_id):
        with self.big_lock:
            return shard_id in self.managed_configuration['schedulers']
    
    
    def is_activated(self):
        with self.big_lock:
            return self.managed_configuration['activated']
    
    
    def assert_spare_sleeping_if_void(self):
        with self.big_lock:
            # Only for spare
            if not self.spare:
                return
            # If we are void, maybe
            # * the spare is still running (from a previous give)
            # * the spare is still with an old configuration
            if not self.is_managing_any_shard() and (self.is_activated() or not self.is_managing_the_valid_configuration_incarnation()):
                self.set_as_inactive()
            
            # Un initialized spare must have a conf, with at least the fact they are spare and idle
            if self.managed_configuration['configuration_incarnation_dump'] == {}:
                self.set_as_inactive()
            self.logger_configuration.debug('ASSERT SPARE SLEEPING => %s' % self.managed_configuration)
    
    
    def push_broks(self, broks):
        # If there are no broks, skip all of this
        if len(broks) == 0:
            return
        
        # If the broker is sleeping, do nothing
        if not self.is_activated():
            return
        
        before = time.time()
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return False
            
            try:
                # Always do a simple ping to avoid a LONG lock
                self.con.get('ping')
                self.con.post('push_broks', {'broks': broks}, wait='long')
                self.logger_name.info('%s Broks are send to the broker [ %.3f ]s' % (len(broks), time.time() - before))
                return True
            except HTTPExceptions:
                self.con = None
                return False
    
    
    def get_external_commands(self):
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return []
            
            try:
                self.con.get('ping')
                tab = self.con.get('get_external_commands', wait='long')
                tab = base64.b64decode(tab)
                tab = zlib.decompress(tab)
                tab = SafeUnpickler.loads(tab, 'External commands get from %s' % self.get_name())
                # Protect against bad return
                if not isinstance(tab, list):
                    self.con = None
                    return []
                return tab
            except HTTPExceptions:
                self.con = None
                return []
            except AttributeError:
                self.con = None
                return []
            except Exception as exp:
                logger.error('Unexpected error in get_external_commands: %s' % exp)
                self.con = None
                return []
    
    
    def get_messages(self) -> 'List[InterDaemonMessage]':
        
        # If the satellite is sleeping, do nothing
        if not self.is_activated():
            return []
        
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return []
            
            try:
                self.con.get('ping')
                get_messages_buffer = self.con.get('get_messages', wait='long')
                get_messages_buffer = base64.b64decode(get_messages_buffer)
                get_messages_buffer = zlib.decompress(get_messages_buffer)
                messages = SafeUnpickler.loads(get_messages_buffer, 'get_messages from %s' % self.get_name())
                # Protect against bad return
                if not isinstance(messages, list):
                    self.con = None
                    return []
                return messages
            except HTTPExceptions:
                self.con = None
                return []
            except AttributeError:
                self.con = None
                return []
            except Exception as exp:
                logger.error('Unexpected error in get_messages: %s' % exp)
                self.con = None
                return []
    
    
    def send_messages(self, messages: 'List[InterDaemonMessage]') -> bool:
        if not messages:
            return False
        
        # If the satellite is sleeping, do nothing
        if not self.is_activated():
            return False
        
        before = time.time()
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return False
            
            try:
                # Always do a simple ping to avoid a LONG lock
                self.con.get('ping')
                self.con.post('push_messages', {'messages': messages}, wait='long')
                self.logger_name.info('Send 〖%s〗 messages to the %s in 〖%s〗s' % (len(messages), self.my_type, logger.format_chrono(before)))
                return True
            except HTTPExceptions:
                self.con = None
                return False
    
    
    def prepare_for_conf(self) -> None:
        with self.big_lock:
            self.cfg = {
                'name'                     : self.get_name(),
                'global'                   : {},
                'schedulers'               : {},
                'arbiters'                 : {},
                'configuration_incarnation': self.configuration_incarnation,
            }
            properties = self.__class__.properties
            running_properties = self.__class__.running_properties
            property_lists = (properties, running_properties)
            for property_list in property_lists:
                for prop, entry in property_list.items():
                    if entry.to_send:
                        val = getattr(self, prop)
                        if entry.conf_send_preparation is not None:
                            val = entry.conf_send_preparation(val)
                        self.cfg['global'][prop] = val
    
    
    # Some parameters for satellites are not defined in the satellites' conf
    # but in the global configuration. We can pass them in the global
    # property
    def add_global_conf_parameters(self, params):
        with self.big_lock:
            for prop in params:
                self.cfg['global'][prop] = params[prop]
    
    
    def get_my_type(self):
        return self.__class__.my_type
    
    
    # Here for poller and reactionner. Scheduler have its own function
    def give_satellite_cfg(self):
        with self.big_lock:
            return {
                'port'               : self.port,
                'address'            : self.address,
                'use_ssl'            : self.use_ssl,
                'hard_ssl_name_check': self.hard_ssl_name_check,
                'name'               : self.get_name(),
                'instance_id'        : self.id,
                'realm'              : self.realm.realm_name,
                'active'             : True,
                'passive'            : getattr(self, 'passive', False),
                'poller_tags'        : getattr(self, 'poller_tags', []),
                'reactionner_tags'   : getattr(self, 'reactionner_tags', []),
                'keep_timeout_time'  : getattr(self, 'keep_timeout_time', []),
                'exec_stat_range'    : getattr(self, 'exec_stat_range', ''),
                'timeout'            : self.timeout,
                'data_timeout'       : self.data_timeout,
            }
    
    
    # Call by pickle for dataify the downtime
    # because we DO NOT WANT REF in this pickled data !
    def __getstate__(self):
        cls = self.__class__
        # id is not in *_properties
        res = {'id': self.id}
        for prop in cls.properties:
            if hasattr(self, prop):
                res[prop] = getattr(self, prop)
        for prop in cls.running_properties:
            if prop != 'con':
                if hasattr(self, prop):
                    res[prop] = getattr(self, prop)
        return res
    
    
    # Inverted function of getstate
    def __setstate__(self, state):
        cls = self.__class__
        
        self.id = state['id']
        for prop in cls.properties:
            if prop in state:
                setattr(self, prop, state[prop])
        for prop in cls.running_properties:
            if prop in state:
                setattr(self, prop, state[prop])
        # con needs to be explicitly set:
        self.con = None
        # As the serialized object do not have the lock (cannot serialize it) we recreate it
        self.big_lock = threading.RLock()
        # Loggers are not saved during pickle, so regenerate them
        self._generate_loggers()


class BaseSatelliteLinks(Items):
    
    # We create the reversed list so search will be faster
    # We also create a duplicates list with id of duplicates
    def find_duplicates(self):
        reversed_list = {}
        duplicates = []
        key_properties = ('address', 'port')
        for id_ in self.items:
            keys = tuple([getattr(self.items[id_], property_name, 'None') for property_name in key_properties])
            if keys in reversed_list:
                duplicated_item = self.items[reversed_list[keys]]
                duplicates.append((self.items[id_], duplicated_item))
                object_names = ['%s and %s' % (getattr(dup[0], 'imported_from'), getattr(dup[1], 'imported_from')) for dup in duplicates]
                error_str = 'Two %s daemons are defined with the same address and port in config files : %s' % (self.inner_class.my_type, ', '.join(object_names))
                # Pythonize
                if BoolProp().pythonize(duplicated_item.enabled) and BoolProp().pythonize(self.items[id_].enabled):
                    self.configuration_errors.append(error_str)
            else:
                reversed_list[keys] = id_
        
        return duplicates
    
    
    def is_correct(self):
        r = super(BaseSatelliteLinks, self).is_correct()
        
        if self.find_duplicates():
            r = False
        
        return r


class SatelliteLinks(BaseSatelliteLinks):
    
    # We must have a realm property, so we find our realm
    def linkify(self, realms, modules):
        self.linkify_s_by_p(realms)
        self.linkify_s_by_plug(modules)
    
    
    def linkify_s_by_p(self, realms):
        for s in self:
            p_name = s.realm.strip()
            # If no realm name, take the default one
            if p_name == '':
                p = realms.get_default()
                s.realm = p
            else:  # find the realm one
                p = realms.find_by_name(p_name)
                s.realm = p
            # Check if what we get is OK or not
            if p is not None:
                s.register_to_my_realm()
            else:
                err = "The %s %s got a unknown realm '%s'" % (s.__class__.my_type, s.get_name(), p_name)
                s.configuration_errors.append(err)
