# !/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict, Iterator, Sequence, Tuple, Any, Callable


class CacheStorageException(Exception):
    pass


class CacheItemNotFound(CacheStorageException):
    def __init__(self, object_id, message=''):
        # type: (str, str) -> None
        if not message:
            message = 'Item with id [ %s ] not found in cache' % object_id
        self.object_id = object_id
        self.message = message


class CacheStorage:  # TODO (Python 3): typing.Generic
    def __init__(self):
        # type: () -> None
        self._cache_list = {}  # type: Dict[str, Any]
    
    
    def flush(self):
        self._cache_list.clear()
    
    
    def invalidate(self, predicate):
        # type: (Callable[[str, Any], bool]) -> List[str]
        to_delete = [object_id for object_id, object_in_cache in self._cache_list.items() if predicate(object_id, object_in_cache)]
        for object_id in to_delete:
            self._cache_list.pop(object_id, None)
        return to_delete
    
    
    def invalidate_by_id(self, predicate):
        # type: (Callable[[str], bool]) -> List[str]
        to_delete = [object_id for object_id in self._cache_list.keys() if predicate(object_id)]
        for object_id in to_delete:
            self._cache_list.pop(object_id, None)
        return to_delete
    
    
    def invalidate_by_object(self, predicate):
        # type: (Callable[[Any], bool]) -> List[str]
        to_delete = [object_id for object_id, object_in_cache in self._cache_list.items() if predicate(object_in_cache)]
        for object_id in to_delete:
            self._cache_list.pop(object_id, None)
        return to_delete
    
    
    def get_all_ids(self):
        # type: () -> Sequence[str]
        return list(self._cache_list.keys())
    
    
    def iter_objects_with_id(self, lazy):
        # type: (bool) -> Iterator[Tuple[str, Any]]
        return iter(list(self._cache_list.items()) if not lazy else iter(self._cache_list.items()))
    
    
    def iter_objects(self, lazy):
        # type: (bool) -> Iterator[Any]
        return iter(list(self._cache_list.values()) if not lazy else iter(self._cache_list.items()))
    
    
    def get_object(self, object_id, pop=False):
        # type: (str, bool) -> Any
        get_object = self._cache_list.__getitem__ if not pop else self._cache_list.pop
        try:
            object_cache = get_object(object_id)
        except (KeyError, TypeError):
            raise CacheItemNotFound(object_id)
        return object_cache
    
    
    def get_object_default(self, object_id, default, pop=False):
        # type: (str, Any, bool) -> Any
        try:
            return self.get_object(object_id, pop=pop)
        except CacheItemNotFound:
            return default
    
    
    def set_object_default(self, object_id, default):
        # type: (str, Any) -> Any
        return self._cache_list.setdefault(object_id, default)
    
    
    def store_object(self, object_id, object_to_cache):
        # type: (str, Any) -> None
        self._cache_list[object_id] = object_to_cache
    
    
    def delete_object(self, object_id, raising=False):
        try:
            del self._cache_list[object_id]
        except KeyError:
            if raising:
                raise CacheItemNotFound(object_id)
