#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import json
import os
import random
import shutil
import time
from subprocess import Popen, PIPE

from shinkensolutions.os_helper import set_ownership

# (!) Must end with a /
WHISPER_DIR = '/opt/graphite/storage/whisper/'
NB_METRICS_COUNT_FILE = '/opt/graphite/storage/whisper/.nb_metrics'


class GRAPHITE_STATS_KEY:
    TIME_READ = 'graphite_stats_time'
    LOCAL_TIME = 'graphite_local_time'


class GraphiteMetricsCounter:
    UNWANTED_METRIC_DIRECTORIES = ('carbon', '.cacheinvalidation', '.cacheinvalidation.lock', '.nb_metrics', 'dummy.txt')
    
    
    def __init__(self):
        self.metrics = 0
        self.level_0 = 0
    
    
    def update_count(self):
        self._update_level_0()
        self._do_compute_number_of_metrics_file()
        graphite_stats_time = int(time.time())
        self._write_metric_file(graphite_stats_time)
    
    
    def _update_level_0(self):
        self.level_0 = len([folder for folder in os.listdir(WHISPER_DIR) if folder not in self.UNWANTED_METRIC_DIRECTORIES])
    
    
    # This will be computed by the iostats_collector script
    def _do_compute_number_of_metrics_file(self):
        # type: () -> None
        self.metrics = 0
        
        try:
            # It's fastest to go with a find command instead of doing it ourselves
            cmd = r'''find %s -type f  -name '*.wsp'  \! -path '%scarbon/*' -printf '.' | wc -c''' % (WHISPER_DIR, WHISPER_DIR)
            p = Popen(cmd, stdout=PIPE, stderr=PIPE, shell=True)
            output, stderr = p.communicate()
            self.metrics = int(output.strip())
            return
        except Exception:
            pass
        # oops, something was wrong, switch to a manual mode (maybe we are no mor allowed to launch command, like under apache)
        for root, dirs, files in os.walk(WHISPER_DIR):
            root = root.replace(WHISPER_DIR, '')
            if not root or root.startswith(self.UNWANTED_METRIC_DIRECTORIES):
                continue
            for basename in files:
                if basename.endswith('.wsp'):
                    self.metrics += 1
    
    
    def _get_count_export(self, graphite_stats_time):
        return {
            'metrics'                   : self.metrics,
            'level_0'                   : self.level_0,
            GRAPHITE_STATS_KEY.TIME_READ: graphite_stats_time
        }
    
    
    def _write_metric_file(self, graphite_stats_time):
        # type: (int) -> None
        tmp_file = '%s.tmp.%d' % (NB_METRICS_COUNT_FILE, random.randint(1, 100000))
        count = self._get_count_export(graphite_stats_time)
        with open(tmp_file, 'w') as f:
            f.write(json.dumps(count))
        
        set_ownership(tmp_file, 'apache', 'apache')
        shutil.move(tmp_file, NB_METRICS_COUNT_FILE)  # atomic move
