# !/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

import errno
import os
import sys

# DO NOT IMPORT SHINKEN IN THIS FILE
TYPE_HINT = False
if TYPE_HINT:
    from shinken.misc.type_hint import List, Tuple
    from shinken.log import PartLogger

if sys.platform.startswith('win'):
    pwd = None
    grp = None
    
    
    def rename_file(source, target):
        # type: (str, str) -> None
        if os.path.exists(target):
            os.remove(target)
        os.rename(source, target)
    
    
    def get_cur_user_id():
        # type: () -> int
        return 0
    
    
    def get_cur_group_id():
        # type: () -> int
        return 0
    
    
    def get_cur_user_name():
        # type: () -> str
        return 'shinken'
    
    
    def get_cur_group_name():
        # type: () -> str
        return 'shinken'
    
    
    # noinspection PyUnusedLocal
    def get_user_name_from_id(user_id):
        # type: (int) -> str
        return 'shinken'
    
    
    # noinspection PyUnusedLocal
    def get_group_name_from_id(group_id):
        # type: (int) -> str
        return 'shinken'
    
    
    # noinspection PyUnusedLocal
    def get_user_id_from_name(user_name):
        # type: (str) -> int
        return 0
    
    
    # noinspection PyUnusedLocal
    def get_group_id_from_name(group_name):
        # type: (str) -> int
        return 0
    
    
    # noinspection PyUnusedLocal
    def get_file_owner_and_group_name(path):
        # type: (str) -> Tuple[str,str]
        return 'shinken', 'shinken'
    
    
    # noinspection PyUnusedLocal
    def set_ownership(path, user='shinken', group='shinken', is_link=False):
        # type: (str, str, str, bool) -> None
        pass
    
    
    def get_all_group():
        # type: () -> List
        return []
    
    
    # noinspection PyUnusedLocal
    def get_file_info(path):
        return 777, 'shinken', 'shinken'

else:
    import pwd
    import grp
    
    # SEF-11935 getpwuid_r internally uses a lock which may be held by a thread while forking. We cache uid / name mappings to avoid calling functions from pwd or grp modules
    __UID_TO_NAME: 'dict[int,str]' = {}
    __NAME_TO_UID: 'dict[str,int]' = {}
    __GID_TO_NAME: 'dict[int,str]' = {}
    __NAME_TO_GID: 'dict[str,int]' = {}
    
    
    def rename_file(source, target):
        # type: (str, str) -> None
        os.rename(source, target)
    
    
    def get_cur_user_id():
        # type: () -> int
        return os.getuid()
    
    
    def get_cur_group_id():
        # type: () -> int
        return os.getgid()
    
    
    def get_cur_user_name():
        # type: () -> str
        return get_user_name_from_id(get_cur_user_id())
    
    
    def get_cur_group_name():
        # type: () -> str
        return get_group_name_from_id(get_cur_group_id())
    
    
    def get_user_name_from_id(user_id):
        # type: (int) -> str
        if user_id not in __UID_TO_NAME:
            __UID_TO_NAME[user_id] = pwd.getpwuid(user_id).pw_name
        return __UID_TO_NAME[user_id]
    
    
    def get_group_name_from_id(group_id):
        # type: (int) -> str
        if group_id not in __GID_TO_NAME:
            __GID_TO_NAME[group_id] = grp.getgrgid(group_id).gr_name
        return __GID_TO_NAME[group_id]
    
    
    def get_user_id_from_name(user_name):
        # type: (str) -> int
        try:
            if user_name not in __NAME_TO_UID:
                __NAME_TO_UID[user_name] = pwd.getpwnam(user_name).pw_uid
            return __NAME_TO_UID[user_name]
        except KeyError:
            raise OSError(errno.EINVAL, 'This user does not exists', user_name)
    
    
    def get_group_id_from_name(group_name):
        # type: (str) -> int
        try:
            if group_name not in __NAME_TO_GID:
                __NAME_TO_GID[group_name] = grp.getgrnam(group_name).gr_gid
            return __NAME_TO_GID[group_name]
        except KeyError:
            raise OSError(errno.EINVAL, 'This group does not exists', group_name)
    
    
    def get_file_owner_and_group_name(path):
        # type: (str) -> Tuple[str,str]
        stat = os.stat(path)
        user_id = stat.st_uid
        group_id = stat.st_gid
        
        return get_user_name_from_id(user_id), get_group_name_from_id(group_id)
    
    
    def set_ownership(path, user='shinken', group='shinken', is_link=False):
        # type: (str, str, str, bool) -> None
        
        if get_cur_user_id() != 0 and not (get_cur_user_name() == 'shinken' and user == 'shinken' and group in ('shinken', 'apache')):
            return
        uid = get_user_id_from_name(user)
        gid = get_group_id_from_name(group)
        os.chown(path, uid, gid)
        if is_link:
            os.lchown(path, uid, gid)
    
    
    def get_all_group():
        # type: () -> List
        return grp.getgrall()
    
    
    def get_file_info(path):
        file_stat = os.stat(path)
        # thanks to https://github.com/naparuba/opsbro/blob/master/opsbro/compliancemgr.py#L69
        file_permissions = int(oct(file_stat.st_mode & 0o777)[2:])  # => to have something like 644
        user_id = file_stat.st_uid
        group_id = file_stat.st_gid
        return file_permissions, get_user_name_from_id(user_id), get_group_name_from_id(group_id)


def make_dirs_and_chown_shinken(directory, mode=0o777):
    # type: (str, int) -> None
    head, tail = os.path.split(directory)
    if not tail:
        head, tail = os.path.split(head)
    if head and tail and not os.path.exists(head):
        make_dirs_and_chown_shinken(head, mode)
        if tail == os.curdir:
            return
    
    try:
        os.mkdir(directory, mode)
    except OSError as e:
        # be happy if someone already created the path
        if e.errno != errno.EEXIST:
            raise
    
    if get_cur_user_id() == 0:
        set_ownership(directory, 'shinken', 'shinken')


def make_file_and_chown_shinken(file_path):
    # type: (str) -> None
    with open(file_path, 'w') as _file:
        _file.write('')
    set_ownership(file_path, 'shinken', 'shinken')


def set_shinken_owner_on_file_or_directory(path, logger=None):
    # type: (str, PartLogger) -> None
    try:
        set_ownership(path, 'shinken', 'shinken')
    except Exception as e:
        if logger:
            logger.error('Can\'t give ownership to shinken:shinken on [ %s ]' % path)
        raise e


def verify_if_shinken_own_file_or_directory(path, logger=None, skip_test_if_not_root=True):
    # type: (str, PartLogger, bool) -> bool
    if skip_test_if_not_root and get_cur_user_id() != 0:
        # Non-root user cannot create file with different owner, no need to crash if write operation is possible
        return True
    
    user_owner, group_owner = get_file_owner_and_group_name(path)
    
    result = True
    if user_owner != 'shinken':
        if logger:
            logger.error('[ %s ] user owner is not [ shinken ] but [ %s ]' % (path, user_owner))
        result = False
    
    if group_owner != 'shinken':
        if logger:
            logger.error('[ %s ] group owner is not [ shinken ] but [ %s ]' % (path, group_owner))
        result = False
    
    return result


def verify_if_dir_or_file_exist(path, logger=None):
    if not os.path.exists(path):
        if logger:
            logger.error('[ %s ] file does not exist' % path)
        return False
    return True


def verify_file_exist_and_can_be_read_by_shinken(path, logger=None):
    # type: (str, PartLogger) -> bool
    if not verify_if_dir_or_file_exist(path, logger=logger):
        return False
    return verify_if_shinken_own_file_or_directory(path, logger=logger, skip_test_if_not_root=False)
