#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import hashlib
import os
import time
import uuid
from collections import namedtuple

import bson
import pymongo
from pymongo.errors import AutoReconnect, ConnectionFailure, PyMongoError

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.thread_helper import Thread
from ..data_hub.data_hub_exception.data_hub_exception import DataHubItemNotFound

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Any, Union, List, Optional
    from pymongo import MongoClient
    from shinkensolutions.data_hub.data_hub import DataHub
    from .sshtunnelmongomgr import ConnectionResult

DatabaseStatisticsItem = namedtuple('DatabaseStatisticsItem', ['uri_hash', 'data'])


class DatabaseStatistics(Thread):
    def __init__(self, uri, connection_result, data_hub, loop_speed, requestor, get_statistics_interval=60):
        # type:(str, ConnectionResult, DataHub, int, str, int) -> None
        super(DatabaseStatistics, self).__init__(loop_speed=loop_speed)
        self._uuid = uuid.uuid4().hex  # type: str
        self.logger = self.logger.get_sub_part(f'{os.getpid()}').get_sub_part('MONGO').get_sub_part('THREAD STATISTICS').get_sub_part(self.get_uuid(), register=False)
        self._uri = uri  # type: str
        self._uri_hash = hashlib.md5(self._uri.strip().encode('utf8', 'ignore')).hexdigest()  # type: str
        self.connection_result = connection_result  # type: ConnectionResult
        self._data_hub = data_hub  # type: DataHub
        self._get_statistics_interval = get_statistics_interval
        self._last_database_statistics_time = 0
        self._is_connected = False
        self._connection_state_message = 'Database information never retrieved yet'
        self._last_ping_time = 0
        self.logger.info('Creating Thread for %s' % requestor)
    
    
    def get_thread_name(self):
        # type:() -> str
        return 'db-stat'
    
    
    def get_uuid(self):
        # type:() -> str
        return self._uuid
    
    
    def get_uri(self):
        # type: () -> str
        return self._uri
    
    
    def get_uri_hash(self):
        # type: () -> str
        return self._uri_hash
    
    
    def get_connection_result(self):
        # type: () -> ConnectionResult
        return self.connection_result
    
    
    def _ping(self):
        mongo_connection = self.connection_result.get_connection()
        if mongo_connection:
            mongo_connection.admin.command('ping')  # use admin database to get mongo connection state
    
    
    def get_mongo_connection_state(self):
        return self._is_connected, self._connection_state_message, self._last_ping_time
    
    
    def _get_statistics(self):
        # STEP 1 : Read old file and get last results
        databases_statistics, last_modification = self._get_database_statistics_and_last_modification_time()
        
        # If a thread is already managing this uri, but not updating it since 90 seconds, continue and replace this thread
        if databases_statistics and databases_statistics.data['file_manager_uuid'] != self.get_uuid() and (time.time() - last_modification) < 90:
            return
        
        # STEP 2 : Update statistics
        databases_statistics = self._update_databases_statistics(databases_statistics)
        
        # STEP 3 : Read old file and get last results
        if databases_statistics:
            self._write_databases_statistics_file(databases_statistics)
        else:
            raise AutoReconnect('no connection available')
    
    
    def loop_turn(self):
        # type:() -> None
        try:
            if time.time() - self._last_database_statistics_time >= self._get_statistics_interval:
                self._get_statistics()
                self._last_database_statistics_time = time.time()
            self._ping()
            self._is_connected = True
            self._connection_state_message = 'Connect to MongoDB with address : [%s]' % self._uri
        except AutoReconnect as e:
            self._is_connected = False
            self._connection_state_message = 'Cannot connect to MongoDB with error : [%s]' % str(e)
            self.logger.error('Cannot connect to MongoDB with error : [%s]. Request will be attempted again on the next loop.' % str(e))
        except ConnectionFailure as exp:
            self._is_connected = False
            self._connection_state_message = 'Cannot connect to mongodb server: %s   (%s)' % (self._uri, exp)
        except PyMongoError as exp:  # TODO : Remove all this auto reconnection handler. We must use the mongoclient. We developed an entire AutoReconnect system, why not use it?
            self._is_connected = False
            self._connection_state_message = 'Mongo error: %s    (%s)' % (self._uri, exp)
        self._last_ping_time = time.time()
    
    
    def _get_database_statistics_and_last_modification_time(self):
        # type: () -> Optional[DatabaseStatisticsItem, float]
        try:
            database_statistics, last_modification = self._data_hub.get_data_and_last_modification(self.get_uri_hash())
            return DatabaseStatisticsItem(uri_hash=self.get_uri_hash(), data=database_statistics), last_modification
        except DataHubItemNotFound:
            return None, time.time()
    
    
    def _write_databases_statistics_file(self, database_statistics):
        # type:(DatabaseStatisticsItem) -> None
        self._data_hub.save_data(database_statistics.uri_hash, database_statistics.data)
    
    
    @staticmethod
    def _get_mongo_databases_name(mongo_connection):
        # type:(MongoClient) -> List[str]
        return mongo_connection.database_names()
    
    
    def _update_databases_statistics(self, previous_database_statistics):
        # type: (Optional[DatabaseStatisticsItem]) -> DatabaseStatisticsItem|None
        
        mongo_connection = self.connection_result.get_connection()
        
        stats = {}
        
        if not mongo_connection:
            return None
        
        databases_name = self._get_mongo_databases_name(mongo_connection)
        if previous_database_statistics:
            pymongo_has_c = previous_database_statistics.data.get('pymongo_has_c', {})
            bson_has_c = previous_database_statistics.data.get('bson_has_c', {})
            stats = previous_database_statistics.data.get('stats', {})
        else:
            pymongo_has_c, bson_has_c = self._check_pymongo_bson_c_extension_installed()
        
        # We want to get dbstats every minute, to not overload the database
        statistics_update_time = time.time()
        self.logger.debug('Updating databases statistics [dbstats] for : [%s]' % ', '.join(databases_name))
        for database in databases_name:
            stats[database] = self._adapt_cluster_mongo_data_for_json_format(mongo_connection[database].command('dbstats'))
        self.logger.info('Databases statistics updated in %s' % self.logger.format_chrono(statistics_update_time))
        
        return DatabaseStatisticsItem(uri_hash=self.get_uri_hash(), data={
            'file_manager_uuid': self.get_uuid(),
            'databases_name'   : databases_name,
            'address'          : self.get_uri(),
            'pymongo_has_c'    : pymongo_has_c,
            'bson_has_c'       : bson_has_c,
            'stats'            : stats
        })
    
    
    @staticmethod
    def _check_pymongo_bson_c_extension_installed():
        # type: () -> (Dict[str, Union[bool, str]], Dict[str, Union[bool, str]])
        logger.debug('check_pymongo_c_extension_installed::start')
        pymongo_has_c = {'installed': True, 'status': 'OK', 'message': 'Your pymongo has C extension installed'}
        bson_has_c = {'installed': True, 'status': 'OK', 'message': 'Your bson lib has C extension installed'}
        if not pymongo.has_c():
            pymongo_has_c = {'installed': False, 'status': 'ERROR', 'message': 'Your pymongo lib has not the C extension installed'}
        if not bson.has_c():
            bson_has_c = {'installed': False, 'status': 'ERROR', 'message': 'Your bson lib has not the C extension installed'}
        
        return pymongo_has_c, bson_has_c
    
    
    @staticmethod
    def _adapt_cluster_mongo_data_for_json_format(stats):
        # type: (Dict) -> Dict[str, Any]
        election_id = stats.get('$gleStats', {}).get('electionId', None)
        last_op_time = stats.get('$gleStats', {}).get('lastOpTime', None)
        if election_id:
            stats['$gleStats']['electionId'] = str(election_id)
        if last_op_time:
            stats['$gleStats']['lastOpTime'] = last_op_time.time
        
        for cluster in stats.get('raw', {}):
            election_id = stats['raw'][cluster].get('$gleStats', {}).get('electionId', None)
            last_op_time = stats['raw'][cluster].get('$gleStats', {}).get('lastOpTime', None)
            if election_id:
                stats['raw'][cluster]['$gleStats']['electionId'] = str(election_id)
            if last_op_time:
                if isinstance(last_op_time, dict):  # Mongo 3.4: it's a dict like {'ts': Timestamp(1664531988, 1), 't': 7}
                    last_op_time_v = last_op_time['ts'].time
                else:  # Mongo 3.0, it's a standard int
                    last_op_time_v = last_op_time.time
                stats['raw'][cluster]['$gleStats']['lastOpTime'] = last_op_time_v
        
        return stats
