#!/usr/bin/python-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import os
import re
import importlib
import traceback

from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from .versions.patch_decorator import is_weather_format_patch

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Any, List, Callable, Dict, Optional
    
    WeatherVersionType = int
    WeatherDictType = Dict[str, Any]
    MigrationHookFunctionType = Callable[[WeatherDictType], None]

default_logger = LoggerFactory.get_logger('WEATHER JSON MIGRATOR')

VERSION_DIRECTORY_NAME_PATTERN = re.compile(r'^(version_([0-9]{4,}))$')
VERSIONS_PKG_NAME = 'versions'
PATCHES_MODULE_NAME = 'patches'


class WeatherJSONMigrator:
    """
    The weather JSON migrator will load all the migration hooks stored in the subpackage 'versions'

    Each migration versions must have the following tree:
    versions/
    ├─ version_XXXX/    # Where 'XXXX' is the target version (ex: 0001)
    │  ├─ __init__.py
    │  ├─ patches.py    # All functions decorated with @weather_format_patches must be stored here
    ├─ __init__.py
    ├─ patch_decorator.py
    """
    
    __migration_dict = {}  # type: Dict[WeatherVersionType, List[MigrationHookFunctionType]]
    
    
    def __new__(cls):
        # Lazy singleton instance
        try:
            instance = getattr(cls, '__instance__')
        except AttributeError:
            instance = object.__new__(cls)
            instance.init_migration_dict()
            setattr(cls, '__instance__', instance)
        return instance
    
    
    def apply_migrations(self, weather_dict, version, logger=None):
        # type: (WeatherDictType, WeatherVersionType, Optional[PartLogger]) -> None
        """
        Apply migrations loaded from plugins with the version :version: to :weather_dict: WITHOUT verifying the actual version
        The function will not do anything if the version does not exist
        """
        if not logger:
            logger = default_logger
        else:
            logger = logger.get_sub_part('WEATHER JSON MIGRATOR')
        logger.info('Applying migration patches for version [ %d ]' % version)
        for patch in self.__migration_dict.get(version, []):
            patch_name = getattr(patch, '__name__', repr(patch))
            try:
                patch(weather_dict)
            except:
                logger.error('--> Error when applying patch [ %s ]' % patch_name)
                logger.error('--> Backtrace: %s' % traceback.format_exc())
                raise
            logger.info('--> Successfully applied patch [ %s ]' % patch_name)
        
        weather_identity = weather_dict.setdefault('identification', {})
        if not isinstance(weather_identity, dict):
            weather_dict['identification'] = weather_identity = dict()
        weather_identity['weather_format_version'] = version
    
    
    @classmethod
    def init_migration_dict(cls):
        # type: () -> None
        
        cls.__migration_dict.clear()
        
        # Represents folders in ./versions/*
        versions_directory = os.path.join(os.path.dirname(__file__), VERSIONS_PKG_NAME)
        
        logger = default_logger
        logger.debug('Version directory: [ %s ]' % versions_directory)
        
        for filename in os.listdir(versions_directory):
            match = VERSION_DIRECTORY_NAME_PATTERN.match(filename)
            if not match:
                continue
            module_name = match.group(1)
            patch_version = int(match.group(2))
            logger.debug('Loading patches for weather format version [ %d ]' % patch_version)
            logger.debug('--> Loading patches from module [ %s ]' % module_name)
            module_name = f'.{VERSIONS_PKG_NAME}.{module_name}.{PATCHES_MODULE_NAME}'
            try:
                patches_module = importlib.import_module(module_name, package=__package__)
            except ImportError:
                logger.error('Failed to import [ %s ] from [ %s ]' % (module_name, __package__))
                continue
            for func_ptr in filter(is_weather_format_patch, vars(patches_module).values()):
                migration_list = cls.__migration_dict.setdefault(patch_version, [])
                migration_list.append(func_ptr)
                logger.debug('----> Patch [ %s ] imported' % getattr(func_ptr, '__name__', repr(func_ptr)))
