#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#     Martin Benjamin, b.martin@shinken-solutions.com
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import threading

from shinken.inter_daemon_message import InterDaemonMessage
from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.thread_helper import Thread

if TYPE_CHECKING:
    from shinken.modulesmanager import ModulesManager
    from shinken.misc.type_hint import List, Callable


class InterDaemonMessagesManagerForDaemon(Thread):
    def __init__(self, daemon_name: 'str', daemon_type: 'str', handle_messages_received_from_arbiter: 'Callable[[InterDaemonMessage], None]') -> None:
        super(InterDaemonMessagesManagerForDaemon, self).__init__()
        
        self.modules_manager: 'ModulesManager|None' = None
        self.daemon_name = daemon_name
        self.daemon_type = daemon_type
        self.handle_messages_received_from_arbiter = handle_messages_received_from_arbiter
        
        self.push_messages_by_arbiter = []  # type: List[InterDaemonMessage]
        self.push_messages_by_arbiter_lock = threading.RLock()
        
        self.messages_to_send_to_arbiter = []  # type: List[InterDaemonMessage]
        self.messages_to_send_to_arbiter_lock = threading.RLock()
        # message to send at arbiter made by the daemon (the messages will be added to messages_to_send_to_arbiter in manage_inter_daemon_messages)
        self.messages_to_send_to_arbiter_internal = []  # type: List[InterDaemonMessage]
        self.messages_to_send_to_arbiter_internal_lock = threading.RLock()
    
    
    def update_on_new_configuration(self, daemon_name: str, modules_manager: 'ModulesManager|None' = None):
        self.daemon_name = daemon_name
        if modules_manager:
            self.modules_manager = modules_manager
    
    
    def send_message_to_arbiter(self, inter_daemon_message):
        # type: (InterDaemonMessage) -> None
        logger_message = LoggerFactory.get_logger('MANAGE INTER DAEMON MESSAGE')
        logger_message.debug('%s queue message to send at arbiter [%s]' % (self.daemon_type, inter_daemon_message.get_message_info()))
        with self.messages_to_send_to_arbiter_internal_lock:
            self.messages_to_send_to_arbiter_internal.append(inter_daemon_message)
    
    
    def get_message_to_send_at_arbiter(self) -> 'list[InterDaemonMessage]':
        with self.messages_to_send_to_arbiter_lock:
            ret = self.messages_to_send_to_arbiter
            self.messages_to_send_to_arbiter = []
        
        return ret
    
    
    def collect_new_messages_from_arbiter(self, messages: 'list[InterDaemonMessage]') -> None:
        with self.push_messages_by_arbiter_lock:
            logger_message = LoggerFactory.get_logger('MANAGE INTER DAEMON MESSAGE')
            self.push_messages_by_arbiter.extend(messages)
            if logger_message.is_debug():
                logger_message.debug('new [%s] messages received from arbiter' % len(messages))
                for m in messages:
                    logger_message.debug('message received : [%s]' % m.get_message_info())
    
    
    def loop_turn(self) -> None:
        self._manage_inter_daemon_messages()
    
    
    def _manage_inter_daemon_messages(self):
        # Step get all message to send at arbiter
        self._collect_messages_to_send_at_arbiter()
        
        # Step process message send by arbiter
        self._process_message_from_arbiter()
    
    
    def _collect_messages_to_send_at_arbiter(self):
        logger_message = LoggerFactory.get_logger('MANAGE INTER DAEMON MESSAGE')
        
        messages_to_send_to_arbiter = []
        if self.modules_manager:
            for module in self.modules_manager.get_all_alive_instances():
                for message_to_send in module.get_messages_to_send_at_arbiter():
                    # If a module want to send a message to a module from the same demon we shortcut the arbiter.
                    if message_to_send.message_to.daemon_name == self.daemon_name:
                        with self.push_messages_by_arbiter_lock:
                            self.push_messages_by_arbiter.append(message_to_send)
                    else:
                        messages_to_send_to_arbiter.append(message_to_send)
        
        with self.messages_to_send_to_arbiter_internal_lock:
            tmp = self.messages_to_send_to_arbiter_internal
            self.messages_to_send_to_arbiter_internal = []
            messages_to_send_to_arbiter.extend(tmp)
        
        if messages_to_send_to_arbiter:
            logger_message.info('Message to send at arbiter:[%s]' % len(messages_to_send_to_arbiter))
        
        if logger_message.is_debug():
            for message_to_send_to_arbiter in messages_to_send_to_arbiter:
                logger_message.debug('Message to send at arbiter:[%s]' % message_to_send_to_arbiter.get_message_info())
        
        with self.messages_to_send_to_arbiter_lock:
            self.messages_to_send_to_arbiter.extend(messages_to_send_to_arbiter)
    
    
    def _process_message_from_arbiter(self):
        logger_message = LoggerFactory.get_logger('MANAGE INTER DAEMON MESSAGE')
        
        with self.push_messages_by_arbiter_lock:
            if not self.push_messages_by_arbiter:
                return
            push_message_by_arbiter = self.push_messages_by_arbiter
            self.push_messages_by_arbiter = []
        
        nb_message_todo = len(push_message_by_arbiter)
        process_messages = []
        for inter_daemon_message in push_message_by_arbiter:
            if inter_daemon_message.message_to.daemon_name != self.daemon_name:
                continue
            if inter_daemon_message.message_to.module_name == '':
                # Message is for our daemon, not its modules
                self.handle_messages_received_from_arbiter(inter_daemon_message)
                process_messages.append(inter_daemon_message)
                continue
            
            # OK : it's a message for one type of module, we give it to all the modules concerned
            if self.modules_manager:
                for module in self.modules_manager.get_all_alive_instances():
                    if inter_daemon_message.message_to.module_name == module.get_name():
                        try:
                            module.handle_messages_received_from_arbiter(inter_daemon_message)
                            process_messages.append(inter_daemon_message)
                        except Exception as e:
                            logger_message.error(str(e))
                            logger_message.print_stack()
        
        for process_message in process_messages:
            logger_message.debug('Processed message from arbiter:[%s]' % process_message.get_message_info())
            push_message_by_arbiter.remove(process_message)
        logger_message.info('Message from arbiter:[%s] processed:[%s] remaining:[%s]' % (nb_message_todo, len(process_messages), len(push_message_by_arbiter)))
        if logger_message.is_debug():
            for remaining_message in push_message_by_arbiter:
                logger_message.debug('Remaining message from arbiter:[%s]' % remaining_message.get_message_info())
        
        with self.push_messages_by_arbiter_lock:
            self.push_messages_by_arbiter.extend(push_message_by_arbiter)
    
    
    def get_thread_name(self) -> str:
        return 'int-dnm-mngr'
