#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import heapq
import os
import shutil

from shinken.compat import bytes_to_unicode
from shinken.log import logger
from shinken.misc.os_utils import make_file_hidden, write_file_set_owner
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.locking.shinken_locking.shinken_interprocess_rlock import ShinkenInterProcessRLock
from shinkensolutions.os_helper import make_dirs_and_chown_shinken

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Optional, Union

ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH = '/var/lib/shinken/arbiter/arbiter_configuration_messages'
COMMON_FILE_NAME = 'shinken_common.json'
arbiter_configuration_messages_uuid = ''
arbiter_configuration_messages_part = ''

lock_arbiter_messages = None
arbiter_messages = []
MESSAGE_BUFFER_SIZE = 10000
SORT_BUFFER_NB_LINE = 100


def init_message_arbiter():
    # type: () -> None
    global lock_arbiter_messages
    lock_arbiter_messages = ShinkenInterProcessRLock(make_file_hidden(os.path.join(os.path.dirname(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH), 'arbiter_configuration_messages.lock')))


def set_arbiter_configuration_messages_uuid(uuid):
    # type: (str) -> None
    global arbiter_configuration_messages_uuid
    arbiter_configuration_messages_uuid = uuid


def read_arbiter_configuration_messages(uuid):
    # type: (str) -> str
    try:
        message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, uuid)
        
        common_file_name = os.path.join(message_folder, 'shinken_final_messages.json')
        with open(common_file_name, 'r') as _file:
            return bytes_to_unicode(_file.read())
    except:
        err_msg = 'Fail to read arbiter messages file at %s' % os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, uuid)
        logger.error(err_msg)
        logger.print_stack()
        raise Exception(err_msg)


def get_arbiter_messages_from_json(_arbiter_messages, message_level=None):
    # type: (List[List[Union[int, str]]], Optional[List[str]]) -> List[str]
    _messages_to_return = []
    if not message_level:
        message_level = ['ERROR']
    
    for _message in _arbiter_messages:
        if _message[2] in message_level:
            _messages_to_return.append(_message[3])
    return _messages_to_return


def add_arbiter_messages(message):
    # type: (str) -> None
    
    if not arbiter_configuration_messages_uuid:
        return
    
    with lock_arbiter_messages:
        arbiter_messages.append(message)
        
        if len(arbiter_messages) > MESSAGE_BUFFER_SIZE:
            flush_arbiter_message_buffer()


def flush_arbiter_message_buffer():
    # type: () -> None
    global arbiter_messages
    
    if not arbiter_configuration_messages_uuid:
        return
    
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    common_file_name = os.path.join(message_folder, arbiter_configuration_messages_part)
    
    with open(common_file_name, 'a') as _file:
        # If we do not have a message, we write an empty file.
        # We keep creating an empty file because when we make the final message, we will need all the files.
        if arbiter_messages:
            if _file.tell() > 0:
                _file.write(',\n')
            _file.write(',\n'.join(arbiter_messages))
    arbiter_messages = []


def start_write_arbiter_messages(part):
    # type: (str) -> None
    global arbiter_configuration_messages_part, arbiter_messages
    
    if not arbiter_configuration_messages_uuid:
        return
    
    with lock_arbiter_messages:
        arbiter_configuration_messages_part = part
        arbiter_messages = []
        
        message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
        common_file_name = os.path.join(message_folder, arbiter_configuration_messages_part)
        
        make_dirs_and_chown_shinken(message_folder)
        write_file_set_owner(common_file_name, b'', set_owner=True)


def end_write_arbiter_messages():
    # type: () -> None
    if not arbiter_configuration_messages_uuid:
        return
    with lock_arbiter_messages:
        flush_arbiter_message_buffer()


def clean_arbiter_messages():
    # type: () -> None
    shutil.rmtree(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, ignore_errors=True)


def write_final_json():
    # type: () -> None
    if not arbiter_configuration_messages_uuid:
        return
    end_write_arbiter_messages()
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    common_file_name = os.path.join(message_folder, 'shinken_final_messages.json')
    write_file_set_owner(common_file_name, b'[', set_owner=True)
    sort_by_time()
    delete_files_except_final()


def sort_by_time():
    # type: () -> None
    
    if not arbiter_configuration_messages_uuid:
        return
    
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    result_file_name = os.path.join(message_folder, 'shinken_final_messages.json')
    
    all_files = []
    for file_name in os.listdir(message_folder):
        if file_name != 'shinken_final_messages.json':
            all_files.append(open(os.path.join(message_folder, file_name), 'r'))
    
    pending_lines = []
    with open(result_file_name, 'a') as result_file:
        # heapq.merge will iterate over lines of each file and sort them by timestamp
        # Line exemple :
        # [824100805.0, "ELEMENT", "WARNING", "FIFTH", "CLUSTER", "978457454878"],
        # to obtain the timestamp as sort key, we keep the first element of line when splitting with , and we skip the first character [
        for index, line in enumerate(heapq.merge(*all_files, key=lambda input_line: float(input_line.removeprefix('[').split(',', 1)[0])), 1):
            nb_pending_lines = len(pending_lines)
            if nb_pending_lines > 0 and pending_lines[-1][-1] == ']':
                pending_lines[-1] = '%s,\n' % pending_lines[-1]
            if nb_pending_lines >= SORT_BUFFER_NB_LINE > 1:
                result_file.writelines(pending_lines[:-1])
                pending_lines = pending_lines[-1:]
            pending_lines.append('[%s,%s' % (index, line.split(',', 1)[1]))
        
        if len(pending_lines) >= 1:
            result_file.writelines(pending_lines)
        result_file.write(']')
    
    for f in all_files:
        f.close()


def delete_files_except_final():
    # type: () -> None
    message_folder = os.path.join(ARBITER_CONFIGURATION_MESSAGES_FOLDER_PATH, arbiter_configuration_messages_uuid)
    for file_name in os.listdir(message_folder):
        if file_name != 'shinken_final_messages.json':
            common_file_name = os.path.join(message_folder, file_name)
            os.remove(common_file_name)
