#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from .constant import SHINKEN_GRID_ELEMENTS_CONSTS
from .shinken_grid_element_template import ShinkenGridElementTemplate
from .square import Square, Point
from ..object.shinken_ui_composite.shinken_ui_property import ShinkenUIIntegerProperty

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Type
    from ..object.messages import ValidatorMessages
    from .shinken_grid import ShinkenGrid


class ShinkenGridElement(ShinkenGridElementTemplate):
    
    def __init__(self, associated_key, index, validator_message_type=None, mandatory=False):
        # type:(str, int, Type[ValidatorMessages], bool) -> None
        super(ShinkenGridElement, self).__init__(associated_key, index, validator_message_type=validator_message_type, mandatory=mandatory)
        self.position_x = ShinkenUIIntegerProperty(SHINKEN_GRID_ELEMENTS_CONSTS.KEY_POS_X, validator_message_type=validator_message_type, mandatory=True, must_be_positive=True)
        self.position_y = ShinkenUIIntegerProperty(SHINKEN_GRID_ELEMENTS_CONSTS.KEY_POS_Y, validator_message_type=validator_message_type, mandatory=True, must_be_positive=True)
        self.is_active = True
        
        # This will be useful to determine if there is overlap
        self.square = None  # type: Optional[Square]
    
    
    def compute_square(self, grid):
        # type: (ShinkenGrid) -> None
        try:
            if (
                    self.width.is_incorrect_or_missing()
                    or self.height.is_incorrect_or_missing()
                    or self.position_x.is_source_incorrect_or_missing()
                    or self.position_y.is_source_incorrect_or_missing()
            ):
                self.is_active = False
            first_point_x = self.position_x.get_computed_value()
            first_point_y = self.position_y.get_computed_value() + self.height.get_computed_value()
            second_point_x = max(grid.nb_tiles_in_width.get_computed_value(), self.position_x.get_computed_value()) if self.width.get_computed_value() == 'all' else self.position_x.get_computed_value() + self.width.get_computed_value()
            second_point_y = self.position_y.get_computed_value()
            self.square = Square(Point(first_point_x, first_point_y), Point(second_point_x, second_point_y))
        except (TypeError, ValueError):
            self.square = Square(Point(None, None), Point(None, None))
