#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.toolbox.box_tools_array import ToolsBoxArray
from .shinken_object import ShinkenObject

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Any
    from synchronizer.component.generic_component.translate_component import TranslatePart


class ShinkenArray(ShinkenObject):
    def __init__(self, translate=None):
        # type: (TranslatePart) -> None
        super(ShinkenArray, self).__init__(translate)
        self.content = []
    
    
    def reset(self):
        # type: () -> None
        self.content = []
    
    
    def get_element(self, index):
        # type: (int) -> Any
        return self.content[index]
    
    
    def add_element(self, to_add):
        # type: (ShinkenObject) -> None
        self.content.append(to_add)
    
    
    def insert_element(self, index, to_add):
        # type: (int, ShinkenObject) -> None
        self.content.insert(index, to_add)
    
    
    def exist(self, to_test):
        # type: (ShinkenObject) -> int
        for index, _current in enumerate(self.content):
            if _current.is_equal(to_test):
                return index
        return -1
    
    
    def exist_by_id(self, _id):
        # type: (unicode) -> int
        for index, _current in enumerate(self.content):
            if _current.is_id(_id):
                return index
        return -1
    
    
    def get_size(self):
        # type: () -> int
        return len(self.content)
    
    
    def get_content(self):
        # type: () -> List
        return self.content
    
    
    def get_html(self):
        # type: () -> unicode
        _to_return = []
        for _current in self.content:
            _to_return.append(_current.get_html())
        return ''.join(_to_return)


class ShinkenArrayWithMapping(ShinkenArray):
    def __init__(self, translate=None):
        # type: (TranslatePart) -> None
        super(ShinkenArrayWithMapping, self).__init__(translate)
        self.mapping = {}
    
    
    def reset(self):
        # type: () -> None
        super(ShinkenArrayWithMapping, self).reset()
        self.mapping = {}
    
    
    def add_element(self, to_add):
        # type: (ShinkenObject) -> None
        self.mapping[to_add.get_uuid()] = ToolsBoxArray.append_and_return_insert_index(self.content, to_add)
    
    
    def insert_element(self, index, to_add):
        # type: (int, ShinkenObject) -> None
        self.content.insert(index, to_add)
        self.compute_mapping()


    def get_element_by_uuid(self, uuid):
        # type: (int) -> Any
        if uuid in self.mapping:
            return self.get_element(self.mapping[uuid])
        return None
    
    
    def compute_mapping(self):
        # type: () -> None
        self.mapping = {}
        for index, _current in enumerate(self.content):
            self.mapping[index] = _current.get_uuid()
    
    
    def exist(self, to_test):
        # type: (ShinkenObject) -> int
        if to_test.get_uuid() in self.mapping:
            return self.mapping[to_test.get_uuid()]
        return -1
