#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.exceptions.business import ShinkenExceptionKeyError, ShinkenExceptionValueError
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.filters.list.items.abstract_list_filter_item import AbstractListFilterItem
from shinkensolutions.filters.paginator import Paginator

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Dict, List, Optional


class PAGINATOR_FILTER_CONSTS:
    PAGE = u'page'  # type: str
    NB_ELEMENTS = u'nb_element'  # type: str


class ListFilterItemPaginator(AbstractListFilterItem):
    def __init__(self, page_index, page_size):
        # type: (int, int) -> None
        if page_index < 0:
            raise ShinkenExceptionValueError(text=u'Wrong value:[ %s ] for page number parameter' % page_index)
        self.paginator = Paginator(page_size)
        self.index = page_index
    
    
    def apply_on(self, list_to_filter):
        # type: (List[Any]) -> List[Any]
        return self.paginator.get_page(list_to_filter, self.index)
    
    
    @classmethod
    def get_valid_keys(cls):
        # type: () -> List[str]
        return [PAGINATOR_FILTER_CONSTS.PAGE, PAGINATOR_FILTER_CONSTS.NB_ELEMENTS]
    
    
    @classmethod
    def from_dict(cls, filter_dict):
        # type: (Dict[str, str]) -> ListFilterItemPaginator
        page_index = cls._get_int_from_dict(filter_dict, PAGINATOR_FILTER_CONSTS.PAGE)
        page_size = cls._get_int_from_dict(filter_dict, PAGINATOR_FILTER_CONSTS.NB_ELEMENTS)
        return cls(page_index=page_index, page_size=page_size)
    
    
    def get_additional_information_from_non_filtered_list(self, object_list, info_dict=None):
        # type: (List[Any], Optional[Dict[str, Any]]) -> Dict[str, Any]
        info_dict = super(ListFilterItemPaginator, self).get_additional_information_from_non_filtered_list(object_list, info_dict=info_dict)
        page_index = self.index
        nb_pages = self.paginator.get_available_page_number(object_list)
        remaining_pages = max(nb_pages - 1 - page_index, 0)
        info_dict.update({
            u'pagination': {
                u'page_index'     : page_index,
                u'nb_pages'       : nb_pages,
                u'remaining_pages': remaining_pages,
            }
        })
        return info_dict
    
    
    __marker = object()
    
    
    @classmethod
    def _get_int_from_dict(cls, filter_dict, key, default=__marker):
        # type: (Dict[str, str], str, Any) -> int
        value = filter_dict.get(key, cls.__marker)
        if value is cls.__marker:
            if default is cls.__marker:
                raise ShinkenExceptionKeyError(text=u'Missing [ %s ] key' % key)
            return default
        try:
            value = int(value)
        except (TypeError, ValueError):
            raise ShinkenExceptionValueError(text=u'Wrong value:[ %s ] for [ %s ] key' % (value, key))
        return value
