#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import datetime
import http.client
import json
import optparse
import random
import socket
import ssl
import sys
import time
from optparse import Option, _builtin_cvt, check_choice, OptionParser

from shinken.log import DEFAULT_LONG_FLUSH_STATS
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import ModuleState
from shinken.runtime_stats.threads_dumper import watchdog_fatal_status, WATCH_DOG_STATUS_CODE
from shinken.safepickle import SERIALIZATION_SECURITY_STATUS_CODE, serialization_security_container
from shinkensolutions.shinken_time_helper import print_human_readable_period, DisplayFormat as TimeDisplayFormat
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Any, Tuple, Optional, List, Sequence

API_VERSION = '2.12'

LONG_OUTPUT_BREAK = '\n'
NEW_LINE = '<br/><br/>'
BREAK_LINE = '<br/>'
NEW_LINE_OUTPUT_AFTER_TABLE = '<br/>'

EXECUTOR_LOAD_LIMIT = 0.95  # %
to_print = ''

random.seed()


def check_builtin(option, opt, value):
    (cvt, what) = _builtin_cvt[option.type]
    try:
        if value == '':
            return option.default
        return cvt(value)
    except ValueError:
        raise ParseOptionError(
            'option %s: invalid %s value: %r' % (opt, what, value))


class ShinkenOption(Option):
    TYPE_CHECKER = {
        'int'    : check_builtin,
        'long'   : check_builtin,
        'float'  : check_builtin,
        'complex': check_builtin,
        'choice' : check_choice,
    }


# Allow \n in the parser output
class ShinkenParser(optparse.OptionParser):
    def __init__(self,
                 usage=None,
                 option_list=None,
                 option_class=ShinkenOption,
                 version=None,
                 conflict_handler='error',
                 description=None,
                 formatter=None,
                 add_help_option=True,
                 prog=None,
                 epilog=None
                 ):
        OptionParser.__init__(self, usage, option_list, option_class, version, conflict_handler, description, formatter, add_help_option, prog, epilog)


class ParseOptionError(SyntaxError):
    pass


class NotFoundException(Exception):
    pass


class RaiseOnExitOptionParser(ShinkenParser):
    def exit(self, status=0, msg=None):
        raise ParseOptionError(msg)


class COLOR:
    BLACK = '#000000'
    GREEN = '#2A9A3D'
    RED = '#FF0000'
    ORANGE = '#f57700'
    GRAY = '#808080'
    BLUE = '#4242DB'
    
    DEFAULT_COLOR = BLACK


class HYPERVISOR:
    HYPERV = 'hyper-v'
    VMWARE = 'vmware'
    KVM = 'kvm'


# NOTE: don't know how to make it a class/union without breaking the HYPERVISOR class
HYPERVISOR_DISPLAY_NAME = {
    HYPERVISOR.HYPERV: 'HyperV',
    HYPERVISOR.VMWARE: 'VMWare',
    HYPERVISOR.KVM   : 'Kvm'
}


class CPU_STOLEN_NAME:
    VPTR = 'vptr'
    LPTR = 'lptr'
    CPU_READY = 'vmware_ready_stat'
    CPU_STEAL = 'cpu_steal'


class EXIT_STATUS:
    OK = 0
    WARNING = 1
    CRITICAL = 2
    UNKNOWN = 3


STATUS_TO_TEXT = {
    EXIT_STATUS.OK      : '[OK]',
    EXIT_STATUS.WARNING : '[WARNING]',
    EXIT_STATUS.CRITICAL: '[CRITICAL]',
    EXIT_STATUS.UNKNOWN : '[UNKNOWN]',
}


class Result:
    def __init__(self):
        self.status = EXIT_STATUS.OK
        self.warnings = []
        self.criticals = []
        self.outputs = []
        self.outputs_no_sort = []
        self.titles = []
        self.long_outputs = []
        self.perf_data = {}
        self.spare_info = ''
    
    
    def set_perf_data(self, perf_data):
        self.perf_data.update(perf_data)
    
    
    def add_perf_data(self, name, value):
        self.perf_data[name] = value
    
    
    def set_logger_stats(self, data):
        if not data:
            return
        logger_stats = data.get('logger_stats', DEFAULT_LONG_FLUSH_STATS)
        is_too_long = logger_stats.get('is_too_long', False)
        write_duration = logger_stats.get('write_duration', 0.0)
        log_path = logger_stats.get('log_path', '')
        # If the logger is going well, bail out
        if not is_too_long:
            return
        
        self.add_check(EXIT_STATUS.WARNING, '%s - Writing logs on disk took too much time ( worth time was %.1fs during the last minute)<br>Path: "%s"' % (HTMLTag.color_text('WARNING', COLOR.ORANGE), write_duration, log_path))
    
    
    def set_spare_info(self, data, daemon_type=None):
        if not data:
            return
        daemon_type = daemon_type or data.get('daemon_type', 'Daemon')
        spare_message = '<div style="' \
                        'margin: 5px 5px 5px 1px;' \
                        'background: #0095da;' \
                        'color: #fff;' \
                        'padding: 1px 7px;' \
                        'border-radius: 5px;' \
                        'box-sizing: border-box;' \
                        'display: inline-block;' \
                        'display: inline-block;' \
                        '">SPARE%s</div>'
        active_message = '<div style="' \
                         'margin: 5px 5px 5px 1px;' \
                         'background: #2a9a3d;' \
                         'color: #fff;' \
                         'padding: 1px 7px;' \
                         'border-radius: 5px;' \
                         'box-sizing: border-box;' \
                         'display: inline-block;' \
                         'display: inline-block;' \
                         '">RUNNING</div><br>'
        
        master_daemon = data.get('master_daemon')
        
        is_spare = data.get('spare', False)
        if is_spare:  # NOTE: the master case is less important, and only put results entry
            if master_daemon is None:  # old format daemon
                idle_message = "This {0} is currently idle because it is configured as a Spare and its main daemon is running well.<br>It will take over another {0} when a main {0} stops working".format(
                    daemon_type.title())
            else:  # new Broker
                if master_daemon != '':
                    idle_message = "I am the SPARE of the master daemon → {0}".format(HTMLTag.tag_value(master_daemon, COLOR.BLUE))
                else:  # useless
                    useless_message = "No master is using this spare daemon  → {0}".format(HTMLTag.tag_value('UNUSED', COLOR.RED))
                    self.spare_info = spare_message % ''
                    self.hard_exit(EXIT_STATUS.CRITICAL, useless_message)
                    return
            # Ok now running or not?
            if data.get('activated', False):  # means running
                # If we have the name of the master daemon, put it
                if master_daemon:
                    _spare_message = spare_message % ' of [ %s ]' % master_daemon  # I don't know why, but the utf8 -> is not working there
                else:
                    _spare_message = spare_message % ''
                self.spare_info = ''.join((_spare_message, active_message))
            
            else:  # idle
                self.spare_info = spare_message % ''
                self.hard_exit(EXIT_STATUS.OK, idle_message)
    
    
    def add_check(self, status=EXIT_STATUS.OK, output='', long_output='', title=False, no_new_line=False):
        # type: (int, Optional[str], Optional[str], bool, bool) -> None
        if self.status < status:
            self.status = status
        
        if output:
            if not no_new_line:
                output = ''.join((output, BREAK_LINE))
            
            if title:
                self.titles.append(output)
            else:
                if status == EXIT_STATUS.CRITICAL:
                    self.criticals.append(output)
                elif status == EXIT_STATUS.WARNING:
                    self.warnings.append(output)
                else:
                    self.outputs.append(output)
                
                self.outputs_no_sort.append(output)
        if long_output:
            if not no_new_line:
                long_output = ''.join((long_output, BREAK_LINE))
            self.long_outputs.append(long_output)
    
    
    def add_title(self, title):
        self.add_check(output=title, title=True)
    
    
    def hard_exit(self, status, output, long_output=''):
        # type: (int, str, str) -> None
        self.status = status
        self.outputs = [output]
        self.outputs_no_sort = []
        self.titles = []
        self.criticals = []
        self.warnings = []
        self.long_outputs = [long_output]
        self.perf_data = {}
        self.exit()
    
    
    def exit(self, sorted_by_level=True):
        # type: (bool) -> None
        global to_print
        
        if self.titles:
            self.titles.append(BREAK_LINE)
        if sorted_by_level:
            warnings = SummaryBox.static_make_output(self.warnings, COLOR.ORANGE, bullet_prefix="=>")
            criticals = SummaryBox.static_make_output(self.criticals, COLOR.RED, bullet_prefix="=>")
            output = ''.join((''.join(self.titles), criticals, warnings, ''.join(self.outputs)))
        else:
            output = ''.join((''.join(self.titles), ''.join(self.outputs_no_sort)))
        if self.long_outputs:
            output = ''.join((output, LONG_OUTPUT_BREAK, ''.join(self.long_outputs)))
        
        # print "exit status[%s] output[%s] <br>" % (self.status, output)
        
        tag = STATUS_TO_TEXT.get(self.status, '')
        if tag:
            tag = HTMLTag.color_text_by_status(tag, self.status, default_color=COLOR.BLACK)
        
        if self.perf_data is None:
            self.perf_data = {}
        
        perfdata = Result._do_perfdata(self.perf_data)
        output = Utils.add_style(output)
        to_print = '%s%s %s' % (self.spare_info, tag, output.strip())
        if perfdata:
            to_print = '%s| %s' % (to_print, perfdata)
        print(to_print)
        sys.exit(self.status)
    
    
    @staticmethod
    def _do_perfdata(performances):
        return ' '.join(['%s=%s' % (k, v) for (k, v) in performances.items()])


class SummaryBox:
    def __init__(self, color=COLOR.BLACK, bullet_prefix='=>'):
        # type: (str, str) -> None
        self._lines = []  # type: List[str]
        self.color = color  # type: str
        self.bullet_prefix = bullet_prefix  # type: str
    
    
    def has_lines(self):
        return bool(self._lines)
    
    
    def add_line(self, line):
        # type: (str) -> None
        if not line:
            return
        self._lines.append(line)
    
    
    def add_multiple_lines(self, *lines):
        # type: (Tuple[str, ...]) -> None
        self._lines.extend(line for line in lines if line)
    
    
    def make_output(self):
        return self.static_make_output(self._lines, color=self.color, bullet_prefix=self.bullet_prefix)
    
    
    @staticmethod
    def static_make_output(messages, color=COLOR.BLACK, bullet_prefix='=>'):
        # type: (Sequence[str], str, str) -> str
        messages = [msg for msg in messages if msg]  # Filter empty strings
        if len(messages) < 1:
            return ''
        prefix = HTMLTag.color_text(bullet_prefix, color) if bullet_prefix and len(messages) > 1 else ''
        output = ''.join('<div class="skn-ln">%s%s</div>' % (prefix, msg) for msg in messages)
        return HTMLTag.tag_border(output, color)


class HTMLTag:
    used = False
    STYLE = '''.skn-ctg{margin:1px;background:#DDD;padding:1px 7px;border-radius:5px;box-sizing:border-box;display:inline-block;border:1px solid} .skn-brd{margin:7px 0px;padding:5px 7px;border-radius:7px;box-sizing:border-box;border:2px solid;width:100%;word-break: break-word;border-collapse: separate;}'''
    EXTRA_STYLE = ''''''
    EXTRA_CLASS = ''
    
    STATE_CRITICAL = 'CRITICAL'
    STATE_OK = 'OK'
    STATE_WARNING = 'WARNING'
    
    CRITICAL = ''
    OK = ''
    WARNING = ''
    
    STATE_COLOR = {
        'FATAL'   : COLOR.RED,
        'CRITICAL': COLOR.RED,
        'OK'      : COLOR.GREEN,
        'WARNING' : COLOR.ORANGE,
    }
    
    EXIT_STATUS_COLOR = {
        EXIT_STATUS.OK      : COLOR.GREEN,
        EXIT_STATUS.WARNING : COLOR.ORANGE,
        EXIT_STATUS.CRITICAL: COLOR.RED,
        EXIT_STATUS.UNKNOWN : COLOR.GRAY
    }
    
    
    @staticmethod
    def color_text(value, color=COLOR.BLACK, bold=True, italic=False, extra_style=''):
        # type: (str, str, bool, bool, str) -> str
        _span = '<span style="color:%(color)s;%(bold)s%(italic)s%(extra_style)s">%(value)s</span>'
        _info = {'color': color, 'value': value, 'bold': '', 'italic': '', 'extra_style': extra_style}
        if bold:
            _info['bold'] = 'font-weight:bold;'
        if italic:
            _info['italic'] = 'font-style:italic;'
        return _span % _info
    
    
    @staticmethod
    def get_color_by_status(status, default_color=COLOR.BLACK):
        # type: (int, str) -> str
        return HTMLTag.EXIT_STATUS_COLOR.get(status, default_color)
    
    
    @staticmethod
    def color_text_by_status(value, status, default_color=COLOR.BLACK, bold=True, italic=False):
        # type: (str, int, str, bool, bool) -> str
        return HTMLTag.color_text(value, HTMLTag.get_color_by_status(status, default_color), bold=bold, italic=italic)
    
    
    @staticmethod
    def tag_border(value, color=COLOR.BLACK):
        HTMLTag.used = True
        _span = '<table class="skn-brd %(extra_class)s" style="border-color:%(color)s; "><tr><td>%(value)s</tr></td></table>'
        return _span % {'color': color, 'value': value, 'extra_class': HTMLTag.EXTRA_CLASS}
    
    
    @staticmethod
    def tag_value(value, color=COLOR.BLACK):
        HTMLTag.used = True
        _span = '<span class="skn-ctg %(extra_class)s" style="color:%(color)s;border-color:%(color)s;">%(value)s</span>'
        return _span % {'color': color, 'value': value, 'extra_class': HTMLTag.EXTRA_CLASS}
    
    
    @staticmethod
    def state_tag(state):
        return HTMLTag.tag_value(state, HTMLTag.STATE_COLOR.get(state, COLOR.DEFAULT_COLOR))
    
    
    @staticmethod
    def load_tag(load):
        if load == -1:
            return HTMLTag.tag_value('unavailable', COLOR.ORANGE)
        elif load == -2:
            return HTMLTag.tag_value('unreachable', COLOR.RED)
        elif load > EXECUTOR_LOAD_LIMIT:
            return HTMLTag.tag_value('No more CPU usable', COLOR.ORANGE)
        else:
            return HTMLTag.tag_value('Resources available', COLOR.BLACK)
    
    
    @staticmethod
    def ram_tag(ram_usage, max_ram_usage):
        if ram_usage == -1:
            return HTMLTag.tag_value('unavailable', COLOR.ORANGE)
        elif ram_usage == -2:
            return HTMLTag.tag_value('unreachable', COLOR.RED)
        elif ram_usage > max_ram_usage:
            return HTMLTag.tag_value('Limit reached', COLOR.RED)
        else:
            return HTMLTag.tag_value('normal', COLOR.BLACK)
    
    
    @staticmethod
    def cpu_queue_tag(cpu_running_queue, max_cpu_queue_per_cpu, nb_cpus):
        if cpu_running_queue == -1:
            return HTMLTag.tag_value('unavailable', COLOR.ORANGE)
        elif cpu_running_queue == -2:
            return HTMLTag.tag_value('unreachable', COLOR.RED)
        elif cpu_running_queue > (max_cpu_queue_per_cpu * nb_cpus):
            return HTMLTag.tag_value('Limit reached', COLOR.RED)
        else:
            return HTMLTag.tag_value('normal', COLOR.BLACK)


class HTMLComment:
    used = False
    STYLE = '''.skn-com{color:grey;font-size:0.8em;text-indent:2em;display:inline-block;}'''
    EXTRA_STYLE = ''
    
    
    @staticmethod
    def add_tag(value):
        HTMLComment.used = True
        return '<div class="skn-com";>%s</div>' % value


class HTMLList:
    used = False
    STYLE = '.skn-ul{padding: 0; margin: 0 0 10px 25px;} .skn-ul-compact{margin-bottom: 0px;padding-left: 0px;margin-left: 25px;}'
    EXTRA_STYLE = ''
    
    
    @staticmethod
    def _list_item(item):
        return "<li>%s</li>" % item
    
    
    @staticmethod
    def _list_header(header, special_class='', ordered_list=False):
        list_type = 'ul'
        if ordered_list:
            list_type = 'ol'
        if special_class != '':
            HTMLList.used = True
        if header is None:
            return "<%s class='skn-ul %s'>" % (list_type, special_class)
        else:
            return "%s&nbsp;<%s class='skn-ul %s'>" % (header, list_type, special_class)
    
    
    @staticmethod
    def _list_footer(ordered_list=False):
        if not ordered_list:
            return "</ul>"
        else:
            return "</ol>"
    
    
    @staticmethod
    def header_list(header, items, compact=False, ordered_list=False):
        # type: (Optional[str], list, bool, bool) -> str
        special_class = 'skn-ul-compact' if compact else ''
        HTMLList.used = True
        return '%(header)s%(items)s%(footer)s' % {
            'header': HTMLList._list_header(header, special_class=special_class, ordered_list=ordered_list),
            'items' : ''.join([HTMLList._list_item(item) for item in items]),
            'footer': HTMLList._list_footer(ordered_list)
        }
    
    
    @staticmethod
    def simple_list(items):
        return HTMLList.header_list(None, items)
    
    
    # Just give me one bullet list
    @staticmethod
    def one_bullet_list(s):
        # type: (Optional[str]) -> str
        HTMLList.used = True  # Let the CSS be inserted
        return '%s%s%s' % (HTMLList._list_header(None, special_class='skn-ul-compact'), HTMLList._list_item(s), HTMLList._list_footer())


class HTMLTable:
    used = False
    STYLE = '''.skn-ict,.skn-ict td,.skn-ict th{border:1px solid #000000 !important;border-collapse:collapse !important;word-break: break-all !important;color:#000000 !important} .skn-ict{width:100% !important;} .skn-ict th{background-color:#DDDDDD !important;padding:2px !important;word-break:break-word !important} .skn-ict td{padding:2px;width:auto !important;font-weight:normal !important;word-break:break-word !important;background-color:#FFFFFF !important}'''
    EXTRA_STYLE = ''''''
    
    
    @staticmethod
    def add_extra_style(extra_style):
        HTMLTable.EXTRA_STYLE = '%s %s' % (HTMLTable.EXTRA_STYLE, extra_style)
    
    
    @staticmethod
    def generate_class_uuid():
        return 'skn-tbl-%05d' % int(random.random() * 100000)
    
    
    @staticmethod
    def table(headers, lines, title=None, left_headers=None, compact_title=False, extra_tags='', all_col_same_width=True, class_uuid='', extra_style='', title_status='', sub_title='', rowspan_activate=False, rowspans=None):
        # rowspan = [{'line': 0, 'cell': 0, 'rowspan_nb': 0}]
        HTMLTable.used = True
        class_uuid = class_uuid or HTMLTable.generate_class_uuid()
        extra_tags = '' or extra_tags
        _table = []
        
        rowspan_lines = []
        rowspan_cells = []
        rowspan_nbs = []
        nb_rowspan = 0
        if rowspan_activate:
            nb_rowspan = len(rowspans) or None
            for rowspan in rowspans:
                rowspan_lines.append(rowspan.get('line', None))
                rowspan_cells.append(rowspan.get('cell', None))
                rowspan_nbs.append(rowspan.get('rowspan_nb', None))
        
        if extra_style:
            HTMLTable.add_extra_style(''.join(extra_style))
        
        if all_col_same_width:
            _extra_style = [
                '.%s .skn-ict{table-layout: fixed;}' % class_uuid,
            ]
        else:
            _extra_style = [
                '.%s .skn-ict .skn-lfh {width: 25%%;}' % class_uuid,
            ]
        
        HTMLTable.add_extra_style(' '.join(_extra_style))
        
        header_string = ''.join(('<th>%s</th>' % header for header in headers))
        
        values_string = []
        for i, line in enumerate(lines):
            line_string = []
            if not rowspan_activate:
                line_string.append(''.join(['<td>%s</td>' % cell for cell in line]))
            else:
                for j, cell in enumerate(line):
                    if rowspan_activate:
                        if cell == 'ROWSPAN':
                            continue
                        rowspan_ok = False
                        for index in range(nb_rowspan):
                            if rowspan_lines[index] == i and rowspan_cells[index] == j:
                                rowspan_ok = True
                                line_string.append('<td rowspan=\"%s\">%s</td>' % (rowspan_nbs[index], cell))
                                break
                        if not rowspan_ok:
                            line_string.append('<td>%s</td>' % cell)
            if left_headers:
                line_string = '<th class="skn-lfh">%s</th>%s' % (left_headers[i], ''.join(line_string))
            values_string.append('<tr>%s</tr>' % ''.join(line_string))
        values_string = ''.join(values_string)
        
        if title:
            if sub_title:
                _table.append('<div class="skn-ich">%s :%s<br>%s</div>' % (title, ' %s' % TAG_FOR_STATE[title_status] if isinstance(title_status, int) else title_status, sub_title))
            else:
                _table.append('<div class="skn-ich">%s :%s</div>' % (title, ' %s' % TAG_FOR_STATE[title_status] if isinstance(title_status, int) else title_status))
            if not compact_title:
                _table.append(BREAK_LINE)
        
        _table.append('<div class="%s"><table class="skn-ict" %s>' % (class_uuid, extra_tags))
        if header_string:
            _table.append('<tr>%s</tr>' % header_string)
        _table.append('%s' % values_string)
        _table.append('</table></div>')
        
        return ''.join(_table)


class Utils:
    
    @staticmethod
    def _http_get_conn(full_uri, timeout, use_ssl, ssl_version=ssl.PROTOCOL_TLSv1):
        if use_ssl:
            # If we are in SSL mode, do not look at certificate too much
            # NOTE: ssl.SSLContext is only available on last python 2.7 versions
            if hasattr(ssl, 'SSLContext'):
                ssl_context = ssl.SSLContext(ssl_version)
                ssl_context.check_hostname = False
                ssl_context.verify_mode = ssl.CERT_NONE
            else:
                ssl_context = None
            
            args = {}
            if ssl_context:
                args['context'] = ssl_context
            conn = http.client.HTTPSConnection(full_uri, timeout=timeout, **args)
        else:
            conn = http.client.HTTPConnection(full_uri, timeout=timeout)
        return conn
    
    
    # First try in HTTP and if fail from the server, retry in HTTPs
    @staticmethod
    def _request_get(base_uri, uri, use_ssl=False, timeout=3):
        if base_uri.startswith('http://'):
            use_ssl = False
            base_uri = base_uri[7:-1]
        elif base_uri.startswith('https://'):
            use_ssl = True
            base_uri = base_uri[8:-1]
        
        start_time = time.time()

        conn = None
        try:
            # the try part handles successful connections (status 200) and http requests on https server in python 2.7
            # the except section handles http requests on https server in python 3.11
            conn = Utils._http_get_conn(base_uri, timeout=timeout, use_ssl=use_ssl)
            conn.request("GET", uri)
            r1 = conn.getresponse()
            buf = r1.read()
            if r1.status == 400 and not use_ssl and b'sent a plain HTTP request' in buf:
                return Utils._request_get(base_uri, uri, use_ssl=True, timeout=timeout)
            if r1.status == 404:
                conn.close()
                raise NotFoundException()
            if r1.status != 200:
                conn.close()
                raise Exception(buf)
        except socket.error as e:
            if e.errno == 104 and not use_ssl:
                return Utils._request_get(base_uri, uri, use_ssl=True, timeout=timeout)
            else:
                if conn:
                    conn.close()
                raise e
        
        return buf, (time.time() - start_time)
    
    
    # First try in HTTP and if fail from the server, retry in HTTPs
    @staticmethod
    def _request_post(base_uri, uri, body=None, headers=None, use_ssl=False, timeout=3):
        start_time = time.time()
        conn = Utils._http_get_conn(base_uri, timeout=float(timeout), use_ssl=use_ssl)
        conn.request('POST', uri, body=body, headers=headers)
        r1 = conn.getresponse()
        buf = r1.read()
        if r1.status == 400 and not use_ssl and b'sent a plain HTTP request' in buf:
            return Utils._request_post(base_uri, uri, body=body, headers=headers, timeout=timeout)
        if r1.status == 404:
            conn.close()
            raise NotFoundException('404 not found : %s://%s%s' % ('https' if use_ssl else 'http', base_uri, uri))
        if r1.status != 200:
            conn.close()
            raise Exception(buf)
        return buf, (time.time() - start_time)
    
    
    # First try in HTTP and if fail from the server, retry in HTTPs
    @staticmethod
    def request_get(result, base_uri, uri, use_ssl=False, raise_exp=False, timeout=3):
        try:
            return Utils._request_get(base_uri, uri, use_ssl, timeout=timeout)
        except NotFoundException:
            raise
        except Exception as e:
            if raise_exp:
                raise
            else:
                msg = str(e)
                if e.__class__.__name__ == 'timeout':
                    msg = 'The request timed out (%ss)' % timeout
                result.hard_exit(EXIT_STATUS.CRITICAL, 'Cannot connect to %s%s with exception : <br/> %s' % (base_uri, uri, msg))
    
    
    # First try in HTTP and if fail from the server, retry in HTTPs
    @staticmethod
    def request_post(result, base_uri, uri, body=None, headers=None, use_ssl=False, raise_exp=False, timeout=3):
        try:
            return Utils._request_post(base_uri, uri, body=body, headers=headers, timeout=timeout, use_ssl=use_ssl)
        except NotFoundException as e:
            raise NotFoundException(str(e))
        except Exception as e:
            if raise_exp:
                raise
            else:
                msg = str(e)
                if e.__class__.__name__ == 'timeout':
                    msg = 'The request timed out (%ss)' % timeout
                result.hard_exit(EXIT_STATUS.CRITICAL, 'Cannot connect to %s%s with exception : <br/> %s' % (base_uri, uri, msg))
    
    
    @staticmethod
    def print_time(_time):
        if not _time:
            return 'No time'
        return datetime.datetime.fromtimestamp(_time).strftime('%x %X')
    
    
    @staticmethod
    def add_style(output):
        html_items = (HTMLList, HTMLTable, HTMLTag, HTMLComment)
        
        active_item = []
        for html_item in html_items:
            if html_item.used:
                active_item.append(html_item.STYLE)
                active_item.append(html_item.EXTRA_STYLE)
                html_item.used = False
        
        if active_item:
            output = '<style type="text/css">%s</style>%s' % (''.join(active_item), output)
        return output
    
    
    @staticmethod
    def print_human_readable_number(number):
        if sys.version_info < (2, 7):
            return str(int(number))
        else:
            return '{:,}'.format(int(number)).replace(',', ' ')
    
    
    # This is just a relay to print_human_readable_period with a specific format for checks, with integers
    @staticmethod
    def print_period(period):
        return print_human_readable_period(period, time_format='auto', display_format=TimeDisplayFormat('%dms', '%ds', '%dm', '%dh', '%d days'), separator=' ')
    
    
    @staticmethod
    def print_human_readable_size(size):
        if isinstance(size, (str, bytes)):
            try:
                size = int(size)
            except ValueError:
                return size
        
        if not isinstance(size, (float, int)):
            return size
        
        if size < 1024:
            return '%.2f octets' % size
        elif 1024 <= size < (1024 * 1024):
            return '%.2f Ko' % (size / 1024)
        elif (1024 * 1024) <= size < (1024 * 1024 * 1024):
            return '%.2f Mo' % (size / 1024 / 1024)
        else:
            return '%.2f Go' % (size / 1024 / 1024 / 1024)
    
    
    @staticmethod
    def print_human_readable_period(time_period, time_format='auto'):
        return print_human_readable_period(time_period, time_format)
    
    
    @staticmethod
    def print_percent(_value, _total):
        return '%0.2f' % ((_value / _total) * 100.0)
    
    
    @staticmethod
    def escape_XSS(_value):
        return ToolsBoxString.escape_XSS_special_quote(_value)


# Object that will do all the logic/verification about arbiters traces:
# * in the healthcheck
# * in a daemon check
class ArbiterTraceVerificationTree:
    def __init__(self):
        # type: () -> None
        self._in_conflict = False  # if the daemon was contacted by 2 distincts architectures, NOT by 2 arbiters
        self._in_architecture_names_conflict = False  # if contacted by 2 architectures, are the name the same?
        self._state = EXIT_STATUS.OK  # overall of the check result
        self._arbiters_tree = {}  # what to display
    
    
    def __detect_architecture_name_conflicts(self, master_arbiter_uuids):
        # type: (Dict) -> None
        in_architecture_names_conflict = False
        architecture_names_conflicts_uuids = {}
        
        # First: loop to detect in_architecture_names_conflict
        for (master_arbiter_uuid, arbiters) in master_arbiter_uuids.items():
            # For display, we are preparing the inner loop with arbiters, then the top level <li> with
            # an error or not based on the number master_arbiter_uuids
            for arbiter in arbiters:
                arbiter_architecture = arbiter.get('architecture_name', '')
                if arbiter_architecture:
                    if arbiter_architecture not in architecture_names_conflicts_uuids:
                        architecture_names_conflicts_uuids[arbiter_architecture] = set()
                    architecture_names_conflicts_uuids[arbiter_architecture].add(master_arbiter_uuid)
                    # DEBUG: uncomment this line to simulate the in_architecture_names_conflict behavior
                    # architecture_names_conflicts_uuids[arbiter_architecture].append(master_arbiter_uuid)
        
        # Maybe two distinct architecture (identified by the Arbiter MASTER server) have the same name: will be a problem in detection
        for (architecture_name, _uuids) in architecture_names_conflicts_uuids.items():
            if len(_uuids) >= 2:
                in_architecture_names_conflict = True
        
        self._in_architecture_names_conflict = in_architecture_names_conflict
    
    
    def __check_arbiter_connection__get_uuids_and_conflicts(self, traces):
        # type: (List) -> (Dict, bool, bool)
        # Now what we want to be checked is arbiter conflicts:
        # * arbiter MASTER & (optional) SPARE with the same master_arbiter_uuids is NOT a problem (normal behavior of switching)
        # * but if there is more than 1 master_arbiter_uuids then this is a CRITICAL error, strange things will happen
        master_arbiter_uuids = {}
        for _arbiter in traces:
            master_arbiter_uuid = _arbiter['master_arbiter_uuid'].strip()  # old versions have \n in it
            if master_arbiter_uuid not in master_arbiter_uuids:
                master_arbiter_uuids[master_arbiter_uuid] = []
            master_arbiter_uuids[master_arbiter_uuid].append(_arbiter)
        
        # architecture_names_conflicts_uuids = {}  # if two architectures are with the same name, error
        self.__detect_architecture_name_conflicts(master_arbiter_uuids)
        
        self._in_conflict = len(master_arbiter_uuids) > 1  # if there is more than one architecture, CONFLICT
        
        return master_arbiter_uuids
    
    
    def analyse_arbiter_traces(self, traces):
        # type: (List) -> None
        worse_state = EXIT_STATUS.OK  # by default, all is well
        
        # Now what we want to be checked is arbiter conflicts:
        # * arbiter MASTER & (optional) SPARE with the same master_arbiter_uuids is NOT a problem (normal behavior of switching)
        # * but if there is more than 1 master_arbiter_uuids then this is a CRITICAL error, strange things will happen
        master_arbiter_uuids = self.__check_arbiter_connection__get_uuids_and_conflicts(traces)
        
        if self._in_conflict:
            worse_state = EXIT_STATUS.CRITICAL
        
        self._arbiters_tree = {}
        
        for (master_arbiter_uuid, arbiters) in master_arbiter_uuids.items():
            
            tree_entry = {'architecture_name': '', 'arbiters': [], 'ip': ''}
            self._arbiters_tree[master_arbiter_uuid] = tree_entry
            
            architecture_name = ''
            # For display, we are preparing the inner loop with arbiters, then the top level <li> with
            # an error or not based on the number master_arbiter_uuids
            for arbiter in arbiters:
                name = arbiter['name']
                ip = arbiter.get('ip', '')
                if ip:
                    tree_entry['ip'] = ip
                
                arbiter_architecture = arbiter.get('architecture_name', '')
                if arbiter_architecture:
                    architecture_name = arbiter_architecture
                
                insert_time = arbiter['insert_time']
                diff_time_with_arbiter = abs(arbiter['diff_time_with_arbiter'])  # abs: never believe a diff time
                expire_period = arbiter['expire_period']
                expire_time = insert_time + expire_period
                # NOTE: take time from the server, so we NEVER use the check local time (that can be wrong), and only the
                #       distant daemon state, even if WE are out of time
                # this entry was missing before v02.07.06-Patched-08 (may 2021)
                server_now = arbiter.get('now', int(time.time()))
                expire_in = expire_time - server_now
                last_connection_time = server_now - insert_time
                if arbiter.get('from_retention', False) is False:
                    last_connection_str = 'last connection %s ago' % Utils.print_period(last_connection_time) if last_connection_time >= 0 else ''
                    worse_state = EXIT_STATUS.OK
                else:
                    last_connection_str = 'loaded from retention %s ago [waiting for Arbiter connection]' % Utils.print_period(last_connection_time) if last_connection_time >= 0 else ''
                    worse_state = EXIT_STATUS.WARNING

                if diff_time_with_arbiter > 30:
                    _out = 'server times are different, time shift of %s' % (Utils.print_period(diff_time_with_arbiter))
                    worse_state = EXIT_STATUS.CRITICAL
                    tree_entry['arbiters'].append((name, _out, EXIT_STATUS.CRITICAL))
                elif expire_in < 0:
                    _out = 'Missed connection from arbiter since %s ( > daemon check_interval * max_check_attempts )' % (Utils.print_period(last_connection_time))
                    tree_entry['arbiters'].append((name, _out, EXIT_STATUS.WARNING))
                    if worse_state == EXIT_STATUS.OK:  # do not change if was CRITICAL
                        worse_state = EXIT_STATUS.WARNING
                else:
                    _out = '%s' % last_connection_str
                    tree_entry['arbiters'].append((name, _out, worse_state))
            
            if not architecture_name:
                architecture_name = '( too old Shinken Enterprise version, cannot have architecture name )'
            tree_entry['architecture_name'] = architecture_name
        
        self._state = worse_state
    
    
    def get_in_conflict(self):
        return self._in_conflict
    
    
    def get_in_architecture_names_conflict(self):
        return self._in_architecture_names_conflict
    
    
    def get_state(self):
        return self._state
    
    
    def get_arbiters_tree(self):
        return self._arbiters_tree


class ShinkenUtils:
    
    @staticmethod
    def request_get_daemon(result, daemon_type, base_uri, uri, use_ssl=False, timeout=3):
        try:
            return Utils.request_get(result, base_uri, uri, use_ssl, timeout=timeout)
        except NotFoundException:
            result.hard_exit(EXIT_STATUS.CRITICAL, 'Cannot connect to uri : "%s%s". %sThe uri does not exists (error 404)' % (base_uri, uri, BREAK_LINE))
        except Exception as e:
            result.hard_exit(EXIT_STATUS.CRITICAL, 'Cannot connect to %s daemon at %s' % (daemon_type, base_uri), 'cause by : %s' % e)
    
    
    @staticmethod
    def add_module_info(result, data):
        module_infos = data.get('modules_info', {})
        if not module_infos:
            return ''
        
        lines = []
        status = EXIT_STATUS.OK
        for module_info in module_infos:
            submodules = module_info.get('modules', [])
            submodules_info = '-'
            if submodules:
                sub = ['%s : %s' % (i['name'], HTMLTag.state_tag(i['status'])) for i in submodules]
                submodules_info = HTMLList.simple_list(sub)
            
            last_restart = module_info.get('last_restart', {})
            last_restart = Utils.print_time(last_restart['timestamp']) if last_restart else ''
            module_state = HTMLTag.STATE_WARNING if module_info.get('nb_restart', 0) and module_info['status'] == ModuleState.OK else module_info['status']
            state_tag = HTMLTag.state_tag(module_state)
            if module_info.get('is_no_longer_supported', False):
                state_tag = HTMLTag.tag_value('NO LONGER SUPPORTED', COLOR.RED)
            
            if module_info['status'] != 'OK':
                status = EXIT_STATUS.WARNING
            
            lines.append((module_info['name'], module_info['type'], state_tag, module_info.get('nb_restart', 0), last_restart, submodules_info))
        
        long_output = HTMLTable.table(('Name', 'Type', 'Status', 'Restart in the last 2h', 'Last restart date', 'Submodules'), lines, 'Module info')
        result.add_check(status=status, long_output=long_output)
    
    
    @staticmethod
    def add_http_error_count_message(result, data):
        http_errors_count = data.get('http_errors_count', {})
        if http_errors_count:
            result.add_check(EXIT_STATUS.WARNING, "Some API calls between daemons failed in the last 24 hours (%d errors). Please look at your daemon logs for more details about these errors.%s" % (sum(http_errors_count.values()), BREAK_LINE))
    
    
    @staticmethod
    def add_warning_module_restart(result, data):
        modules_info = data.get('modules_info', [])
        
        restart_module = []
        for module_info in modules_info:
            module_restarts = module_info.get('restarts', [])
            if len(module_restarts) <= 0:
                continue
            
            # Manage restarts stored in the old format : [ts1, ts2, ts3]
            if isinstance(module_restarts[-1], float):
                module_restarts = [{'timestamp': ts, 'reason': "The reason for this restart was not saved"} for ts in module_restarts]
            
            limit_dt = datetime.datetime.now() - datetime.timedelta(minutes=120)
            restart_count = len([i for i in module_restarts if datetime.datetime.fromtimestamp(i['timestamp']) > limit_dt])
            module_info['nb_restart'] = restart_count
            module_info['last_restart'] = module_restarts[-1]
            if restart_count > 0:
                restart_module.append('The module %s has restarted [%s times]. A restart is removed from count after 2h.' % (module_info['name'], restart_count))
        
        if restart_module:
            result.add_check(EXIT_STATUS.WARNING, output=HTMLList.header_list('Some modules have restarted since the last 2h', restart_module))
    
    
    @staticmethod
    def check_arbiter_connection(result, uri, daemon_type, timeout=3):
        # type: (Result, str, str, int) -> None
        data, _ = ShinkenUtils.request_get_daemon(result, daemon_type, uri, '/arbiter_traces_get', timeout=timeout)
        traces = json.loads(data.decode('utf8', 'ignore'))
        if len(traces) == 0:
            # Not yet contacted by an arbiter.
            result.hard_exit(EXIT_STATUS.WARNING, 'Daemon has not been contacted by an arbiter for now.')
            return
        
        verification_tree = ArbiterTraceVerificationTree()
        verification_tree.analyse_arbiter_traces(traces)
        
        arbiters_tree = verification_tree.get_arbiters_tree()
        in_conflict = verification_tree.get_in_conflict()
        in_architecture_names_conflict = verification_tree.get_in_architecture_names_conflict()
        state = verification_tree.get_state()
        arbiters_tree_sorted = sorted(arbiters_tree.items(), key=lambda item: (item[1]['architecture_name'], item[1]['ip']))
        master_arbiter_uuid_outputs = []
        for master_arbiter_uuid, tree_entry in arbiters_tree_sorted:
            architecture_name = tree_entry['architecture_name']
            
            arbiter_outputs = []
            for arbiter_entry in tree_entry['arbiters']:
                # 0: name, 1: txt, 2: status, 3: arbiter status
                if arbiter_entry[2] == EXIT_STATUS.OK:
                    prefix = ' '
                elif arbiter_entry[2] == EXIT_STATUS.WARNING:
                    prefix = '%s ' % HTMLTag.tag_value('WARNING', COLOR.ORANGE)
                elif arbiter_entry[2] == EXIT_STATUS.CRITICAL:
                    prefix = '%s ' % HTMLTag.tag_value('ERROR', COLOR.RED)
                else:
                    prefix = '%s ' % HTMLTag.tag_value('UNKNOWN', COLOR.GRAY)

                message = '%s%s : <span style="color:purple">%s.</span>' % (prefix, arbiter_entry[0], arbiter_entry[1])
                if in_conflict:
                    comment = HTMLComment.add_tag('Defined on the server with uuid %s (/var/lib/shinken/server.uuid)' % master_arbiter_uuid)
                    message = '%s %s' % (message, comment)
                arbiter_outputs.append(message)
            architecture_header = '<b>%s</b>  [<span style="color:purple">%s</span>] ' % (architecture_name, tree_entry['ip'])
            master_arbiter_output = '%s<br/>' % HTMLList.header_list(architecture_header, arbiter_outputs, compact=True)
            master_arbiter_uuid_outputs.append(master_arbiter_output)
        
        if in_conflict:
            title = HTMLTag.tag_value('<b>Arbiters CONFLICT</b>', COLOR.RED)
        else:
            title = 'Arbiters connection:'
        
        architecture_list = HTMLList.header_list('<p>Architecture List :</p>', master_arbiter_uuid_outputs, ordered_list=True)
        output = '%s<BLOCKQUOTE style="font-size:14px;border:none;">%s</BLOCKQUOTE>' % (title, architecture_list)
        
        # We have architecture_name conflict, we need to help the user about how to find the servers
        if in_architecture_names_conflict:
            output += 'NOTE: <i><ul>Some architecture have the same name. We advise you to change it in the configuration of their architecture_export module.</i></ul>'
        
        # Only display architecture if there is some conflicts
        if state != EXIT_STATUS.OK:
            result.add_check(state, output)
    
    
    @staticmethod
    def _check_versions(daemon_type, daemon_api_version, daemon_version, arbiter_version, shinken_supervisor_version):
        # type: (str, str, str, str, str) -> Optional[str]
        # Let's check first the API version
        if daemon_api_version and daemon_api_version != API_VERSION:
            if daemon_version and shinken_supervisor_version:
                return 'Your %s is alive but this daemon (%s) and the Shinken installation (%s) that monitors it are not in the same version.' % (
                    daemon_type, get_version_for_output(daemon_version), get_version_for_output(shinken_supervisor_version))
            return 'Your %s is alive but not up to date. Please update.' % daemon_type
        
        if not daemon_version or not daemon_api_version:
            return 'Your %s is alive but not up to date. Please update.' % daemon_type
        
        if daemon_type != 'synchronizer' and not arbiter_version:
            return 'Your %s is alive but not up to date. Please update.' % daemon_type
        
        # the synchronizer has no arbiter_version because it doesn't use 'put_conf' but it's always on the same machine as the arbiter, so we don't need to test it
        if daemon_type != 'synchronizer':
            if daemon_version != arbiter_version:
                return 'Your %s is alive but its version (%s) is not the same as its arbiter (%s). Please update.' % (daemon_type, get_version_for_output(daemon_version), get_version_for_output(arbiter_version))
        
        return None
    
    
    @staticmethod
    def minimal_check(result, data, daemon_type, shinken_supervisor_version):
        is_interrupted = data.get('is_interrupted', False)
        if is_interrupted:
            output = 'The %s is performing a shutdown.' % daemon_type
            result.hard_exit(EXIT_STATUS.WARNING, output)
        
        have_conf = data.get('have_conf', False)
        if not have_conf:
            output = 'No configuration given by an Arbiter for now.'
            result.hard_exit(EXIT_STATUS.WARNING, output)
        
        # If the daemon have his type in get_raw_stats, we compare it with the wanted daemon_type. Else, We pass because lower API_VERSION can be checked
        checked_daemon_type = data.get('daemon_type', None)
        if checked_daemon_type and checked_daemon_type != daemon_type:
            output = 'The daemon being checked is not a %s. This daemon is a %s. Please check the connection information (address, port) in the check configuration.' % (daemon_type, checked_daemon_type)
            result.hard_exit(EXIT_STATUS.WARNING, output)
        
        daemon_version = data.get('daemon_version', None)
        arbiter_version = data.get('arbiter_version', None)
        daemon_api_version = data.get('api_version', None)
        output = ShinkenUtils._check_versions(daemon_type, daemon_api_version, daemon_version, arbiter_version, shinken_supervisor_version)
        if output:
            result.hard_exit(EXIT_STATUS.WARNING, output)
        
        result.set_spare_info(data, daemon_type)
        result.set_logger_stats(data)  # if logger is too slow, will be WARNING
        
        # Before any test, we need to check if the daemon is deadlocked, because other tests would be meaningless
        assert_daemon_is_not_deadlocked_from_monitoring_check(result, data)
        check_daemon_serialization_security_errors(result, data)
    
    
    @staticmethod
    def _check_cpu_stolen(result, hypervisor_type, dict_value_cpu, cpu_steal_threshold_warning, cpu_steal_threshold_critical, output_in_li):
        # type: (Result, str, Dict, int, int, bool) -> None
        return_msg, exit_status, perfdatas = ShinkenUtils._compute_status_for_cpu_stolen(hypervisor_type, dict_value_cpu, cpu_steal_threshold_warning, cpu_steal_threshold_critical, output_in_li)
        # In all cases, stack the perfdatas
        for metric_name, metric_value in perfdatas.items():  # items => manage both py2 and py3
            result.add_perf_data(metric_name, metric_value)
        if exit_status is None:
            return
        no_new_line = output_in_li  # if we are asking a list output, do not force a new line too
        result.add_check(exit_status, return_msg, no_new_line=no_new_line)
    
    
    @staticmethod
    def _compute_status_for_cpu_stolen(hypervisor_type, dict_value_cpu, cpu_steal_threshold_warning, cpu_steal_threshold_critical, output_in_li=False):
        # type: (str, Dict, int, int, bool) -> Tuple[Optional[str], Optional[int], Dict[str, float]]
        exit_status = None
        return_msg = None
        perfdatas = {}
        
        _hypervisor_display = HYPERVISOR_DISPLAY_NAME.get(hypervisor_type, 'UNKNOWN')
        if hypervisor_type == HYPERVISOR.VMWARE:
            cpu_ready = dict_value_cpu[CPU_STOLEN_NAME.CPU_READY]
            message_by_status, exit_status = ShinkenUtils._compute_status_code_for_cpu_stolen(hypervisor_type, cpu_ready, cpu_steal_threshold_warning, cpu_steal_threshold_critical)
            return_msg = message_by_status + '<br/>&nbsp;&nbsp;&nbsp;→ On the VCenter search the data <b>CPU %ready + %costop</b><br/>&nbsp;&nbsp;&nbsp;→ Please have a look at the Shinken Enterprise documentation about advices to reduce it'
            perfdatas['cpu_stolen__vmware'] = '%.2f%%' % cpu_ready
        elif hypervisor_type == HYPERVISOR.HYPERV:
            vptr = dict_value_cpu[CPU_STOLEN_NAME.VPTR]
            lptr = dict_value_cpu[CPU_STOLEN_NAME.LPTR]
            cpu_stolen_value = lptr - vptr
            message_by_status, exit_status = ShinkenUtils._compute_status_code_for_cpu_stolen(hypervisor_type, cpu_stolen_value, cpu_steal_threshold_warning, cpu_steal_threshold_critical)
            return_msg = message_by_status + '%s LPTR: %2.f&#37; VPTR: %2.f&#37; ' % (_hypervisor_display, lptr, vptr)
        elif hypervisor_type == HYPERVISOR.KVM:
            cpu_steal = dict_value_cpu[CPU_STOLEN_NAME.CPU_STEAL]
            message_by_status, exit_status = ShinkenUtils._compute_status_code_for_cpu_stolen(hypervisor_type, cpu_steal, cpu_steal_threshold_warning, cpu_steal_threshold_critical)
            return_msg = message_by_status + '%s CPU steal%% : %2.f&#37.' % (_hypervisor_display, cpu_steal)
        if output_in_li and exit_status == EXIT_STATUS.OK:  # we only want the <li> for ok, because on other
            # we want it in a full bloc
            return_msg = HTMLList.one_bullet_list(return_msg)
        return return_msg, exit_status, perfdatas
    
    
    @staticmethod
    def _compute_status_code_for_cpu_stolen(hypervisor_type, cpu_stolen_value, cpu_steal_threshold_warning, cpu_steal_threshold_critical):
        # type: (str, int, int, int) -> Tuple[Optional[str], Optional[int]]
        _msg_cpu_ok = 'You don\'t have a stolen cpu on your machine → '
        
        if hypervisor_type == HYPERVISOR.VMWARE:
            _msg_cpu_steal_not_ok = 'Your machine got <b>%d%% of CPU STOLEN</b> from the Hypervisor ( <i>Type %s</i> )' % (cpu_stolen_value, HYPERVISOR_DISPLAY_NAME.get(hypervisor_type, 'UNKNOWN'))
        else:
            _msg_cpu_steal_not_ok = 'Your machine got <b>CPU STOLEN</b> from the Hypervisor ( <i>Type %s</i> ) → ' % (HYPERVISOR_DISPLAY_NAME.get(hypervisor_type, 'UNKNOWN'))
        
        message_by_status = _msg_cpu_ok
        exit_status = EXIT_STATUS.OK
        if cpu_stolen_value == 0:
            message_by_status = _msg_cpu_ok
        elif cpu_stolen_value >= cpu_steal_threshold_critical:
            exit_status = EXIT_STATUS.CRITICAL
            message_by_status = _msg_cpu_steal_not_ok
        elif cpu_stolen_value >= cpu_steal_threshold_warning:
            exit_status = EXIT_STATUS.WARNING
            message_by_status = _msg_cpu_steal_not_ok
        
        return message_by_status, exit_status
    
    
    # NOTE: result != None => call from a check
    #       result == None => ?
    @staticmethod
    def check_hypervisor_vm_cpu_stolen(data, cpu_steal_threshold_warning, cpu_steal_threshold_critical, result=None, output_in_li=False):
        # type: (Dict[str, Any], int, int, Result, bool) -> Tuple[Optional[str], Optional[int], Dict]
        message_by_status = None
        exit_status = None
        perfdatas = {}
        if data.get('vmware_stats_enabled') and data.get(HYPERVISOR.VMWARE, False) and data.get('vmware_vm', False):
            dict_value_cpu = {
                CPU_STOLEN_NAME.CPU_READY: data[CPU_STOLEN_NAME.CPU_READY]
            }
            hypervisor_type = HYPERVISOR.VMWARE
        elif data.get(HYPERVISOR.HYPERV, False):
            dict_value_cpu = {
                CPU_STOLEN_NAME.VPTR: data[CPU_STOLEN_NAME.VPTR],
                CPU_STOLEN_NAME.LPTR: data[CPU_STOLEN_NAME.LPTR]
            }
            hypervisor_type = HYPERVISOR.HYPERV
        elif data.get(HYPERVISOR.KVM, False):
            dict_value_cpu = {
                CPU_STOLEN_NAME.CPU_STEAL: data[CPU_STOLEN_NAME.CPU_STEAL]
            }
            hypervisor_type = HYPERVISOR.KVM
        else:
            return message_by_status, exit_status, perfdatas
        
        if result:
            ShinkenUtils._check_cpu_stolen(result, hypervisor_type, dict_value_cpu, cpu_steal_threshold_warning, cpu_steal_threshold_critical, output_in_li=output_in_li)
        else:
            message_by_status, exit_status, perfdatas = ShinkenUtils._compute_status_for_cpu_stolen(hypervisor_type, dict_value_cpu, cpu_steal_threshold_warning, cpu_steal_threshold_critical)
            return message_by_status, exit_status, perfdatas
    
    
    # Look at raw data about the spare information:
    # is_spare yes/no? look at spare_daemon, master_daemon (if existing!)
    # NOTE: spare case is manage on the minimal check, because it can break/stops all results
    @staticmethod
    def check_master_information(data, result):
        # type: (Dict[str, Any], Result) -> Tuple[Optional[str], Optional[int]]
        
        # check only daemons that manage it(currently Brokers)
        if 'spare' not in data or 'master_daemon' not in data or 'spare_daemon' not in data:
            return None, None
        
        is_spare = data.get('spare')  # we did assert it exists
        spare_daemon = data.get('spare_daemon')
        if not is_spare:
            if spare_daemon != '':
                return_msg = 'The %s daemon is %s' % (HTMLTag.tag_value('SPARE', COLOR.BLUE), HTMLTag.tag_value(spare_daemon, COLOR.GREEN))
                if not data.get('spare_must_have_the_same_list_of_module_type', True):
                    return_msg = "%s ( the broker configuration doesn't require the same list of modules on its spare )" % return_msg
            else:
                return_msg = 'This daemon do not have any spare defined'
            result.add_check(EXIT_STATUS.OK, HTMLList.one_bullet_list(return_msg), no_new_line=True)
    
    
    @staticmethod
    def get_satellites_connection(result, base_uri, timeout, raise_excepton=False):
        # type: (Result, str, int, Optional[bool]) -> (List, float)
        buf, connexion_time = Utils.request_get(result, base_uri, '/check_satellites_connexion?timeout=%s' % timeout, timeout=timeout, raise_exp=raise_excepton)
        return json.loads(buf.decode('utf8', 'ignore')), connexion_time


HTMLTag.CRITICAL = HTMLTag.state_tag(HTMLTag.STATE_CRITICAL)
HTMLTag.WARNING = HTMLTag.state_tag(HTMLTag.STATE_WARNING)
HTMLTag.OK = HTMLTag.state_tag(HTMLTag.STATE_OK)

TAG_OK = HTMLTag.color_text('OK', COLOR.GREEN)
TAG_WARNING = HTMLTag.color_text('WARNING', COLOR.ORANGE)
TAG_CRITICAL = HTMLTag.color_text('CRITICAL', COLOR.RED)
TAG_UNKNOWN = HTMLTag.color_text('UNKNOWN', COLOR.BLACK)

TAG_FOR_STATE = {
    EXIT_STATUS.OK      : TAG_OK,
    EXIT_STATUS.WARNING : TAG_WARNING,
    EXIT_STATUS.CRITICAL: TAG_CRITICAL,
    EXIT_STATUS.UNKNOWN : TAG_UNKNOWN,
}


def assert_daemon_is_not_deadlocked_from_monitoring_check(result, raw_stats):
    # type: (Result, Dict) -> None
    dead_lock_status = raw_stats.get('dead_lock', None)
    if dead_lock_status is None:  # old daemon (before 001.9, december 2021), does not have this info, so skip test
        return
    # Let the watch dog fatal object check itself if it's ok or not ^^
    dead_lock_status_code, dead_lock_output = watchdog_fatal_status.check_status_and_output_html(dead_lock_status)
    if dead_lock_status_code == WATCH_DOG_STATUS_CODE.CRITICAL:
        result.hard_exit(EXIT_STATUS.CRITICAL, dead_lock_output, '')


def check_daemon_serialization_security_errors(result, raw_stats):
    # type: (Result, Dict) -> None
    serialization_security_errors = raw_stats.get('serialization_security_errors', None)
    
    if serialization_security_errors is None:  # old daemon (before feb 2022), drop the check
        return
    
    # Checking if we got security errors for the daemon
    serialization_security_status_code, serialization_security_output = serialization_security_container.check_status_and_output_html(serialization_security_errors)
    if serialization_security_status_code == SERIALIZATION_SECURITY_STATUS_CODE.CRITICAL:
        result.add_check(EXIT_STATUS.WARNING, serialization_security_output, '')


def get_version_for_output(version):
    # type: (str) -> str
    return version if version.startswith('V') else 'V%s' % version
