#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import threading

from ..portalocker import RLock, LOCK_EX, LOCK_SH
from ...os_helper import set_ownership

SHINKEN_RLOCK_FILE_OPEN_KWARGS = {'buffering': 0}


def create_tree(path, user='shinken', group='shinken', mode=0o755):
    # Warning! the mode will be modified by os.umask if set
    drive, path = os.path.splitdrive(path)
    full_path = [p for p in path.split(os.sep) if p]
    parents = ('%s%s' % (drive, os.sep)) if os.path.isabs(path) else ''
    for sub_dir in full_path:
        current_path = os.path.join(parents, sub_dir)
        
        if not os.path.exists(current_path):
            try:
                os.mkdir(current_path, mode)
                set_ownership(current_path, user, group)
            except FileExistsError:
                # We may have a race condition here
                pass
        # Now, next level
        parents = current_path


class ShinkenInterProcessRLock:
    def __init__(self, filename, shared_lock=False):
        
        self._lock_file = filename
        self._lock_mode = LOCK_SH if shared_lock else LOCK_EX
        self._locks = {}
        dirname = os.path.dirname(filename)
        create_tree(dirname)
        if not os.path.exists(filename):
            open(filename, 'w+').close()  # File must exist
            set_ownership(filename)
    
    
    def destroy(self):
        for thread_locks in self._locks.values():
            for other_lock in thread_locks.values():
                if other_lock and other_lock.fh and not other_lock.fh.closed:
                    other_lock.fh.close()
        self._locks = {}
        try:
            os.unlink(self._lock_file)
        except:
            pass
    
    
    def _get_lock(self):
        # type: () -> RLock
        pid = os.getpid()
        if pid not in self._locks:
            # after a fork, cleanup rogue locks
            for thread_locks in self._locks.values():
                for other_lock in thread_locks.values():
                    if other_lock and other_lock.fh and not other_lock.fh.closed:
                        other_lock.fh.close()
            self._locks = {pid: {}}
        thread_id = threading.current_thread().ident
        if thread_id not in self._locks[pid]:
            self._locks[pid][thread_id] = RLock(self._lock_file, timeout=None, mode='a+b', flags=self._lock_mode)
            self._locks[pid][thread_id].file_open_kwargs = SHINKEN_RLOCK_FILE_OPEN_KWARGS
        
        return self._locks[pid][thread_id]
    
    
    def acquire(self):
        self._get_lock().acquire()
    
    
    def release(self):
        self._get_lock().release()
    
    
    def __enter__(self):
        self._get_lock().acquire()
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self._get_lock().release()
