#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import functools
from abc import ABCMeta, abstractmethod

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.validators.validation import ValidationMessages

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Dict, Any, Optional


class ValidationResult:
    def __init__(self, should_be_done, messages=None):
        # type: (bool, Optional[ValidationMessages]) -> None
        self.messages = messages or ValidationMessages(warnings=[], errors=[], critical=[])
        self.should_be_done = bool(should_be_done)


class AbstractActionValidator(object, metaclass=ABCMeta):
    @abstractmethod
    def validate(self, action_dict, logger=None):
        # type: (Dict[str, Any], Optional[PartLogger]) -> ValidationResult
        """
        Validates the action dictionary and optionally modifies it
        """
        raise NotImplementedError()
    
    
    # noinspection PyMethodMayBeStatic
    def pythonize(self, action_dict, logger=None):
        # type: (Dict[str, Any], Optional[PartLogger]) -> Any
        """Obtain a specific object from dict"""
        return action_dict


class TrustActionValidator(AbstractActionValidator):
    """Accepts any given data"""
    
    
    def validate(self, action_dict, logger=None):
        # type: (Dict[str, Any], Optional[PartLogger]) -> ValidationResult
        return ValidationResult(should_be_done=True)


class ActionValidatorPool(AbstractActionValidator):
    def __init__(self, name_to_validator_dict):
        # type: (Dict[str, AbstractActionValidator]) -> None
        self._name_to_validator_dict = name_to_validator_dict
    
    
    @staticmethod
    def _merge_result(lhs_result, rhs_result):
        # type: (ValidationResult, ValidationResult) -> ValidationResult
        return ValidationResult(
            should_be_done=lhs_result.should_be_done and rhs_result.should_be_done,
            messages=lhs_result.messages + rhs_result.messages,
        )
    
    
    def validate(self, action_dict, logger=None):
        # type: (Dict[str, Any], Optional[PartLogger]) -> ValidationResult
        result = functools.reduce(
            self._merge_result,
            (validator.validate(action_dict, logger=logger) for validator in iter(self._name_to_validator_dict.values())),
            ValidationResult(should_be_done=True),
        )
        return result
    
    
    def pythonize(self, action_dict, logger=None):
        # type: (Dict[str, Any], Optional[PartLogger]) -> Dict[str, Any]
        return dict((key, validator.pythonize(action_dict, logger=logger)) for key, validator in self._name_to_validator_dict.items())
