#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import PART_INITIALISATION, LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.lib_modules.configuration_reader_mixin import SeparatorFormat, ConfigurationFormat, TypeConfiguration, ConfigurationReaderMixin

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from shinken.misc.type_hint import Optional


class MongoConf(ConfigurationReaderMixin):
    def __init__(self, _conf, logger=None, default_database='shinken', prefix_module_property=''):
        # type: (ShinkenModuleDefinition, PartLogger, str, str) -> None
        self.name_database = None  # type: Optional[str]
        self.uri = None  # type: Optional[str]
        self.replica_set = None  # type: Optional[str]
        self.ssh_keyfile = None  # type: Optional[str]
        self.ssh_user = None  # type: Optional[str]
        self.use_ssh_retry_failure = None  # type: Optional[bool]
        self.ssh_tunnel_timeout = None  # type: Optional[int]
        self.use_ssh_tunnel = None  # type: Optional[bool]
        self.auto_reconnect_max_try = None  # type: Optional[int]
        self.auto_reconnect_sleep_between_try = None  # type: Optional[int]
        
        if logger:
            self.logger = logger  # type: PartLogger
        else:
            self.logger = LoggerFactory.get_logger()  # type: PartLogger
        
        self.logger_init = self.logger.get_sub_part(PART_INITIALISATION).get_sub_part('MONGO')  # type: PartLogger
        
        configuration_format = self.get_configuration_format(prefix_module_property, default_database)
        ConfigurationReaderMixin.__init__(self, configuration_format, _conf, self.logger_init)
        self.read_configuration()
    
    
    @staticmethod
    def get_configuration_format(prefix_module_property='', default_database='shinken'):
        if prefix_module_property:
            prefix_module_property = '%s__' % prefix_module_property
        configuration_format = [
            SeparatorFormat('Connection parameters'),
            ConfigurationFormat(['database', 'name_database', 'mongodb_database', '%sdatabase__name' % prefix_module_property], default_database, TypeConfiguration.STRING, 'name_database'),
            ConfigurationFormat(['uri', 'mongodb_uri', '%sdatabase__uri' % prefix_module_property], 'mongodb://localhost/?w=1&fsync=false', TypeConfiguration.STRING, 'uri'),
            ConfigurationFormat(['replica_set', 'mongodb_replica_set', '%sdatabase__replica_set' % prefix_module_property], '', TypeConfiguration.STRING, 'replica_set'),
            ConfigurationFormat(['use_ssh_tunnel', 'mongodb_use_ssh_tunnel', '%suse_ssh_tunnel' % prefix_module_property, '%sdatabase__use_ssh_tunnel' % prefix_module_property], 0, TypeConfiguration.INT, 'use_ssh_tunnel'),
            ConfigurationFormat(['use_ssh_retry_failure', 'mongodb_use_ssh_retry_failure', '%suse_ssh_retry_failure' % prefix_module_property, '%sdatabase__use_ssh_retry_failure' % prefix_module_property], 1, TypeConfiguration.INT,
                                'use_ssh_retry_failure'),
            ConfigurationFormat(['ssh_user', 'mongodb_ssh_user', '%sssh_user' % prefix_module_property, '%sdatabase__ssh_user' % prefix_module_property], 'shinken', TypeConfiguration.STRING, 'ssh_user'),
            ConfigurationFormat(['ssh_keyfile', 'mongodb_ssh_keyfile', '%sssh_keyfile' % prefix_module_property, '%sdatabase__ssh_keyfile' % prefix_module_property], '~shinken/.ssh/id_rsa', TypeConfiguration.STRING, 'ssh_keyfile'),
            ConfigurationFormat(['ssh_tunnel_timeout', 'mongodb_ssh_tunnel_timeout', 'mongodb_retry_timeout', '%sssh_tunnel_timeout' % prefix_module_property, '%sdatabase__ssh_tunnel_timeout' % prefix_module_property], 10,
                                TypeConfiguration.INT, 'ssh_tunnel_timeout'),
            
            SeparatorFormat('AutoReconnect Management'),
            ConfigurationFormat(['auto_reconnect_max_try', 'mongodb_auto_reconnect_max_try', '%sdatabase__retry_connection_X_times_before_considering_an_error' % prefix_module_property], 5, TypeConfiguration.INT, 'auto_reconnect_max_try'),
            ConfigurationFormat(['auto_reconnect_sleep_between_try', 'mongodb_auto_reconnect_sleep_between_try', '%sdatabase__wait_X_seconds_before_reconnect' % prefix_module_property], 5, TypeConfiguration.INT,
                                'auto_reconnect_sleep_between_try'),
        
        ]
        return configuration_format
