#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2019:
#
# This file is part of Shinken Enterprise, all rights reserved.

import re
from shinken.daterange import Daterange


class TimePeriodDay:
    def __init__(self, token):
        self.value = token[0]
        self.week = token[1]
        if len(token) > 2:
            self.month = token[2]
            self.flat_value = '%s %s %s' % (self.week, self.month, self.value)
        else:
            self.month = "None"
            self.flat_value = '%s %s' % (self.week, self.value)
    
    
    def __repr__(self):
        return 'value=%s week=%s month=%s' % (self.value, self.week, self.month)


class TimePeriodAdvanced:
    def __init__(self, token):
        self.key = token[0]
        self.value = token[1]
        self.flat_value = '%s %s' % (self.key, self.value)


    def __repr__(self):
        return 'key=%s value=%s' % (self.key, self.value)


class PERIOD_TYPE:
    BASIC_DAY = "basic_day"
    DAY_WITH_OPTIONS = "day_with_options"
    ADVANCED = "advanced"


class TimePeriodParser:
    SEPARATOR_DAY_DEF = "-#-"
    SEPARATOR_DAY_OPT = "~#~"
    SEPARATOR_ADVANCED_DEF = "-#-"
    SEPARATOR_ADVANCED_KEY_VALUE = "~#~"
    
    # Matches one or multiple time ranges HH:MM-HH:MM [, HH:MM-HH:MM ...]
    TIME_RANGES_REGEX = re.compile(r'(\d\d:\d\d-\d\d:\d\d)(\s*,\s*\d\d:\d\d-\d\d:\d\d)*')
    TIME_RANGES_REGEX_EXACT_MATCH = re.compile(r'^(\d\d:\d\d-\d\d:\d\d)(\s*,\s*\d\d:\d\d-\d\d:\d\d)*$')
    
    # Matches signed integers
    BASIC_OPT_REGEX = re.compile(r'^-?[\d+]$')


    # Parsing methods  for rendering to User
    
    @staticmethod
    def tokenize_day(to_split):
        _to_return = []
        _defs = to_split.split(TimePeriodParser.SEPARATOR_DAY_DEF)
        for _def in _defs:
            _options = _def.split(TimePeriodParser.SEPARATOR_DAY_OPT)
            _to_return.append(_options)
        return _to_return
    
    
    @staticmethod
    def tokenize_advanced(to_split):
        _to_return = []
        _defs = to_split.split(TimePeriodParser.SEPARATOR_ADVANCED_DEF)
        for _def in _defs:
            _options = _def.split(TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE)
            _to_return.append(_options)
        return _to_return
    
    
    @staticmethod
    def parse_day(tokens):
        _to_return = {
            'value': tokens[0][0],
            'defs': []
        }
        for _token in tokens[1:]:
            _to_add = TimePeriodDay(_token)
            _to_return['defs'].append(_to_add)
        return _to_return
    
    
    @staticmethod
    def parse_advanced(tokens):
        _to_return = []
        for _token in tokens:
            _to_add = TimePeriodAdvanced(_token)
            _to_return.append(_to_add)
        return _to_return


    # Methods for generating Timeperiod values for the Timeperiod parser in the arbiter
    
    @staticmethod
    def explode_advanced_time_periods_into_multiple_definitions(key, params):
        return [advanced_period.replace(TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE, ' ')
                for advanced_period in params[key].split(TimePeriodParser.SEPARATOR_ADVANCED_DEF)]


    @staticmethod
    def explode_simple_time_period_into_multiple_definitions(key, params):
        exploded_time_periods = []
        periods = params[key].split(TimePeriodParser.SEPARATOR_DAY_DEF)
        for period in periods:
            if period == '':
                continue
            options = period.split(TimePeriodParser.SEPARATOR_DAY_OPT)
            hour_ranges = options.pop(0)
            options.append(hour_ranges)

            exploded_time_periods.append('%s %s' % (key, " ".join(options)))
        return exploded_time_periods


    ### Methods for parsing timeperiods in the CFG format
    
    @staticmethod
    def parse_time_periods_from_cfg(first_word, period_value):
        period_match = TimePeriodParser.TIME_RANGES_REGEX.search(period_value)
        if period_match is None:
            parsed_period = period_value
            period_type = PERIOD_TYPE.ADVANCED
        else:
            time_ranges = period_value[period_match.start():period_match.end()]
            if first_word not in Daterange.weekdays:
                return {
                    'value': "%s%s%s" % (period_value[:period_match.start()].strip(), TimePeriodParser.SEPARATOR_ADVANCED_KEY_VALUE, time_ranges),
                    'type' : PERIOD_TYPE.ADVANCED
                }
            if period_match.start() == 0:
                period_type = PERIOD_TYPE.BASIC_DAY
                parsed_period = time_ranges
            else:
                parsed_opts = [opt.strip() for opt in period_value[:period_match.start()].split()]
                opt_results = TimePeriodParser.SEPARATOR_DAY_OPT.join(parsed_opts)
                period_type = PERIOD_TYPE.DAY_WITH_OPTIONS
                for opt in parsed_opts:
                    if TimePeriodParser.BASIC_OPT_REGEX.match(opt):
                        continue
                    if opt in Daterange.months:
                        continue
                    period_type = PERIOD_TYPE.ADVANCED

                if period_type == PERIOD_TYPE.ADVANCED:
                    parsed_period = "%s%s%s" % (period_value[:period_match.start()].strip(), TimePeriodParser.SEPARATOR_DAY_OPT, time_ranges)
                else:
                    parsed_period = "%s%s%s" % (time_ranges, TimePeriodParser.SEPARATOR_DAY_OPT, opt_results)

        period_match = parsed_period.replace('`\t', ' ')
        period_match = " ".join(period_match.split())
        period_match = period_match.strip()
        return {'value': period_match, 'type': period_type}

