#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.
import json
import traceback
import uuid

import pickle
import base64
from shinken.compat import SHINKEN_PICKLE_PROTOCOL
from shinken.daemon import Interface, Daemon
from shinken.http_client import HTTPClient
from shinken.http_daemon import HTTPDaemonBusinessError
from shinken.log import LoggerFactory
from shinken.misc.type_hint import Callable, List, Any, Dict, Tuple, TYPE_CHECKING

if TYPE_CHECKING:
    import types

logger = LoggerFactory.get_logger('remote_callable')

remote_callables = {}  # type: Dict[Tuple[str, str], Callable]


class RemoteCallException(Exception):
    def __init__(self, text):
        # type: (str) -> None
        self.text = text
    
    
    def __repr__(self):
        # type: () -> str
        return self.text
    
    
    def __str__(self):
        # type: () -> str
        return 'RemoteCallException[text:%s]' % self.text


class RemoteCallBusinessError(HTTPDaemonBusinessError):
    def __init__(self, exception):
        # type: (Exception) -> None
        self.exception = exception
        self.stack = traceback.format_exc().splitlines()
    
    
    def __repr__(self):
        # type: () -> str
        return json.dumps({
            'exception': self.exception,
            'stack'    : self.stack,
        })
    
    
    def __str__(self):
        # type: () -> str
        return 'RemoteCallBusinessError[exception:%s]' % self.exception


def is_remote_callable(f):
    # type: (Callable) -> Callable
    f.is_remote_callable = True
    return f


class RemoteCallableObject:
    def __init__(self, remote_instance_id=None):
        if remote_instance_id is None:
            remote_instance_id = uuid.uuid4().hex
        self._remote_instance_id = remote_instance_id
        
        all_parents = []
        to_scan_parents = list(self.__class__.__bases__)
        watch_dog = 0
        while to_scan_parents:
            cursor = to_scan_parents.pop()
            if cursor is object:
                continue
            watch_dog += 1
            if watch_dog > 50:
                break
            all_parents.append(cursor)
            
            for next_to_scan_parents in cursor.__bases__:
                if next_to_scan_parents not in to_scan_parents and next_to_scan_parents not in all_parents:
                    to_scan_parents.append(next_to_scan_parents)
        
        for instance_member_name in dir(self):
            instance_member = getattr(self, instance_member_name)
            
            if callable(instance_member) and next((getattr(getattr(class_parent, instance_member_name, object), 'is_remote_callable', False) for class_parent in all_parents), False):
                remote_callables[(self._remote_instance_id, instance_member.__name__)] = instance_member


class RemoteCaller:
    def internal_call(self, function_to_call, *args, **kwargs):
        # type: (types.MethodType, List[Any], Dict[str, Any]) -> Any
        
        remote_instance_id = getattr(function_to_call.__self__, '_remote_instance_id', None)
        method_name = function_to_call.__name__
        
        if not remote_instance_id:
            raise RemoteCallException('function to call [%s] was not in a RemoteCallableObject' % function_to_call)
        
        return self.internal_call(remote_instance_id, method_name, *args, **kwargs)
    
    
    def internal_call_by_instance_id(self, remote_instance_id, method_name, *args, **kwargs):
        # type: (str, str, List[Any], Dict[str, Any]) -> Any
        raise NotImplementedError()


class HTTPRemoteCaller(RemoteCaller):
    
    def __init__(self, host='', port=0, use_ssl=False):
        # type: (str, int, bool) -> None
        self.con = HTTPClient(address=host, port=port, use_ssl=use_ssl)
    
    
    def internal_call_by_instance_id(self, remote_instance_id, method_name, *args, **kwargs):
        # type: (str, str, List[Any], Dict[str, Any]) -> Any
        
        params = {
            'remote_instance_id': remote_instance_id,
            'method_name'       : method_name,
            'args'              : args,
            'kwargs'            : kwargs,
        }
        post = self.con.post('remote_call', params)
        return pickle.loads(base64.b64decode(post))


class HTTPRemoteCallRoute(Interface):
    def __init__(self, app):
        # type: (Daemon) -> None
        super(HTTPRemoteCallRoute, self).__init__(app, default_lock=False)
    
    
    def remote_call(self, remote_instance_id, method_name, args, kwargs):
        # type: (str, str, List[Any], Dict[str, Any]) -> bytes
        remote_callable = remote_callables.get((remote_instance_id, method_name), None)
        if not remote_callable:
            raise RemoteCallException('function to call [%s] was not found' % method_name)
        
        try:
            ret_value = remote_callable(*args, **kwargs)
            ret_value = base64.b64encode(pickle.dumps(ret_value, SHINKEN_PICKLE_PROTOCOL))
            return ret_value
        except Exception as e:
            raise RemoteCallBusinessError(e)
    
    
    remote_call.method = 'post'
