#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2024:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.url_helper import BaseUrl, Protocol

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Iterator, Union, Optional, Tuple


class GraphiteBackendException(Exception):
    pass


class GraphiteBackendConsts:
    PROTOCOLS__SUPPORTED_PROTOCOLS = [
        Protocol('http', 80, False),
        Protocol('https', 443, True)
    ]
    PROTOCOLS__SUPPORTED_PROTOCOL_NAMES = [p.get_protocol_name() for p in PROTOCOLS__SUPPORTED_PROTOCOLS]
    
    SEPARATOR__REALM = '='
    SEPARATOR__BACKEND_LIST = ','
    SEPARATOR__PROTOCOL = '://'
    SEPARATOR__HOST_PORT = ':'
    
    FORMAT__EXPECTED = '<REALM>=<PROTOCOL>://<HOSTNAME>:<PORT>'
    
    ERROR__MISSING_REALM = 'No realm name found'
    ERROR__TOO_MUCH_REALM_SEPARATOR = 'Too much realm separators found ( %s )' % SEPARATOR__REALM
    ERROR__MISSING_PROTOCOL = 'No protocol found ( http:// )'
    ERROR__MISSING_HOSTNAME = 'No hostname or IP address found'
    ERROR__NOT_SUPPORTED_PROTOCOL = 'Protocol not supported. Supported protocol list : [ %s ]' % ', '.join(PROTOCOLS__SUPPORTED_PROTOCOL_NAMES)
    ERROR__MISSING_PORT = 'No HTTP port found'


class GraphiteBackend:
    def __init__(self, graphite_backend, strict=True):
        # type: (str, bool) -> None
        graphite_backend = graphite_backend.strip()
        self._errors = []  # type: List[str]
        self.cfg_url = graphite_backend
        self.origin_backend = graphite_backend
        self._strict = strict
        
        self.realm, self.base_url = self._validate_backend(graphite_backend)
        self._add_base_url_errors()
    
    
    def _validate_backend(self, graphite_backend):
        # type: (str) -> Tuple[Optional[str], Optional[BaseUrl]]
        realm = None  # type: Optional[str]
        base_url = None  # type: Optional[BaseUrl]
        # We still want to get the realm of the graphite_backend has error to eventually display it later
        has_valid_protocol_separator = True
        if GraphiteBackendConsts.SEPARATOR__REALM not in graphite_backend:
            self._handle_error(GraphiteBackendConsts.ERROR__MISSING_REALM)
            has_valid_protocol_separator = False
        if len(graphite_backend.split(GraphiteBackendConsts.SEPARATOR__REALM)) > 2:
            self._handle_error(GraphiteBackendConsts.ERROR__TOO_MUCH_REALM_SEPARATOR)
            has_valid_protocol_separator = False
        if GraphiteBackendConsts.SEPARATOR__PROTOCOL not in graphite_backend:
            self._handle_error(GraphiteBackendConsts.ERROR__MISSING_PROTOCOL)
        elif has_valid_protocol_separator:
            realm_protocol, url = graphite_backend.split(GraphiteBackendConsts.SEPARATOR__PROTOCOL, 1)
            can_process_url = True
            if GraphiteBackendConsts.SEPARATOR__REALM not in realm_protocol:
                self._handle_error(GraphiteBackendConsts.ERROR__MISSING_REALM)
                can_process_url = False
            else:
                realm, protocol = realm_protocol.split(GraphiteBackendConsts.SEPARATOR__REALM)
                
                if protocol.strip() == '':
                    self._handle_error(GraphiteBackendConsts.ERROR__MISSING_PROTOCOL)
                    can_process_url = False
                if realm.strip() == '':
                    self._handle_error(GraphiteBackendConsts.ERROR__MISSING_REALM)
                    can_process_url = False
                if protocol.strip() not in GraphiteBackendConsts.PROTOCOLS__SUPPORTED_PROTOCOL_NAMES:
                    self._handle_error(GraphiteBackendConsts.ERROR__NOT_SUPPORTED_PROTOCOL, is_format_error=False)
            
            if GraphiteBackendConsts.SEPARATOR__HOST_PORT not in url:
                self._handle_error(GraphiteBackendConsts.ERROR__MISSING_PORT)
                can_process_url = False
            else:
                host, port = url.split(GraphiteBackendConsts.SEPARATOR__HOST_PORT)
                if host == '':
                    self._handle_error(GraphiteBackendConsts.ERROR__MISSING_HOSTNAME)
                    can_process_url = False
                if port == '':
                    self._handle_error(GraphiteBackendConsts.ERROR__MISSING_PORT)
                    can_process_url = False
            
            if can_process_url:
                realm, url = graphite_backend.split(GraphiteBackendConsts.SEPARATOR__REALM)
                base_url = BaseUrl.from_url(url, strict=self._strict)
        
        return realm, base_url
    
    
    def _handle_error(self, what_is_missing, is_format_error=True):
        # type: (str, bool) -> None
        if is_format_error:
            error_msg = 'Graphite Backend [ %s ] format error. %s. Expected format : %s' % (self.cfg_url, what_is_missing, GraphiteBackendConsts.FORMAT__EXPECTED)
        else:
            error_msg = 'Graphite Backend [ %s ] URL error : %s' % (self.cfg_url, what_is_missing)
        
        exp = GraphiteBackendException(error_msg)
        if self._strict:
            raise exp
        self._errors.append(str(exp))
    
    
    def _add_base_url_errors(self):
        # type: () -> None
        if self.has_errors():
            return
        for error in self.base_url.get_error_messages():
            exp = GraphiteBackendException('Graphite Backend [ %s ] URL error : %s' % (self.cfg_url, error))
            if self._strict:
                raise exp
            self._errors.append(str(exp))
    
    
    def __iter__(self):
        # type: () -> Iterator[Union[BaseUrl, str]]
        return iter((self.realm, self.base_url, self.cfg_url))
    
    
    @staticmethod
    def get_graphite_backend_string_formatted(realm, protocol, host, port):
        # type: (str, str, str, Union[str, int]) -> str
        protocol = protocol.strip()
        if protocol not in GraphiteBackendConsts.PROTOCOLS__SUPPORTED_PROTOCOL_NAMES:
            raise GraphiteBackendException(GraphiteBackendConsts.ERROR__NOT_SUPPORTED_PROTOCOL)
        return '%s=%s://%s:%s' % (realm.strip(), protocol, host.strip(), port.strip() if isinstance(port, str) else port)
    
    
    @classmethod
    def parse_graphite_backends(cls, graphite_backends, strict=True):
        # type: (str, bool) -> List[GraphiteBackend]
        backends = graphite_backends.split(GraphiteBackendConsts.SEPARATOR__BACKEND_LIST)
        parsed_backends = []
        for backend in (back for back in backends if back):
            parsed_backends.append(cls(backend, strict))
        return parsed_backends
    
    
    def has_errors(self):
        # type: () -> bool
        return bool(self._errors)
    
    
    def get_errors(self):
        # type: () -> List[str]
        return self._errors
