#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import hashlib

from .item import Item, Items
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.property import StringProp, ListProp
from shinken.util import strip_and_uniq
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReader

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Callable, List, Optional, Tuple

# Used in modules manager for automatically loaded modules
FIRST_MODULE_ID_FOR_MODULES_NOT_INSTANTIATED_BY_ARBITER = 10_000


class Module(Item):
    id = 1  # zero is always special in database, so we do not take risk here
    my_type = 'module'
    
    if TYPE_CHECKING:
        module_name = None  # type: Optional[str]
        module_type = None  # type: Optional[str]
        modules = None  # type: Optional[List]
    
    properties = Item.properties.copy()
    properties.update({
        'module_name': StringProp(),
        'module_type': StringProp(),
        'modules'    : ListProp(default=''),
    })
    
    running_properties = Item.running_properties.copy()
    running_properties.update({
        'configuration_default_value': StringProp(default={}),
    })
    
    macros = {}
    
    CONFIGURATION_INCARNATION_KEY = 'configuration_incarnation'
    HASH_IGNORE_KEYS = [CONFIGURATION_INCARNATION_KEY]
    
    
    def __init__(self, params=None, skip_useless_in_configuration=False):
        self.module_type = ''
        self.hash = ''
        if params is None:
            params = {}
        super(Module, self).__init__(params=params)
        self.configuration_properties = set(params.keys())
    
    
    @classmethod
    def reset_new_first_id(cls, value: int) -> None:
        cls.id = value
    
    
    def compute_hash(self):
        hash_ = hashlib.md5()  # don't bother me about blabla md5 security blabla. We don't care here baka
        keys = sorted(self.configuration_properties)
        for k in keys:
            if k in Module.HASH_IGNORE_KEYS:
                continue
            if isinstance(k, str):
                hash_.update(k.encode('utf8', 'ignore'))
            else:
                hash_.update(k)
            value = getattr(self, k, '')
            
            # Because of fill_configuration_incarnation a compute_hash is call after pythonize so value in modules is instance of shinken.objects.module.Module
            # And the str() of this instance change always
            if k == 'modules' and not isinstance(value, str):
                new_value = []
                for m in value:
                    if isinstance(m, Module):
                        new_value.append(m.hash)
                    else:
                        new_value.append(m)
                value = new_value
            
            if isinstance(value, str):
                hash_.update(value.encode('utf8', 'ignore'))
            else:  # maybe it's a list due to set the key multiple times.
                hash_.update(str(value).encode('utf8', 'ignore'))
        self.hash = hash_.hexdigest()
    
    
    def set_some_properties(self, properties):
        # type: (List[Tuple[str, Any, Optional[Callable]]]) -> None
        for property_name, property_value, value_convert in properties:
            if value_convert:
                try:
                    property_value = value_convert(property_value)
                except ValueError:
                    pass
            setattr(self, property_name, property_value)
            self.configuration_properties.add(property_name)
        self.compute_hash()
    
    
    def set_father_config(self, config):
        self.father_config = config
    
    
    def get_name(self):
        return getattr(self, 'module_name', 'unnamed')
    
    
    def get_type(self):
        return getattr(self, 'module_type', '')
    
    
    # For debugging purpose only (nice name)
    def get_modules(self):
        return self.modules


class Modules(Items):
    name_property = "module_name"
    inner_class = Module
    
    
    def fill_running_properties(self, config):
        for module in self:
            module.set_some_properties([
                ('properties_display_text', getattr(config, 'properties_display_text'), None),
                ('configuration_default_value', config.default_properties_values, None),
                ('minimal_time_before_an_element_become_missing_data', config.minimal_time_before_an_element_become_missing_data, int),
                ('minimal_time_before_an_element_become_missing_data_at_startup', config.minimal_time_before_an_element_become_missing_data_at_startup, int),
                ('global__acknowledge__automatic_deletion_condition', config.global__acknowledge__automatic_deletion_condition, str),
                ('modules_dir', config.modules_dir, None),
                ('max_plugins_output_length', config.max_plugins_output_length, int),
                ('language', config.language, str),
            ])
    
    
    def fill_configuration_incarnation(self, configuration_incarnation):
        for module in self:
            module.set_some_properties([
                (Module.CONFIGURATION_INCARNATION_KEY, configuration_incarnation, None),
            ])
    
    
    def linkify(self):
        self.linkify_s_by_plug()
    
    
    def linkify_s_by_plug(self):
        for s in self:
            new_modules = []
            mods = s.modules.split(',')
            mods = strip_and_uniq(mods)
            for plug_name in mods:
                plug_name = plug_name.strip()
                
                # don't read void names
                if plug_name == '':
                    continue
                
                # We are the modules, we search them :)
                plug = self.find_by_name(plug_name)
                if plug is not None:
                    new_modules.append(plug)
                else:
                    err = "[module] unknown %s module from %s" % (plug_name, s.get_name())
                    logger.error(err)
                    s.configuration_errors.append(err)
            
            s.modules = new_modules
            # We need to update the module hash with the ones from the new modules, because if one
            # submodule change, we should change too
            modules_sorted = sorted(s.modules, key=lambda x: x.get_name())
            hash_compute = hashlib.md5(s.hash.encode('utf8', 'ignore'))  # start with the current hash as param and not value, but will do the trick
            for m in modules_sorted:
                hash_compute.update(m.hash.encode('utf8', 'ignore'))
            s.hash = hash_compute.hexdigest()
    
    
    def explode(self):
        pass
    
    
    def is_correct(self):
        # type: () -> bool
        return_value = super(Modules, self).is_correct()
        tmp_list = {}
        for module in self:
            module_name = getattr(module, Modules.name_property, '')
            import_from = getattr(module, 'imported_from', '')
            if module_name:
                if module_name not in tmp_list:
                    tmp_list[module_name] = import_from
                else:
                    self.configuration_errors.insert(0, 'The module name [%s] is duplicate in file : %s and %s' % (module_name, tmp_list[module_name], import_from))
            else:
                self.configuration_errors.insert(0, 'A module with a missing name was found in [%s].' % import_from)
        return return_value and not bool(self.configuration_errors)
    
    
    def update_with_module_info(self, modules_dir):
        ConfigurationReader.update_all_modules_with_module_info(modules_dir, self)
