#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import json
import os
import re

from shinken.log import logger
from shinken.objects.item import Item, Items
from shinken.property import StringProp


class Pack(Item):
    id = 1
    my_type = 'pack'
    pack_name: str
    
    properties = Item.properties.copy()
    properties.update({'pack_name': StringProp(fill_brok=['full_status'])})
    
    running_properties = Item.running_properties.copy()
    running_properties.update({'macros': StringProp(default={})})
    
    
    def get_name(self):
        try:
            return self.pack_name
        except AttributeError:
            return 'UnnamedPack'


class Packs(Items):
    name_property = 'pack_name'
    inner_class = Pack
    
    
    # We will dig into the path and load all .pack files
    def load_file(self, path):
        # Now walk for it
        for root, dirs, files in os.walk(path):
            for file in files:
                if re.search(r'\.pack$', file):
                    p = os.path.join(root, file)
                    try:
                        with open(p, 'r', encoding='utf-8') as fd:
                            buffer = fd.read()
                    except IOError as exp:
                        logger.error('Cannot open pack file "%s" for reading: %s' % (p, exp))
                        # ok, skip this one
                        continue
                    self.create_pack(buffer, file[:-5], root)
    
    
    # Create a pack from the string buf, and get a real object from it
    def create_pack(self, buffer, pack_name, pack_folder):
        # type: (str, str, str) -> None
        try:
            raw_pack = json.loads(buffer)
            if 'name' not in raw_pack:
                logger.error("[Pack] no name in the pack '%s'" % pack_name)
                return
            pack = Pack({})
            pack.pack_name = raw_pack['name']
            pack.dir_path = pack_folder
            pack.description = raw_pack.get('description', '')
            pack.macros = raw_pack.get('macros', {})
            pack.templates = raw_pack.get('templates', [pack.pack_name])
            pack.path = raw_pack.get('path', 'various/')
            pack.doc_link = raw_pack.get('doc_link', '')
            pack.services = raw_pack.get('services', {})
            pack.commands = raw_pack.get('commands', [])
            if not pack.path.endswith('/'):
                pack.path += '/'
            # Ok, add it
            self[pack.id] = pack
        except ValueError as exp:
            logger.error("[Pack] error in loading pack file '%s': '%s'" % (pack_name, exp))
            raise ValueError(" - During loading file '%s/%s.pack': '%s' " % (pack_folder, pack_name, exp))
