#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import io
import pickle
from abc import ABCMeta

from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, IO

from ._pickledb_dict import _SAFE_PICKLEABLE_CLASSES_DATABASE


class ShinkenPickleableMeta(ABCMeta):
    
    def __new__(mcs, name, bases, namespace, accept_dots=False):
        cls = super(ShinkenPickleableMeta, mcs).__new__(mcs, name, bases, namespace)
        
        if not getattr(cls, '__module__', None):
            raise TypeError(f'{cls} does not have __module__ defined')
        
        try:
            fullname = cls.__qualname__
        except AttributeError:
            fullname = cls.__name__
        
        if not accept_dots and '.' in fullname:
            raise TypeError(f'{cls}: The ShinkenUnpickler does not support having classes which are not in top-level module.')
        
        return cls


class ShinkenSecurityUnpicklingError(pickle.UnpicklingError):
    pass


class ShinkenUnpickler(pickle.Unpickler):
    __slots__ = ()
    
    
    def __init__(self, file):
        # type: (IO[bytes]) -> None
        super(ShinkenUnpickler, self).__init__(file, encoding='utf8')
    
    
    def find_class(self, module, name):
        # type: (str, str) -> Any
        if module not in _SAFE_PICKLEABLE_CLASSES_DATABASE or name not in _SAFE_PICKLEABLE_CLASSES_DATABASE[module]:
            raise ShinkenSecurityUnpicklingError('Unpickle [ %s.%s ] is forbidden' % (module, name))
        return super().find_class(module, name)
    
    
    @classmethod
    def loads(cls, pickle_string):
        # type: (bytes) -> Any
        with io.BytesIO(pickle_string) as fp:
            return cls(fp).load()
