#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import datetime
import json
import os
import re
import threading
import time
from collections import OrderedDict, namedtuple
from io import StringIO
from weakref import proxy as weak_ref

from shinken.exceptions.business import ShinkenExceptionValueError, ShinkenExceptionKeyError
from shinken.log import logger
from shinken.misc.filter import get_all_my_visible_items, check_user_item_access, get_count_of_my_visible_items
from shinken.misc.perfdata import PerfDatas
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.broker.monitoring_item_manager.helper import get_ack_on_item, get_downtime_on_item, peek_in_set, get_item_context as _get_context, get_item_state as _get_state

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional, List, Collection, Set, Union, Tuple
    from shinken.objects.contact import Contact
    from shinken.objects.broker.broker_host import BrokerHost
    from shinken.objects.broker.broker_service import BrokerService
    from shinken.webui.bottlewebui import MultiDict
    from shinken.misc.monitoring_item_manager.monitoring_item_manager import MonitoringItemManager

# ## Will be populated by the UI with its own value
FRONTEND_DATE_FORMAT = '%Y-%m-%d %H:%M:%S'
DEFAULT_SORT = {'tag': 'host_name', 'value': 'asc'}

DateFilter = namedtuple('DateFilter', ['operator', 'delay'])


class DateFilterOperator:
    NEWER = 'newer'
    OLDER = 'older'


class StaticCache:
    configuration_uuid = -1
    hosts_by_name = {}  # type: Dict[str,BrokerHost]
    hosts_by_uuid = {}  # type: Dict[str,BrokerHost]
    checks_by_uuid = {}  # type: Dict[str,BrokerService]
    checks_by_name = {}  # type: Dict[str,List[BrokerService]]
    
    
    @staticmethod
    def _init_static_cache(item):
        # type: (Union[BrokerHost,BrokerService]) -> None
        item_type = 'host'
        extended_item_type = item_type
        if item.__class__.my_type == 'service':
            item_type = 'check'
            if item.host.got_business_rule:
                extended_item_type = 'check_cluster'
            else:
                extended_item_type = 'check_host'
        elif item.got_business_rule:
            item_type = 'cluster'
            extended_item_type = item_type
        
        list_cache_static = {
            'display_name'        : item.display_name,
            'contact_groups'      : [cg.strip() for cg in getattr(item, 'contact_groups', '') if cg.strip()],
            'contacts'            : [c.get_name() for c in item.contacts],
            'max_attempts'        : item.max_check_attempts,
            'maintenance_period'  : item.maintenance_period.get_name() if item.maintenance_period else '',
            'notification_period' : item.notification_period.get_name() if item.notification_period else '',
            'check_period'        : item.check_period.get_name() if item.check_period else '',
            'notification_options': item.notification_options,
            'type'                : item_type,
            'extended_type'       : extended_item_type,
            'is_unknown'          : False,
        }
        
        if item_type == 'check':
            list_cache_static['host_name'] = item.host.host_name
            list_cache_static['service_description'] = item.service_description
            list_cache_static['name'] = item.service_description
            list_cache_static['realm'] = item.host.realm
            list_cache_static['uuid'] = '%s-%s' % (item.host.uuid, item.uuid)
            list_cache_static['check_uuid'] = '%s-%s' % (item.host.uuid, item.uuid)
            list_cache_static['uuid_parent'] = item.host.uuid
            list_cache_static['host_groups'] = [i.get_name() for i in item.host.hostgroups]
        else:
            if item_type == 'host':
                list_cache_static['address'] = item.address
            
            list_cache_static['host_name'] = item.host_name
            list_cache_static['name'] = item.host_name
            list_cache_static['realm'] = item.realm
            list_cache_static['host_templates'] = [i for i in item.tags if i != 'shinken-host']
            list_cache_static['host_groups'] = [i.get_name() for i in item.hostgroups]
            list_cache_static['uuid'] = item.uuid
            list_cache_static['uuid_parent'] = item.uuid
        item.list_cache_static = list_cache_static
        
        filter_cache_static = {
            'display_name'        : list_cache_static['display_name'].lower(),
            'contact_groups'      : {i.lower() for i in list_cache_static['contact_groups']},
            'contacts'            : {i.lower() for i in list_cache_static['contacts']},
            'maintenance_period'  : list_cache_static['maintenance_period'].lower(),
            'check_period'        : list_cache_static['check_period'].lower(),
            'notification_period' : list_cache_static['notification_period'].lower(),
            'notification_options': set(list_cache_static['notification_options']),
            'type'                : {item_type},
            'extended_type'       : {extended_item_type},
            'name'                : {list_cache_static['name'].lower()},
            'uuid'                : {list_cache_static['uuid'].lower()},
            'uuid_parent'         : {list_cache_static['uuid_parent'].lower()},
            'host_name'           : {list_cache_static['host_name'].lower()},
            'host_name_like'      : list_cache_static['host_name'].lower(),
            'realm'               : {list_cache_static['realm'].lower()},
            'host_groups'         : {i.lower() for i in list_cache_static['host_groups']}
        }
        if list_cache_static['type'] == 'host':
            filter_cache_static['service_description_like'] = ''
            filter_cache_static['service_description'] = {''}
            filter_cache_static['check_uuid'] = {''}
            filter_cache_static['address'] = item.address.lower()
            filter_cache_static['host_templates'] = {i.lower() for i in list_cache_static['host_templates']}
        if list_cache_static['type'] == 'cluster':
            filter_cache_static['service_description_like'] = ''
            filter_cache_static['service_description'] = {''}
            filter_cache_static['check_uuid'] = {''}
            filter_cache_static['address'] = ''
            filter_cache_static['host_templates'] = {i.lower() for i in list_cache_static['host_templates']}
        if list_cache_static['type'] == 'check':
            filter_cache_static['service_description'] = {item.service_description.lower()}
            filter_cache_static['service_description_like'] = item.service_description.lower()
            filter_cache_static['check_uuid'] = {list_cache_static['uuid'].lower()}
            filter_cache_static['address'] = item.host.address.lower()
            filter_cache_static['host_templates'] = {i.lower() for i in item.host.tags if i != 'shinken-host'}
        
        item.filter_cache_static = filter_cache_static
        
        sort_cache_static = {}
        for _tag, _value in filter_cache_static.items():
            sort_cache_static[_tag] = _build_static_sort_key(_value)
        
        item.sort_cache_static = sort_cache_static
    
    
    # force_reset parameter is used in tests to clear previous data from StaticCache
    @classmethod
    def refresh(cls, items, configuration_uuid, force_reset=False):
        # type: (List[Union[BrokerHost, BrokerService]], str, bool) -> None
        
        if cls.configuration_uuid != configuration_uuid or force_reset:
            # Reset cache only on new configuration_uuid
            cls.hosts_by_name = {}
            cls.hosts_by_uuid = {}
            cls.checks_by_name = {}
            cls.checks_by_uuid = {}
            cls.configuration_uuid = configuration_uuid
        
        for i in items:
            cls._init_static_cache(i)
            item_static_cache = getattr(i, 'list_cache_static')
            name = item_static_cache['name'].lower()
            uuid = item_static_cache['uuid'].lower()
            if item_static_cache['type'] == 'check':
                cls.checks_by_uuid[uuid] = i
                if name in cls.checks_by_name:
                    cls.checks_by_name[name].append(weak_ref(i))
                else:
                    cls.checks_by_name[name] = [weak_ref(i)]
            else:
                cls.hosts_by_name[name] = weak_ref(i)
                cls.hosts_by_uuid[uuid] = weak_ref(i)
    
    
    @classmethod
    def get_host_by_uuid(cls, uuid, user=None):
        # type: (str, Optional[Contact]) -> Optional[BrokerHost]
        if user and not check_user_item_access(user, uuid):
            return None
        if uuid in cls.hosts_by_uuid:
            return cls.hosts_by_uuid[uuid]
        return None
    
    
    @classmethod
    def get_check_by_uuid(cls, uuid, user=None):
        # type: (str, Optional[Contact]) -> Optional[BrokerService]
        if user and not check_user_item_access(user, uuid):
            return None
        if uuid in cls.checks_by_uuid:
            return cls.checks_by_uuid[uuid]
        return None
    
    
    @classmethod
    def get_host_by_name(cls, name, user=None):
        # type: (str, Optional[Contact]) -> Optional[BrokerHost]
        if name in cls.hosts_by_name:
            host = cls.hosts_by_name[name]
            if not user or check_user_item_access(user, host.get_uuid()):
                return host
        return None
    
    
    @classmethod
    def get_check_by_name(cls, host, name, user=None):
        # type: (BrokerHost, str, Optional[Contact]) -> Optional[BrokerService]
        if not host or not name:
            return None
        host_uuid = host.get_uuid()
        if user and not check_user_item_access(user, host_uuid):
            return None
        if name in cls.checks_by_name:
            for s in cls.checks_by_name[name]:
                if s.host.uuid == host_uuid:
                    return s
        return None


def _build_static_sort_key(_value):
    return [int(s) if s.isdigit() else s.lower() for s in re.split(r'(\d+)', _value if isinstance(_value, str) else str(_value))]


def _build_var_sort_key(item, tag):
    _value = ''
    if tag == 'problem_has_been_inherited_acknowledged':
        _value = 0
        if item.is_in_inherited_acknowledged():
            if item.__class__.my_type == 'service':
                _value = 1
            else:
                _value = 2
    elif tag == 'next_check':
        if item.got_business_rule:
            _value = -1
        else:
            _value = item.next_chk
    elif tag == 'last_check':
        if item.got_business_rule:
            _value = -1
        else:
            _value = item.last_chk
    elif tag == 'business_impact':
        _value = item.business_impact
    elif tag == 'status':
        _value = str(_get_state(item))
    elif tag == 'context':
        _value = _get_context(item)[1]
    elif tag == 'status_since':
        _value = int(item.last_state_change)
    elif tag == 'last_hard_state_change':
        _value = int(item.last_hard_state_change if item.state_type == 'HARD' else 0)
    elif tag == 'is_status_confirmed':
        _value = (item.state_type == 'HARD')
    elif tag == 'attempts':
        _value = item.attempt
    elif tag == 'is_root_problem':
        _value = item.is_problem
    elif tag == 'output':
        _value = item.output
    elif tag == 'long_output':
        _value = item.long_output
    elif tag == 'perf_data':
        _value = item.perf_data
    elif tag == 'got_business_rule':
        _value = item.got_business_rule
    elif tag == 'problem_has_been_acknowledged':
        _value = bool(item.acknowledgement and not item.is_in_inherited_acknowledged())
    elif tag == 'in_flapping':
        _value = item.is_flapping
    elif tag == 'inherited_flapping':
        _value = item.inherited_flapping
    elif tag == 'in_partial_flapping':
        _value = item.is_partial_flapping
    elif tag == 'thresholds_display':
        _value = item.thresholds_display
    
    return [int(s) if s.isdigit() else s.lower() for s in re.split(r'(\d+)', _value.decode('utf-8', 'ignore') if isinstance(_value, bytes) else str(_value))]


def sort_list_api_part_all(sorts, items):
    if not sorts:
        sorts = [DEFAULT_SORT]
    for sort in sorts:
        tag = sort['tag']
        reverse = (sort['value'] == 'desc')
        items.sort(key=lambda item: item.sort_cache_static[tag] if tag in item.sort_cache_static else _build_var_sort_key(item, tag), reverse=reverse)


def list_all_values(items):
    all_values = {
        'host_templates'     : set(),
        'realm'              : set(),
        'host_groups'        : set(),
        'contact_groups'     : set(),
        'contacts'           : set(),
        'notification_period': set(),
        'maintenance_period' : set(),
        'check_period'       : set(),
    }
    update_realm = all_values['realm'].add
    update_host_template = all_values['host_templates'].update
    update_host_group = all_values['host_groups'].update
    update_contact_group = all_values['contact_groups'].update
    update_contacts = all_values['contacts'].update
    update_notification_period = all_values['notification_period'].add
    update_maintenance_period = all_values['maintenance_period'].add
    update_check_period = all_values['check_period'].add
    
    for item in items:
        list_cache_static = item.list_cache_static
        update_contact_group(list_cache_static['contact_groups'])
        update_contacts(list_cache_static['contacts'])
        update_notification_period(list_cache_static['notification_period'])
        update_maintenance_period(list_cache_static['maintenance_period'])
        update_check_period(list_cache_static['check_period'])
        if not list_cache_static['type'] == 'check':
            update_host_template(list_cache_static['host_templates'])
            update_host_group(list_cache_static['host_groups'])
            update_realm(list_cache_static['realm'])
    
    if all_values['maintenance_period']:
        all_values['maintenance_period'].remove('')
    
    for (_type, values) in all_values.items():
        all_values[_type] = list(values)
    return all_values


class _FilterPagination:
    def __init__(self):
        self.pages = []
        self.start_page = 0
        self.counter = 0
        self.current_page = 0
        self.last = None
    
    
    def __str__(self):
        return '_FilterPagination(pages=[%s], start_page=[%s], counter=[%s], current_page=[%s], last=[%s])' % (self.pages, self.start_page, self.counter, self.current_page, self.last)
    
    
    def __repr__(self):
        return self.__str__()


class _CheckSet:
    def __init__(self):
        self.result_set = {}
        self.nb = 0
    
    
    def add(self, host_name, elements):
        # type: (str, list) -> None
        self.result_set.update({host_name: elements})
        self.nb += len(elements)


class FilterItem:
    FILTER_TYPE = {
        'display_name'            : 'string',
        'thresholds_display'      : 'string',
        'business_impact'         : 'int',
        'contact_groups'          : 'array',
        'contacts'                : 'array',
        'status'                  : 'int',
        'context'                 : 'array',
        'status_since'            : 'date',
        'last_check'              : 'date',
        'last_hard_state_change'  : 'date',
        'is_status_confirmed'     : 'bool',
        'attempts'                : 'string',
        'next_check'              : 'date',
        'is_root_problem'         : 'bool',
        'output'                  : 'string',
        'long_output'             : 'string',
        'perf_data'               : 'string',
        'notification_period'     : 'string_array',
        'maintenance_period'      : 'string_array',
        'check_period'            : 'string_array',
        'notification_options'    : 'array',
        'type'                    : 'array',
        'extended_type'           : 'array',
        'host_name'               : 'array',
        'host_name_like'          : 'string_array',
        'service_description'     : 'array',
        'service_description_like': 'string_array',
        'realm'                   : 'array',
        'address'                 : 'string',
        'host_templates'          : 'array',
        'host_groups'             : 'array',
        'uuid_parent'             : 'array',
        'check_uuid'              : 'array',
        'uuid'                    : 'array',
        'name'                    : 'array'
    }
    
    
    def __init__(self, monitoring_item_manager, user):
        # type: (MonitoringItemManager, Optional[Contact]) -> None
        self.page = None
        self.page_size = None
        self.sorts = None
        self.full_sort_for_log = None
        self.configuration_id_in_client = None
        self.filters = None
        self.full_filter_for_log = None
        self.flattened_view = True
        self.checks_in_tree = False
        self.monitoring_item_manager = monitoring_item_manager
        self.user = user
        self.ALLOWED_OUTPUT = None
        self.compute_not_found = True
        
        if not self.monitoring_item_manager:
            raise (NotImplementedError('monitoring_item_manager not available'))
    
    
    @classmethod
    def _translate_filter_value(cls, filter_tag, filter_value, only_allowed_values=None):
        filter_value = [v.lower() for v in filter_value]
        
        if filter_tag == 'extended_type' and 'check' in filter_value:
            filter_value = [val for val in filter_value if val != 'check']
            if 'check_host' not in filter_value and (not only_allowed_values or filter_tag not in only_allowed_values or 'check_host' in only_allowed_values[filter_tag]):
                filter_value.append('check_host')
            if 'check_cluster' not in filter_value and (not only_allowed_values or filter_tag not in only_allowed_values or 'check_cluster' in only_allowed_values[filter_tag]):
                filter_value.append('check_cluster')
        
        valid_values = []
        if only_allowed_values and filter_tag in only_allowed_values:
            valid_values = only_allowed_values[filter_tag]
        elif filter_tag == 'status':
            valid_values = ['0', '1', '2', '3']
        elif filter_tag == 'context':
            valid_values = ['nothing', 'acknowledged', 'partial-acknowledged', 'inherited-acknowledged', 'downtime', 'partial-downtime', 'inherited-downtime', 'flapping', 'partial-flapping', 'disabled']
        elif filter_tag == 'is_status_confirmed':
            valid_values = ['true', 'false']
        elif filter_tag == 'extended_type':
            valid_values = ['host', 'cluster', 'check', 'check_host', 'check_cluster']
        elif filter_tag == 'type':
            valid_values = ['host', 'cluster', 'check']
        elif filter_tag == 'business_impact':
            valid_values = ['0', '1', '2', '3', '4', '5']
        
        if valid_values:
            all_values = filter_value
            if not isinstance(all_values, list):
                all_values = [all_values]
            wrong_values = [val for val in all_values if val not in valid_values]
            if wrong_values:
                raise ShinkenExceptionValueError(text='wrong value%s %s' % ('s' if len(wrong_values) > 1 else '', wrong_values))
        
        compare_type = cls.FILTER_TYPE[filter_tag]
        if compare_type == 'int':
            filter_value = {int(v) for v in filter_value}
        elif compare_type == 'bool':
            for v in filter_value:
                if v not in ['true', 'false']:
                    raise ShinkenExceptionValueError(text='value [ %s ] is not a boolean' % v)
            filter_value = {('true' == v) for v in filter_value}
        elif compare_type == 'array':
            filter_value = set(filter_value)
        elif compare_type == 'date':
            val = filter_value[0]
            if val.startswith('latest|') or val.startswith('old|') or val.startswith('in_more_than|') or val.startswith('in_less_than|') or val.startswith('in-more-than|') or val.startswith('in-less-than|') or val.startswith(
                    'later-than|') or val.startswith('older-than|'):
                operator, delay = val.split('|')
                try:
                    delay = int(delay)
                except ValueError as e:
                    raise ShinkenExceptionValueError(text='[ %s ] => %s' % (operator, str(e)))
                operator = DateFilterOperator.NEWER if operator == 'latest' or operator == 'in_more_than' or operator == 'in-more-than' or operator == 'later-than' else DateFilterOperator.OLDER
                filter_value = [DateFilter(operator, delay)]
            else:
                raise ShinkenExceptionKeyError(text='unknown date constraint [ %s ]' % val)
        return filter_value
    
    
    @classmethod
    def parse_filter(cls, get_filters, allowed_filter=None, only_allowed_values=None, default_if_empty=None, filter_item_instance=None):
        filters = []
        cpt = - 1
        for get_filter in get_filters:
            _filter = {}
            cpt = cpt + 1
            for search_type in get_filter.split('~'):
                split = search_type.split(':', 1)
                filter_tag = split[0].lower()
                filter_tag_name = filter_tag
                if allowed_filter:
                    if filter_tag not in allowed_filter:
                        error_text = 'filtering[%s]: invalid field name [ %s ]' % (cpt, filter_tag_name)
                        if filter_item_instance and hasattr(filter_item_instance, 'ALLOWED_FILTER'):
                            error_text = 'filtering[%s]: field name [ %s ] not allowed in this route ' % (cpt, filter_tag_name) if filter_tag in getattr(filter_item_instance, 'ALLOWED_FILTER') else error_text
                        raise ShinkenExceptionKeyError(text=error_text)
                    filter_tag = allowed_filter[filter_tag]
                if len(split) > 1:
                    filter_value = split[1].lower().split('^^')
                else:
                    filter_value = None
                if not filter_value and default_if_empty and filter_tag in default_if_empty:
                    filter_value = default_if_empty[filter_tag]
                if not filter_value or '' in filter_value:
                    raise ShinkenExceptionValueError(text='filtering[%s]: missing value for field [ %s ]' % (cpt, filter_tag_name))
                
                try:
                    filter_value = cls._translate_filter_value(filter_tag, filter_value, only_allowed_values)
                except (ValueError, ShinkenExceptionValueError) as e:
                    raise ShinkenExceptionValueError(text='filtering[%s]: field [ %s ] => %s' % (cpt, filter_tag_name, str(e)))
                except ShinkenExceptionKeyError as e:
                    raise ShinkenExceptionKeyError(text='filtering[%s]: field [ %s ] %s' % (cpt, filter_tag_name, str(e)))
                
                if filter_tag in _filter and _filter[filter_tag] != filter_value:
                    raise ShinkenExceptionKeyError(text='filtering[%s]: value for field [ %s ] is already set, can not redefine to [ %s ]' % (cpt, filter_tag_name, split[1].lower()))
                _filter[filter_tag] = filter_value
            if default_if_empty:
                # extra_default_filter = {k: self._translate_filter_value(k, default_if_empty[k]) for k in default_if_empty if k not in _filter}
                extra_default_filter = {}
                for k in default_if_empty:
                    if k not in _filter:
                        extra_default_filter.update({k: cls._translate_filter_value(k, default_if_empty[k])})
                if extra_default_filter:
                    _filter.update(extra_default_filter)
            filters.append(_filter)
        return filters
    
    
    @classmethod
    def filter_root_problems(cls, filters, items):
        items = cls.filter_elements_with_search(items, [{'is_root_problem': {'true'}}])
        # remove the occurrence of is_root_problem in filter
        for _filter in filters[:]:
            _filter.pop('is_root_problem', None)
        return items
    
    
    @classmethod
    def filter_elements_with_search(cls, items, _filters, counters=None):
        new_elements = []
        # counter on found elements, with this field matching :
        # 0 hosts found,
        # 1 clusters found
        # 2 checks found
        if not isinstance(counters, list) or len(counters) != 3:
            counters = [0, 0, 0]
        
        for item in items:
            add_element = False
            
            if item.__class__.my_type == 'host':
                if item.got_business_rule:
                    type_found = 1
                else:
                    type_found = 0
            else:
                type_found = 2
            
            for _filter in _filters:
                if cls._apply_filter(_filter, item):
                    add_element = True
                    break
            
            if add_element:
                counters[type_found] = counters[type_found] + 1
                new_elements.append(item)
        
        return new_elements
    
    
    # Yes this function is ugly but do not refactor without check performance first
    @staticmethod
    def get_value_for_filter(item, filter_name):
        if filter_name in item.filter_cache_static:
            return item.filter_cache_static[filter_name]
        
        if filter_name == 'business_impact':
            return item.business_impact
        if filter_name == 'status':
            return _get_state(item)
        if filter_name == 'context':
            return {_get_context(item)[1].lower()}
        if filter_name == 'status_since':
            return int(item.last_state_change)
        if filter_name == 'last_hard_state_change':
            return int(item.last_hard_state_change if item.state_type == 'HARD' else 0)
        if filter_name == 'last_check':
            return int(item.last_chk)
        if filter_name == 'next_check':
            return item.next_chk
        if filter_name == 'is_status_confirmed':
            return item.state_type == 'HARD'
        if filter_name == 'attempts':
            return '%s/%s' % (item.attempt, item.max_check_attempts)
        if filter_name == 'is_root_problem':
            return item.is_problem
        if filter_name == 'output':
            return item.output.lower()
        if filter_name == 'long_output':
            return item.long_output.lower()
        if filter_name == 'perf_data':
            return item.perf_data.lower()
        if filter_name == 'thresholds_display':
            return item.thresholds_display.lower()
    
    
    @classmethod
    def _apply_filter(cls, _filter, item):
        # Yes this function is ugly but do not refactor without check performance first
        
        match_filter = True
        for (filter_tag, filter_values) in _filter.items():
            value = cls.get_value_for_filter(item, filter_tag)
            compare_type = cls.FILTER_TYPE[filter_tag]
            # 0 is a value
            if value != 0 and not value:
                match_filter = False
                break
            
            _match_tag = False
            if compare_type == 'bool' or compare_type == 'int':
                _match_tag = value in filter_values
            elif compare_type == 'array':
                _match_tag = bool(filter_values & value)
            elif compare_type == 'string':
                _match_tag = filter_values[0] in value
            elif compare_type == 'string_array':
                _match_tag = next((True for v in filter_values if v in value), False)
            elif compare_type == 'date':
                filter_value = filter_values[0]
                # SEF-10916 For the "next_check" property, the visu adds a "-" to the delay because it's in the future,
                # so we force the negativity to be compatible with modules that use filters (ex: broker-module-livedata v2)
                if isinstance(filter_value, DateFilter):
                    if filter_tag == 'next_check':
                        delay = -abs(filter_value.delay)
                    else:
                        delay = filter_value.delay
                    
                    if filter_value.operator == DateFilterOperator.NEWER:
                        _match_tag = delay >= time.time() - value
                    else:
                        _match_tag = delay <= time.time() - value
                else:
                    value = datetime.datetime.fromtimestamp(value).strftime(FRONTEND_DATE_FORMAT)
                    _match_tag = filter_value in value
            else:
                logger.warning('unknown filter type : %s' % compare_type)
                _match_tag = False
            
            if not _match_tag:
                match_filter = False
                break
        return match_filter
    
    
    @staticmethod
    def api_list_element_dict(item):
        # type: (Union[BrokerHost,BrokerService]) -> Dict
        _new_cache_var = {}
        list_cache_var = getattr(item, 'list_cache_var', {})
        if not list_cache_var:
            _context, ___ = _get_context(item)
            _new_cache_var = {
                'business_impact'       : item.business_impact,
                'status'                : _get_state(item),
                'context'               : _context,
                'status_since'          : int(item.last_state_change),
                'last_hard_state_change': int(item.last_hard_state_change if item.state_type == 'HARD' else 0),
                'is_status_confirmed'   : (item.state_type == 'HARD'),
                'attempts'              : item.attempt,
                'max_attempts'          : item.max_check_attempts,
                'is_root_problem'       : item.is_problem,
                'output'                : item.output,
                'long_output'           : item.long_output,
                'perf_data'             : item.perf_data,
                'smart_perf_data'       : PerfDatas(item.perf_data).as_list(),
                'got_business_rule'     : item.got_business_rule,
                'in_flapping'           : item.is_flapping,
                'inherited_flapping'    : item.inherited_flapping,
                'in_partial_flapping'   : item.is_partial_flapping,
                'notes_multi_url'       : item.notes_multi_url,
                'notes_url'             : item.notes_url,
                'active_checks_enabled' : item.active_checks_enabled,
                'thresholds_display'    : item.thresholds_display
            }
            get_ack_on_item(item, _new_cache_var)
            get_downtime_on_item(item, _new_cache_var)
            
            if not item.got_business_rule:
                _new_cache_var['next_check'] = item.next_chk
                _new_cache_var['last_check'] = item.last_chk
            
            _new_cache_var.update(getattr(item, 'list_cache_static', {}))
            item.list_cache_var = _new_cache_var
            return _new_cache_var
        
        return list_cache_var
    
    
    def set_range(self, range_param):
        # type: (str) -> None
        start_time = time.time()
        self.page = None
        self.page_size = None
        # range_param = self.app.request.GET.get('range', '').strip()
        if isinstance(range_param, bytes):
            range_param = range_param.decode('UTF-8')
        range_param = range_param.strip()
        if range_param:
            range_split = range_param.split('~')
            if len(range_split) != 2:
                raise ShinkenExceptionValueError(text='Wrong format for pagination parameters, expected:[name:integer~name:integer], got:[ %s ]' % range_param)
            
            param_start = range_split[0].split(':')
            param_nb = range_split[1].split(':')
            
            if len(param_start) != 2:
                raise ShinkenExceptionKeyError(text='Missing page number parameter')
            if len(param_nb) != 2:
                raise ShinkenExceptionKeyError(text='Missing page size parameter')
            
            try:
                self.page = int(param_start[1])
                if self.page < 0:
                    raise ShinkenExceptionValueError()
            except:
                raise ShinkenExceptionValueError(text='Wrong value:[ %s ] for page number parameter' % param_start[1])
            
            try:
                self.page_size = int(param_nb[1])
                if self.page_size <= 0:
                    raise ShinkenExceptionValueError()
            except:
                raise ShinkenExceptionValueError(text='Wrong value:[ %s ] for page size parameter' % param_nb[1])
        logger.log_perf(start_time, 'get_list_api_part_all', 'page %s page_size %s' % (self.page, self.page_size))
    
    
    def set_sort(self, sort_param, allowed_sort=None):
        # type: (str, Optional[dict]) -> None
        start_time = time.time()
        self.sorts = []
        # get_sorts = self.app.request.GET.get('sort', '').strip()
        if sort_param is None:
            return
        get_sorts = sort_param.strip()
        if isinstance(get_sorts, bytes):
            get_sorts = get_sorts.decode('UTF-8')
        if get_sorts:
            for get_sort in get_sorts.split('~'):
                split = get_sort.split(':', 1)
                sort_tag = split[0].lower()
                if len(split) >= 2:
                    sort_value = split[1].lower()
                else:
                    sort_value = 'asc'
                if allowed_sort and sort_tag not in allowed_sort:
                    raise ShinkenExceptionKeyError(text='sort: invalid field name [ %s ]' % sort_tag)
                if sort_value not in ['asc', 'desc']:
                    raise ShinkenExceptionValueError(text='sort: invalid sort direction [ %s ] for field [ %s ]' % (sort_value, sort_tag))
                self.sorts.append({'tag': allowed_sort[sort_tag], 'value': sort_value})
        self.full_sort_for_log = get_sorts[:]
        logger.log_perf(start_time, 'get_list_api_part_all', 'sorts %s' % self.sorts)
    
    
    def set_flattened_mode(self, flattened_mode, checks_in_tree=False):
        # type: (bool, Optional[bool]) -> None
        self.flattened_view = bool(flattened_mode)
        self.checks_in_tree = bool(checks_in_tree)
    
    
    def get_flattened_mode(self):
        return self.flattened_view, self.checks_in_tree
    
    
    def set_filters(self, filters_json_str, allowed_filters=None, only_allowed_values=None, default_if_empty=None):
        # type: (str, Optional[dict], Optional[dict], Optional[dict]) -> None
        # allowed_filters = { "public_field_name": "internal_field_name", ... }
        # only_allowed_values = { "internal_field_nameY1": ["valueY1", "valueY2", ...], ...}
        # default_if_empty = { "internal_field_nameX1": ["valueX1", "valueX2", ...], ...}
        #
        # allowed_filters translates publicly exposed field names to internal field name, and limit the set of usable filters
        # only_allowed_values allows to check for user provided filtering values and silently ignore the ones not in its set
        # default_if_empty defines default values for filtering tags not provided by user
        start_time = time.time()
        self.filters = []
        self.full_filter_for_log = ''
        
        get_filters = []
        if filters_json_str:
            cnt = 0
            filters_json = filters_json_str.strip()
            filters_dict = json.loads(filters_json)
            get_filter = filters_dict.get('filter%s' % cnt, '')
            while get_filter:
                get_filters.append(get_filter.strip())
                cnt += 1
                get_filter = filters_dict.get('filter%s' % cnt, '')
            
            self.full_filter_for_log = get_filters[:]
        
        if get_filters:
            self.filters = self.parse_filter(get_filters, allowed_filters, only_allowed_values, default_if_empty, filter_item_instance=self)
        elif default_if_empty:
            # self.filters = [{k: self._translate_filter_value(k, default_if_empty[k]) for k in default_if_empty}]
            self.filters = {}
            for k in default_if_empty:
                self.filters.update({k: self._translate_filter_value(k, default_if_empty[k])})
            self.filters = [self.filters]
        logger.log_perf(start_time, 'get_list_api_part_all', 'filters %s' % self.filters)
    
    
    def set_output_fields(self, fields):
        # type: (dict) -> None
        self.ALLOWED_OUTPUT = fields
    
    
    def get_output_fields(self):
        # type: () -> Optional[dict]
        return self.ALLOWED_OUTPUT
    
    
    def add_output_fields(self, fields):
        # type: (dict) -> None
        if self.ALLOWED_OUTPUT:
            self.ALLOWED_OUTPUT.update(fields)
        else:
            self.ALLOWED_OUTPUT = fields
    
    
    def set_compute_not_found(self, mode):
        # type: (bool) -> None
        self.compute_not_found = mode
    
    
    def _get_elements(self, initial_set, check_element, page_counters, kept_checks, log_name, counters=None):
        # type: (OrderedDict, OrderedDict, _FilterPagination, Optional[_CheckSet], str, Optional[List[int,int]]) -> list
        start_time = time.time()
        initial_set = list(initial_set.values())
        if (self.sorts and self.sorts != [DEFAULT_SORT]) or self.filters:
            sort_list_api_part_all(self.sorts, initial_set)
        logger.log_perf(start_time, 'get_list_api_part_all', 'sort %s' % log_name)
        
        if not isinstance(counters, list) or len(counters) != 3:
            counters = [0, 0, 0]
        result_set = []
        first = None
        last = None
        if self.page is not None:
            if first is None and self.page == page_counters.current_page:
                first = len(result_set)
            if last is None and self.page < page_counters.current_page:
                last = len(result_set)
        for host in initial_set:
            result_set.append(host)
            page_counters.counter += 1
            if self.page is None or self.page == page_counters.current_page:
                if host.got_business_rule:
                    counters[1] = counters[1] + 1
                else:
                    counters[0] = counters[0] + 1
            check_for_host = check_element.get(host.host_name, [])
            if check_for_host:
                if self.page is None or self.page == page_counters.current_page:
                    counters[2] = counters[2] + len(check_for_host)
                    if self.sorts and self.sorts != [DEFAULT_SORT]:
                        sort_list_api_part_all(self.sorts, check_for_host)
                    if kept_checks is not None:
                        kept_checks.add(host.host_name, check_for_host)
                    else:
                        result_set.extend(check_for_host)
                        page_counters.counter += len(check_for_host)
            if self.page_size and page_counters.counter >= self.page_size:
                page_counters.pages.append({'start': page_counters.start_page, 'end': page_counters.start_page + page_counters.counter})
                page_counters.start_page = page_counters.start_page + page_counters.counter
                page_counters.counter = 0
                page_counters.current_page += 1
                if self.page is not None:
                    if first is None and self.page == page_counters.current_page:
                        first = len(result_set)
                    if last is None and self.page < page_counters.current_page:
                        last = len(result_set)
        
        logger.log_perf(start_time, 'get_list_api_part_all', 'sort %s checks' % log_name)
        start_time = time.time()
        if first is not None or last is not None:
            result_set = result_set[first:last]
            page_counters.last = last
        elif self.page is not None:
            result_set = []
        logger.log_perf(start_time, 'get_list_api_part_all', 'Result pagination %s checks' % log_name)
        return result_set
    
    
    def _jsonify_elements(self, host_elements, check_elements, output_string, start_header=None, with_counter=False, item_name=None):
        # type: (list, Optional[dict], StringIO, Optional[str], Optional[bool], Optional[str]) -> None
        try:
            add_counter = with_counter and (not self.ALLOWED_OUTPUT or 'nb_element_filter' in self.ALLOWED_OUTPUT)
            if not item_name:
                item_name = 'element'
            
            if start_header:
                output_string.write('%s' % start_header)
            
            if add_counter:
                output_string.write('"nb_%s":%d,' % (item_name, len(host_elements)))
            output_string.write('"%s":[' % item_name)
            
            for output_element in host_elements:
                output_element = self.api_list_element_dict(output_element)
                host_name = output_element['host_name']
                if self.ALLOWED_OUTPUT:
                    # output_element = {self.ALLOWED_OUTPUT[k]: output_element[k] for k in self.ALLOWED_OUTPUT if k in output_element}
                    tmp_output = {}
                    for k in self.ALLOWED_OUTPUT:
                        if k in output_element:
                            tmp_output.update({self.ALLOWED_OUTPUT[k]: output_element[k]})
                    output_element = tmp_output
                output_string.write('%s' % json.dumps(output_element, separators=(',', ':')))
                
                if check_elements is not None:
                    output_string.seek(0, os.SEEK_END)
                    output_string.seek(output_string.tell() - 1, os.SEEK_SET)
                    if add_counter:
                        output_string.write(',"nb_check":%d' % len(check_elements.get(host_name, [])))
                    output_string.write(',"checks":[')
                    
                    if host_name in check_elements:
                        checks = check_elements[host_name]
                        for output_check in checks:
                            output_check = self.api_list_element_dict(output_check)
                            if self.ALLOWED_OUTPUT:
                                # output_check = {self.ALLOWED_OUTPUT[k]: output_check[k] for k in self.ALLOWED_OUTPUT if k in output_check}
                                tmp_output = {}
                                for k in self.ALLOWED_OUTPUT:
                                    if k in ['uuid_parent', 'host_name']:
                                        continue
                                    if k in output_check:
                                        tmp_output.update({self.ALLOWED_OUTPUT[k]: output_check[k]})
                                output_check = tmp_output
                            output_string.write('%s,' % json.dumps(output_check, separators=(',', ':')))
                        if checks:
                            output_string.seek(0, os.SEEK_END)
                            output_string.seek(output_string.tell() - 1, os.SEEK_SET)
                    output_string.write(']}')
                
                output_string.write(',')
            if host_elements:
                output_string.seek(0, os.SEEK_END)
                output_string.seek(output_string.tell() - 1, os.SEEK_SET)
            output_string.write(']')
        except:
            # TODO: remove these logs ?
            logger.print_stack()
            output_string.seek(0, os.SEEK_END)
            logger.error('Could not build JSON document from element list. Buffer size was : [%d]' % output_string.tell())
            
            # Free memory if we could not complete
            output_string.close()
            raise
    
    
    def get_list_elements(self, jsonify_elements=True, specific_module_pagination=None):
        # type: (bool, Optional[_FilterPagination]) -> Union[OrderedDict,str]
        full_start_time = time.time()
        
        start_time = time.time()
        user = self.user
        
        thread_id = '%s' % threading.current_thread().name
        if user:
            log_prefix = '%-22s ] [ user= %s ] [ get_data_visualisation_list' % (thread_id, getattr(user, 'uuid', 0))
        else:
            log_prefix = '%-22s ] [ get_data_visualisation_list' % thread_id
        
        items = self.monitoring_item_manager.get_all_hosts_and_services()
        full_nb_items_in_broker = len(items)
        logger.log_perf(start_time, log_prefix, 'get_all_hosts_and_services len [%s]' % full_nb_items_in_broker)
        
        filter_skip, items, nb_item_found = self.make_filter_skip(items)
        
        # Filter with authorization
        if user:
            items_count = get_count_of_my_visible_items(user, self.monitoring_item_manager)
            start_time = time.time()
            if not filter_skip:
                items = get_all_my_visible_items(user, self.monitoring_item_manager)[:]
            logger.log_perf(start_time, log_prefix, 'only_related_to')
        else:
            items_count = self.monitoring_item_manager.get_count_items()
            if not filter_skip:
                items = items[:]
        
        elements = []
        
        start_time = time.time()
        hosts = {}
        for item in items:
            if item.__class__.my_type == 'host':
                hosts[item.host_name] = item
        logger.log_perf(start_time, log_prefix, 'build host dict')
        
        # Filter with filter parameters
        start_time = time.time()
        if self.filters:
            filters_contain_root_problems = 'true' in self.filters[0].get('is_root_problem', {'false'})
            if filters_contain_root_problems:
                items = self.filter_root_problems(self.filters, items)
            
            if filter_skip:
                nb_hosts_found = nb_item_found[0]
                nb_clusters_found = nb_item_found[1]
                nb_checks_found = nb_item_found[2]
            else:
                # normal filters
                nb_found = [0, 0, 0]
                items = self.filter_elements_with_search(items, self.filters, nb_found)
                nb_hosts_found = nb_found[0]
                nb_clusters_found = nb_found[1]
                nb_checks_found = nb_found[2]
        else:
            nb_hosts_found = items_count['hosts']
            nb_clusters_found = items_count['clusters']
            nb_checks_found = items_count['checks']
        logger.log_perf(start_time, log_prefix, 'filter_elements_with_search ')
        
        nb_items_post_filter = len(items)
        
        host_kept_checks = _CheckSet()
        cluster_kept_checks = _CheckSet()
        elements_hosts = []
        elements_clusters = []
        nb_hosts_in_page = 0
        nb_clusters_in_page = 0
        nb_checks_in_page = 0
        
        if self.flattened_view:
            start_time = time.time()
            if (self.sorts and self.sorts != [DEFAULT_SORT]) or self.filters:
                sort_list_api_part_all(self.sorts, items)
            logger.log_perf(start_time, log_prefix, 'sort all (flattened view)')
            
            start_time = time.time()
            _pages = []
            if self.page_size:
                if specific_module_pagination:
                    _pages = specific_module_pagination.pages
                else:
                    _pages = [{'start': start_index, 'end': min(start_index + self.page_size, len(items))} for start_index in range(0, len(items), self.page_size)]
            
            elements = items
            nb_element_return = len(elements)
            
            # Result pagination
            if self.page is not None and _pages:
                if specific_module_pagination:
                    elements = elements[_pages[0]['start']:_pages[0]['end']]
                elif len(_pages) > self.page:
                    elements = elements[_pages[self.page]['start']:_pages[self.page]['end']]
                else:
                    elements = []
                nb_element_return = len(elements)
                for page_item in elements:
                    if page_item.__class__.my_type == 'host':
                        if page_item.got_business_rule:
                            nb_clusters_in_page = nb_clusters_in_page + 1
                        else:
                            nb_hosts_in_page = nb_hosts_in_page + 1
                    else:
                        nb_checks_in_page = nb_checks_in_page + 1
                logger.log_perf(start_time, log_prefix, 'Result pagination')
        else:
            start_time = time.time()
            host_element = OrderedDict()
            cluster_element = OrderedDict()
            check_element = OrderedDict()
            cluster_check_element = OrderedDict()
            
            if self.checks_in_tree:
                for i in items:
                    if i.__class__.my_type == 'host':
                        if i.got_business_rule:
                            cluster_element[i.host_name] = i
                        else:
                            host_element[i.host_name] = i
                    elif i.__class__.my_type == 'service':
                        check_host_name = i.host.host_name
                        if i.host.got_business_rule:
                            if check_host_name not in cluster_check_element:
                                cluster_check_element[check_host_name] = []
                            cluster_check_element[check_host_name].append(i)
                        else:
                            if check_host_name not in check_element:
                                check_element[check_host_name] = []
                            check_element[check_host_name].append(i)
                
                for check_host_name in check_element:
                    # If the host of the check isn't in host_element we add it
                    # We don't want orphan check
                    if check_host_name not in host_element:
                        host = hosts[check_host_name]
                        host_element[host.host_name] = host
                for check_host_name in cluster_check_element:
                    if check_host_name not in cluster_element:
                        host = hosts[check_host_name]
                        cluster_element[host.host_name] = host
            else:
                host_element = OrderedDict((i.host_name, i) for i in items if (i.__class__.my_type != 'service'))
                for i in items:
                    if i.__class__.my_type == 'service':
                        check_host_name = i.host.host_name
                        if check_host_name not in check_element:
                            check_element[check_host_name] = []
                        check_element[check_host_name].append(i)
                        
                        # If the host of the check isn't in host_element we add it
                        # We don't want orphan check
                        if check_host_name not in host_element:
                            host = hosts[check_host_name]
                            host_element[host.host_name] = host
            
            logger.log_perf(start_time, log_prefix, 'append host')
            
            page_counters = specific_module_pagination if specific_module_pagination else _FilterPagination()
            start_time = time.time()
            elements_in_page_counters = [0, 0, 0]
            
            if self.checks_in_tree:
                host_kept_checks = _CheckSet()
                cluster_kept_checks = _CheckSet()
                
                elements_clusters = self._get_elements(cluster_element, cluster_check_element, page_counters, cluster_kept_checks, 'clusters', elements_in_page_counters)
                elements_hosts = self._get_elements(host_element, check_element, page_counters, host_kept_checks, 'hosts', elements_in_page_counters)
                
                nb_hosts_found = len(elements_hosts)
                nb_clusters_found = len(elements_clusters)
                nb_checks_found = cluster_kept_checks.nb + host_kept_checks.nb
                nb_element_return = len(elements_clusters) + len(elements_hosts) + cluster_kept_checks.nb + host_kept_checks.nb
            else:
                elements = self._get_elements(host_element, check_element, page_counters, None, 'host', elements_in_page_counters)
                nb_element_return = len(elements)
            
            nb_hosts_in_page = elements_in_page_counters[0]
            nb_clusters_in_page = elements_in_page_counters[1]
            nb_checks_in_page = elements_in_page_counters[2]
            
            if page_counters.counter != 0 and not specific_module_pagination:
                page_counters.pages.append({'start': page_counters.start_page, 'end': page_counters.start_page + page_counters.counter})
            _pages = page_counters.pages
            
            logger.log_perf(start_time, log_prefix, 'Result pagination')
        
        # Extra check on bad filters (element not found) :
        lost_elements = []
        lost_elements_nb = 0
        lost_fathers_nb = 0
        lost_checks_nb = 0
        if self.compute_not_found:
            start_time = time.time()
            lost_element_set = set()
            if self.filters:
                for _filter in self.filters:
                    if 'uuid_parent' in _filter:
                        for uuid in _filter['uuid_parent']:
                            if not StaticCache.get_host_by_uuid(uuid, user) and (uuid, '', '', '') not in lost_element_set:
                                lost_element_set.add((uuid, '', '', '',))
                    if 'host_name' in _filter:
                        for name in _filter['host_name']:
                            if not StaticCache.get_host_by_name(name, user) and ('', name, '', '') not in lost_element_set:
                                lost_element_set.add(('', name, '', '',))
                    if 'check_uuid' in _filter:
                        for uuid in _filter['check_uuid']:
                            if ('-' not in uuid or not StaticCache.get_check_by_uuid(uuid, user)) and ('', '', '', uuid) not in lost_element_set:
                                lost_element_set.add(('', '', '', uuid,))
                    if 'service_description' in _filter and ('host_name' in _filter or 'uuid_parent' in _filter):
                        for check_name in _filter['service_description']:
                            if 'host_name' in _filter:
                                for host_name in _filter['host_name']:
                                    if ('', host_name, '', '') not in lost_element_set:
                                        host = StaticCache.get_host_by_name(host_name, user)
                                        host_uuid = getattr(host, 'uuid', '')
                                        if (not host or not StaticCache.get_check_by_name(host, check_name, user)) and (host_uuid, host_name, check_name, '') not in lost_element_set:
                                            lost_element_set.add((host_uuid, host_name, check_name, '',))
                            if 'uuid_parent' in _filter:
                                for uuid in _filter['uuid_parent']:
                                    host = StaticCache.get_host_by_uuid(uuid, user)
                                    host_name = getattr(host, 'host_name', '')
                                    if (not host or not StaticCache.get_check_by_name(host, check_name, user)) and (uuid, host_name, check_name, '') not in lost_element_set:
                                        lost_element_set.add((uuid, host_name, check_name, '',))
            if lost_element_set:
                lost_elements = []
                lost_elements_nb = len(lost_element_set)
                for (host_uuid, host_name, check_name, check_uuid) in lost_element_set:
                    lost_element = {}
                    if host_uuid:
                        lost_element.update({'uuid_parent': host_uuid})
                    if host_name:
                        lost_element.update({'host_name': host_name})
                    if check_name:
                        lost_element.update({'service_description': check_name})
                    if check_uuid:
                        lost_element.update({'check_uuid': check_uuid})
                    if check_name or check_uuid:
                        lost_checks_nb = lost_checks_nb + 1
                    else:
                        lost_fathers_nb = lost_fathers_nb + 1
                    if self.ALLOWED_OUTPUT:
                        tmp_element = {}
                        for k in lost_element:
                            if k in self.ALLOWED_OUTPUT:
                                tmp_element.update({self.ALLOWED_OUTPUT[k]: lost_element[k]})
                        lost_element = tmp_element
                    if lost_element:
                        lost_elements.append(lost_element)
            logger.log_perf(start_time, log_prefix, 'Not found filters analysis')
        
        elements_count = OrderedDict()
        elements_count.update({'nb_elements_total': items_count['all']})
        elements_count.update({'nb_hosts_total': items_count['hosts']})
        elements_count.update({'nb_clusters_total': items_count['clusters']})
        elements_count.update({'nb_checks_total': items_count['checks']})
        elements_count.update({'nb_elements_filtered': nb_hosts_found + nb_clusters_found + nb_checks_found})
        elements_count.update({'nb_hosts_filtered': nb_hosts_found})
        elements_count.update({'nb_clusters_filtered': nb_clusters_found})
        elements_count.update({'nb_checks_filtered': nb_checks_found})
        if self.page is not None and _pages:
            elements_count.update({'nb_elements_in_page': nb_hosts_in_page + nb_clusters_in_page + nb_checks_in_page})
            elements_count.update({'nb_hosts_in_page': nb_hosts_in_page})
            elements_count.update({'nb_clusters_in_page': nb_clusters_in_page})
            elements_count.update({'nb_checks_in_page': nb_checks_in_page})
        if lost_elements:
            elements_count.update({'nb_elements_not_found': lost_elements_nb})
            elements_count.update({'nb_fathers_not_found': lost_fathers_nb})
            elements_count.update({'nb_checks_not_found': lost_checks_nb})
        
        filtered_elements = OrderedDict()
        filtered_elements.update({'request_statistics': elements_count})
        output_string = StringIO()
        try:
            output_start = '{'
            elements_count_name = 'elements_count'
            if self.ALLOWED_OUTPUT and elements_count_name in self.ALLOWED_OUTPUT:
                elements_count_name = self.ALLOWED_OUTPUT[elements_count_name]
                tmp_count = OrderedDict()
                for k in elements_count:
                    if k in self.ALLOWED_OUTPUT:
                        tmp_count.update({self.ALLOWED_OUTPUT[k]: elements_count[k]})
                
                if tmp_count:
                    if jsonify_elements:
                        output_string.write('%s"%s":%s' % (output_start, elements_count_name, json.dumps(tmp_count, separators=(',', ':'))))
                        output_start = ','
                    else:
                        filtered_elements.update({'request_statistics': tmp_count})
            
            if self.page is not None and _pages:
                pagination = self._set_pagination_info(_pages)
                pagination_name = 'pagination'
                if self.ALLOWED_OUTPUT:
                    if pagination_name in self.ALLOWED_OUTPUT:
                        pagination_name = self.ALLOWED_OUTPUT[pagination_name]
                    else:
                        pagination_name = ''
                if pagination_name:
                    filtered_elements.update({pagination_name: pagination})
                    output_string.write('%(start_character)s"%(field_name)s":%(data_set)s' % {'start_character': output_start,
                                                                                              'field_name'     : pagination_name,
                                                                                              'data_set'       : json.dumps(pagination, separators=(',', ':'))})
                    output_start = ','
            
            # Drop UI compatibility ...
            # if self.ALLOWED_OUTPUT:
            #     # ret = {self.ALLOWED_OUTPUT[k]: ret[k] for k in ret if k in self.ALLOWED_OUTPUT}
            #     tmp_ret = {}
            #     for k in ret:
            #         if k in self.ALLOWED_OUTPUT:
            #             tmp_ret.update({self.ALLOWED_OUTPUT[k]: ret[k]})
            #     ret = tmp_ret
            #
            # for k in ret:
            #     output_string.write(',"%s":%s' % (k, json.dumps(ret[k], separators=(',', ':'))))
            
            if jsonify_elements:
                if self.flattened_view or not self.checks_in_tree:
                    start_time = time.time()
                    self._jsonify_elements(elements, None, output_string, output_start, False, 'elements_found')
                else:
                    start_time = time.time()
                    output_string.write('%s"%s":{' % (output_start, 'elements_found'))
                    self._jsonify_elements(elements_clusters, cluster_kept_checks.result_set, output_string, '', False, 'clusters')
                    self._jsonify_elements(elements_hosts, host_kept_checks.result_set, output_string, ', ', False, 'hosts')
                    output_string.write('}')
                
                if lost_elements:
                    self._sort_lost_elements(lost_elements)
                    not_found_name = self._get_not_found_property_name()
                    output_string.write(',"%s":%s' % (not_found_name, json.dumps(lost_elements, separators=(',', ':'))))
            else:
                if self.flattened_view or not self.checks_in_tree:
                    elements_found = self._get_elements_with_allowed_output(elements)
                else:
                    elements_found = {
                        'hosts'   : self._get_elements_with_allowed_output(elements_hosts, host_kept_checks.result_set),
                        'clusters': self._get_elements_with_allowed_output(elements_clusters, cluster_kept_checks.result_set)
                    }
                filtered_elements.update({'elements_found': elements_found})
                
                if lost_elements:
                    self._sort_lost_elements(lost_elements)
                    not_found_name = self._get_not_found_property_name()
                    filtered_elements.update({not_found_name: lost_elements})
                return filtered_elements
            
            output_string.write('}')
            
            logger.log_perf(start_time, log_prefix, 'json.dumps')
            
            logger.log_perf(full_start_time, log_prefix, 'nb item in broker [%s] nb item filter [%s] nb item post add missing host [%s] nb item display [%s] filter [%s] sort [%s]' % (
                full_nb_items_in_broker, nb_items_post_filter, nb_element_return, len(elements), str(self.full_filter_for_log), str(self.full_sort_for_log)))
            
            output_string.seek(0, os.SEEK_END)
            
            return output_string.getvalue()
        finally:
            if not output_string.closed:
                output_string.close()
    
    
    def make_filter_skip(self, items):
        # type: (List[Union[BrokerService, BrokerHost]]) -> Tuple[bool, List[Union[BrokerService, BrokerHost]], Tuple[int,int,int]]
        filter_skip = False
        item_found = None
        if self.filters:
            # -- SKIP : search one host by uuid
            is_filter_on_one_host_by_uuid = (
                    len(self.filters) == 1 and
                    len(self.filters[0]) == 2 and
                    'extended_type' in self.filters[0] and
                    'uuid_parent' in self.filters[0] and
                    (self.filters[0]['extended_type'] & {'cluster', 'host'}) and
                    len(self.filters[0]['uuid_parent']) == 1
            )
            
            if is_filter_on_one_host_by_uuid:
                filter_skip = True
                host_uuid = peek_in_set(self.filters[0]['uuid_parent'])
                item = self.monitoring_item_manager.get_host_by_uuid(host_uuid)
                items = [item] if item and (not self.user or check_user_item_access(self.user, item.get_instance_uuid())) else []
                
                if not item:
                    item_found = (0, 0, 0)
                elif item.got_business_rule:
                    item_found = (0, 1, 0)
                else:
                    item_found = (1, 0, 0)
            
            # -- SKIP : search one host by name
            is_filter_on_one_host_by_name = (
                    not filter_skip and
                    len(self.filters) == 1 and
                    len(self.filters[0]) == 2 and
                    'extended_type' in self.filters[0] and
                    'host_name' in self.filters[0] and
                    (self.filters[0]['extended_type'] & {'cluster', 'host'}) and
                    len(self.filters[0]['host_name']) == 1
            )
            
            if is_filter_on_one_host_by_name:
                filter_skip = True
                host_name = peek_in_set(self.filters[0]['host_name'])
                item = self.monitoring_item_manager.get_host_case_insensitive(host_name)
                items = [item] if item and (not self.user or check_user_item_access(self.user, item.get_instance_uuid())) else []
                
                if not item:
                    item_found = (0, 0, 0)
                elif item.got_business_rule:
                    item_found = (0, 1, 0)
                else:
                    item_found = (1, 0, 0)
            
            # -- SKIP : search one check by uuid
            is_filter_on_one_check_by_uuid = \
                not filter_skip and \
                len(self.filters) == 1 and \
                (
                        (
                                len(self.filters[0]) == 3 and
                                'extended_type' in self.filters[0] and
                                'uuid_parent' in self.filters[0] and
                                'check_uuid' in self.filters[0] and
                                (self.filters[0]['extended_type'] & {'check', 'check_host', 'check_cluster'}) and
                                len(self.filters[0]['uuid_parent']) == 1 and
                                len(self.filters[0]['check_uuid']) == 1
                        )
                        or
                        (
                                len(self.filters[0]) == 2 and
                                'check_uuid' in self.filters[0] and
                                'uuid_parent' in self.filters[0] and
                                len(self.filters[0]['uuid_parent']) == 1 and
                                len(self.filters[0]['check_uuid']) == 1
                        )
                )
            
            if is_filter_on_one_check_by_uuid:
                filter_skip = True
                check_uuid = peek_in_set(self.filters[0]['check_uuid'])
                item = self.monitoring_item_manager.get_service_by_uuid(check_uuid)
                items = [item] if item and (not self.user or check_user_item_access(self.user, item.get_instance_uuid())) else []
                if item:
                    item_found = (0, 0, 1)
                else:
                    item_found = (0, 0, 0)
            
            # -- SKIP : search one check by name
            is_filter_on_one_check_by_name = \
                not filter_skip and \
                len(self.filters) == 1 and \
                (
                        (
                                len(self.filters[0]) == 3 and
                                'extended_type' in self.filters[0] and
                                'host_name' in self.filters[0] and
                                'service_description' in self.filters[0] and
                                (self.filters[0]['extended_type'] & {'check', 'check_host', 'check_cluster'}) and
                                len(self.filters[0]['host_name']) == 1 and
                                len(self.filters[0]['service_description']) == 1
                        )
                        or (
                                len(self.filters[0]) == 2 and
                                'host_name' in self.filters[0] and
                                'service_description' in self.filters[0] and
                                len(self.filters[0]['host_name']) == 1 and
                                len(self.filters[0]['service_description']) == 1
                        )
                )
            
            if is_filter_on_one_check_by_name:
                filter_skip = True
                host_name = peek_in_set(self.filters[0]['host_name'])
                check_name = peek_in_set(self.filters[0]['service_description'])
                item = self.monitoring_item_manager.get_service_case_insensitive(host_name, check_name)
                items = [item] if item and (not self.user or check_user_item_access(self.user, item.get_instance_uuid())) else []
                if item:
                    item_found = (0, 0, 1)
                else:
                    item_found = (0, 0, 0)
        
        if filter_skip:
            logger.info('[get_all_hosts_and_services] skip filter and found directly item')
        
        return filter_skip, items, item_found
    
    
    def _sort_lost_elements(self, lost_elements):
        if self.ALLOWED_OUTPUT:
            lost_elements.sort(key=lambda elt: (elt.get(self.ALLOWED_OUTPUT.get('host_name', 'None'), ''),
                                                elt.get(self.ALLOWED_OUTPUT.get('uuid_parent', 'None'), ''),
                                                elt.get(self.ALLOWED_OUTPUT.get('check_name', 'None'), ''),
                                                elt.get(self.ALLOWED_OUTPUT.get('check_uuid', 'None'), '')))
        else:
            lost_elements.sort(key=lambda elt: (elt.get('host_name', ''), elt.get('uuid_parent', ''), elt.get('check_name', ''), elt.get('check_uuid', '')))
    
    
    def _get_not_found_property_name(self, not_found_name='element_not_found'):
        # type: (str) -> str
        if self.ALLOWED_OUTPUT and not_found_name in self.ALLOWED_OUTPUT:
            not_found_name = self.ALLOWED_OUTPUT[not_found_name]
        else:
            available_output = getattr(self, 'AVAILABLE_OUTPUT', {})
            if not_found_name in available_output:
                not_found_name = available_output[not_found_name]
        return not_found_name
    
    
    def _update_element_output(self, output_element, unauthorized_properties):
        # type: (Dict, List[str]) -> Dict
        if self.ALLOWED_OUTPUT:
            tmp_output = {}
            for k in self.ALLOWED_OUTPUT:
                if k in unauthorized_properties:
                    continue
                if k in output_element:
                    tmp_output.update({self.ALLOWED_OUTPUT[k]: output_element[k]})
            return tmp_output
    
    
    def _get_elements_with_allowed_output(self, elements, fathers_checks=None):
        # type: (List[Union[BrokerHost,BrokerService]], Dict) -> List
        updated_fathers = []
        for output_element in elements:
            output_element = self.api_list_element_dict(output_element)
            host_name = output_element['host_name']
            output_element = self._update_element_output(output_element, [])
            updated_fathers.append(output_element)
            
            if fathers_checks is None:
                continue
            if host_name in fathers_checks:
                checks = fathers_checks[host_name]
                for output_check in checks:
                    output_check = self._update_element_output(self.api_list_element_dict(output_check), ['uuid_parent', 'host_name'])
                    if 'checks' not in output_element and output_check:
                        output_element['checks'] = []
                    output_element['checks'].append(output_check)
        return updated_fathers
    
    
    def _set_pagination_info(self, pages):
        # type: (List[Dict[str, int]]) -> Dict
        if self.page is None:
            return {}
        
        has_next_page = True if self.page < (len(pages) - 1) else False
        pagination = OrderedDict()
        pagination.update({'has_next_page': has_next_page})
        pagination.update({'nb_total_page': len(pages)})
        pagination.update({'page': self.page})
        pagination.update({'page_size': self.page_size})
        return pagination


class RequestManagerLivedataV2(FilterItem):
    ALLOWED_FILTER = {
        'description'                : 'display_name',
        'status'                     : 'status',
        'context'                    : 'context',
        'is_status_confirmed'        : 'is_status_confirmed',
        'type'                       : 'extended_type',
        'father_name'                : 'host_name',
        'father_name_contains'       : 'host_name_like',
        'father_uuid'                : 'uuid_parent',
        'check_name'                 : 'service_description',
        'check_name_contains'        : 'service_description_like',
        'check_uuid'                 : 'check_uuid',
        'realm'                      : 'realm',
        'notification_options'       : 'notification_options',
        'notification_period'        : 'notification_period',
        'maintenance_period'         : 'maintenance_period',
        'address'                    : 'address',
        'notification_contacts'      : 'contacts',
        'notification_contact_groups': 'contact_groups',
        'father_templates'           : 'host_templates',
        'host_groups'                : 'host_groups',
        'business_impact'            : 'business_impact',
        'status_since'               : 'status_since',
        'status_confirmed_since'     : 'last_hard_state_change',
        'last_check'                 : 'last_check',
        'next_check'                 : 'next_check',
        'thresholds_display'         : 'thresholds_display'
    }
    ALLOWED_SORT = ALLOWED_FILTER
    
    # Can be used to build ALLOWED_OUTPUT (which converts internal names to public names, and select allowed output fields)
    AVAILABLE_OUTPUT = {
        'element_not_found'     : 'elements_not_found',
        'elements_count'        : 'request_statistics',
        'nb_elements_total'     : 'nb_elements_total',
        'nb_hosts_total'        : 'nb_hosts_total',
        'nb_clusters_total'     : 'nb_clusters_total',
        'nb_checks_total'       : 'nb_checks_total',
        'nb_elements_filtered'  : 'nb_elements_filtered',
        'nb_hosts_filtered'     : 'nb_hosts_filtered',
        'nb_clusters_filtered'  : 'nb_clusters_filtered',
        'nb_checks_filtered'    : 'nb_checks_filtered',
        'nb_elements_in_page'   : 'nb_elements_in_page',
        'nb_hosts_in_page'      : 'nb_hosts_in_page',
        'nb_clusters_in_page'   : 'nb_clusters_in_page',
        'nb_checks_in_page'     : 'nb_checks_in_page',
        'nb_elements_not_found' : 'nb_elements_not_found',
        'nb_fathers_not_found'  : 'nb_fathers_not_found',
        'nb_checks_not_found'   : 'nb_checks_not_found',
        'address'               : 'address',
        'attempts'              : 'attempts',
        'business_impact'       : 'business_impact',
        'service_description'   : 'check_name',
        'contacts'              : 'notification_contacts',
        'contact_groups'        : 'notification_contact_groups',
        'context'               : 'context',
        'display_name'          : 'description',
        'extended_type'         : 'type',
        'host_name'             : 'father_name',
        'host_groups'           : 'host_groups',
        'host_templates'        : 'father_templates',
        'is_status_confirmed'   : 'is_status_confirmed',
        'last_check'            : 'last_check',
        'long_output'           : 'long_output',
        'max_attempts'          : 'max_attempts',
        'next_check'            : 'next_check',
        'notification_options'  : 'notification_options',
        'notification_period'   : 'notification_period',
        'maintenance_period'    : 'maintenance_period',
        'check_period'          : 'check_period',
        'output'                : 'output',
        'perf_data'             : 'raw_perf_data',
        'realm'                 : 'realm',
        'smart_perf_data'       : 'perf_data',
        'status'                : 'status',
        'status_since'          : 'status_since',
        'last_hard_state_change': 'status_confirmed_since',
        'check_uuid'            : 'check_uuid',
        'uuid_parent'           : 'father_uuid',
        'pagination'            : 'pagination',
        'notes_url'             : 'notes_url',
        'notes_multi_url'       : 'notes_multi_url',
        'thresholds_display'    : 'thresholds_display'
    }
    POST_AVAILABLE_PARAMETERS = {'filter', 'output_field', 'output_format', 'sort', 'with_element_not_found'}
    
    
    def __init__(self, monitoring_item_manager, user, mandatory_output_fields=None, user_allowed_output_field=None, user_filters=None, user_allowed_filters=None, page_settings='', bottle_post=None, additional_format='', compute_not_found=None):
        # type:(MonitoringItemManager, Optional[Contact], Optional[List[str]], Optional[List[str]], str, List[str], str, Optional[MultiDict, Dict[str, str]], str, bool) -> None
        super(RequestManagerLivedataV2, self).__init__(monitoring_item_manager, user)
        
        self.update_modified_properties(
            mandatory_output_fields=mandatory_output_fields,
            user_allowed_output_field=user_allowed_output_field,
            user_filters=user_filters,
            user_allowed_filters=user_allowed_filters,
            page_settings=page_settings,
            bottle_post=bottle_post,
            additional_format=additional_format,
            compute_not_found=compute_not_found
        )
    
    
    def update_modified_properties(self, mandatory_output_fields=None, user_allowed_output_field=None, user_filters=None, user_allowed_filters=None, page_settings='', bottle_post=None, additional_format='', compute_not_found=None):
        # type:(Optional[List[str]], Optional[List[str]], str, List[str], str, Optional[MultiDict, Dict[str, str]], str, bool) -> None
        self.set_allowed_output(bottle_post=bottle_post, mandatory_fields=mandatory_output_fields, user_allowed_output_field=user_allowed_output_field)
        if user_allowed_filters:
            user_allowed_filters = self.build_restricted_allowed_filters(user_allowed_filters)
        
        if user_filters:
            self.set_filters(user_filters, user_allowed_filters)
        
        if page_settings:
            self.set_range(page_settings)
        
        if bottle_post:
            self.set_output_format(bottle_post, additional_format=additional_format)
        
        if compute_not_found:
            self.set_compute_not_found(compute_not_found)
    
    
    @staticmethod
    def validate_bottle_post_parameter(bottle_post, enabled_parameters=None, disabled_parameters=None):
        # type: (MultiDict, Optional[Set], Optional[Set]) -> None
        if not bottle_post:
            return
        for key in bottle_post.keys():
            if isinstance(key, bytes):
                key = key.decode('UTF-8')
            if disabled_parameters and (key in disabled_parameters or (key[0:6] == 'filter' and 'filter' in disabled_parameters)):
                raise ShinkenExceptionKeyError(text='POST parameter [ %s ] is not available for this route' % key)
            if enabled_parameters and key not in enabled_parameters and (key[0:6] != 'filter' or 'filter' not in enabled_parameters):
                raise ShinkenExceptionKeyError(text='POST parameter [ %s ] is unknown' % key)
    
    
    @staticmethod
    def build_filters_from_bottle_post(bottle_post):
        # type: (MultiDict) -> str
        filters = ''
        if not bottle_post:
            return filters
        cpt = 0
        for key in bottle_post.keys():
            if isinstance(key, bytes):
                key = key.decode('UTF-8')
            if len(key) >= 6 and key[0:6] == 'filter':
                for value in bottle_post.getall(key):
                    if isinstance(value, bytes):
                        value = value.decode('UTF-8')
                    new_filters = '"filter%(cpt)d":"%(value)s"' % {'cpt': cpt, 'value': value}
                    if len(filters) > 0:
                        filters = '%(filters)s, %(new_filter)s' % {'filters': filters, 'new_filter': new_filters}
                    else:
                        filters = new_filters
                    cpt += 1
        return '{%s}' % filters
    
    
    def build_restricted_allowed_filters(self, restricted_filter_tags):
        # type: (Optional[Collection[str]]) -> Dict
        # reduce usable filter fields set
        # input: public filter name, output: dict for FilterItem
        if not restricted_filter_tags:
            return self.ALLOWED_FILTER
        
        allowed_filters = {}
        for tag in restricted_filter_tags:
            if tag not in self.ALLOWED_FILTER:
                raise ShinkenExceptionKeyError(text='incorrect filter tag name [ %s ]' % tag)
            allowed_filters.update({tag: self.ALLOWED_FILTER[tag]})
        return allowed_filters
    
    
    def build_allowed_output(self, output_fields, user_allowed_output_field):
        # type: (List, Optional[List[str]]) -> Dict
        # input: list of public field names, output: dict for Filter output managing
        new_output = {}
        if output_fields:
            # reversed_output = {v: k for k, v in item_set.get_output_fields().items()}
            reversed_output = {}
            for k, v in self.AVAILABLE_OUTPUT.items():
                reversed_output.update({v: k})
            for field in output_fields:
                if field in reversed_output:
                    if field not in user_allowed_output_field:
                        raise ShinkenExceptionKeyError(text='output_field: field name [ %s ] not allowed in this route' % field)
                    new_output.update({reversed_output[field]: field})
                else:
                    raise ShinkenExceptionKeyError(text='output_field: invalid field name [ %s ]' % field)
        return new_output
    
    
    def set_allowed_output(self, bottle_post=None, mandatory_fields=None, user_allowed_output_field=None):
        # type: (Optional[MultiDict], Optional[List[str]], Optional[List[str]]) -> None
        output_fields = None
        
        if user_allowed_output_field:
            user_allowed_output_field.extend(mandatory_fields)
        else:
            user_allowed_output_field = set(self.AVAILABLE_OUTPUT.values())
        
        if bottle_post:
            output_fields = bottle_post.get('output_field', None)
            if output_fields and isinstance(output_fields, bytes):
                output_fields = output_fields.decode('UTF-8')
        
        if output_fields:
            output_fields = output_fields.strip().split('~')
        
        if not output_fields:
            output_fields = []
        
        if mandatory_fields:
            output_fields.extend(mandatory_fields)
        
        self.set_output_fields(self.build_allowed_output(output_fields, user_allowed_output_field))
    
    
    def set_output_format(self, bottle_post, additional_format=''):
        # type: (MultiDict, str) -> None
        output_format = bottle_post.get('output_format', None)
        if output_format is not None:
            if output_format not in ['checks_attached_to_father', 'elements_on_same_level', additional_format]:
                if isinstance(output_format, bytes):
                    output_format = output_format.decode('UTF-8')
                raise ShinkenExceptionValueError(text='output_format: invalid value [ %s ]' % output_format)
            self.set_flattened_mode((output_format == 'elements_on_same_level' or output_format == additional_format), True)
    
    
    def set_print_element__not_found(self, bottle_post):
        # type: (MultiDict) -> None
        mode = bottle_post.get('with_element_not_found', None)
        if mode is not None:
            if mode not in ['true', 'false']:
                raise ShinkenExceptionValueError(text='with_element_not_found: invalid value [ %s ]' % mode)
            self.set_compute_not_found(mode == 'true')
