#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

from abc import ABCMeta, abstractmethod

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.pickledb import ShinkenPickleableMeta

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import List, Any, Tuple, ContextManager


class AbstractDataHubDriverConfig(object, metaclass=ShinkenPickleableMeta):
    def __init__(self, name, namespace=''):
        # type: (str, str) -> None
        self.name = name
        self.namespace = namespace
        
        # set in DataHubFactory.build_driver
        self.data_type = ''
        self.data_hub_id = ''


class AbstractDataHubDriver(object, metaclass=ABCMeta):
    def __init__(self, logger, driver_config):
        # type: (PartLogger, AbstractDataHubDriverConfig) -> None
        self._name = driver_config.name
        self._namespace = driver_config.namespace
        self._data_type = driver_config.data_type
        self._data_hub_id = driver_config.data_hub_id
        
        self._logger = logger.get_sub_part(self._name)
        self._logger_init = self._logger.get_sub_part('INIT')
        self._logger_reader = self._logger.get_sub_part('READ')
        self._logger_writer = self._logger.get_sub_part('WRITE')
        
        self.rights_access_correctly_granted = True
        self.rights_access_per_data_id = {}
    
    
    @abstractmethod
    def init(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    @abstractmethod
    def get_number_of_stored_data(self):
        # type: () -> int
        raise NotImplementedError()
    
    
    @abstractmethod
    def get_all_data_id(self):
        # type: () -> List[str]
        raise NotImplementedError()
    
    
    # Default implementation to read all available data
    # Can be overridden for an optimized way (a.k.a. MONGO)
    def get_all_data(self, log_error=True):
        # type: (bool) -> List[Any]
        return self.read_all(self.get_all_data_id(), log_error=log_error)
    
    
    # Default implementation to read all available data with their ids
    # Can be overridden for an optimized way (a.k.a. MONGO)
    def get_all_data_with_id(self, log_error=True):
        # type: (bool) -> List[Tuple[str, Any]]
        return self.read_all_with_id(self.get_all_data_id(), log_error=log_error)
    
    
    @abstractmethod
    def find_data_id(self, filters):
        # type: (Any) -> List[str]
        raise NotImplementedError()
    
    
    # Default implementation to read all data using a given filter
    # Can be overridden for an optimized way (a.k.a. MONGO)
    def find_data(self, filters, log_error=True):
        # type: (Any, bool) -> List[Any]
        return self.read_all(self.find_data_id(filters), log_error=log_error)
    
    
    # Default implementation to read all available data with their ids using a given filter
    # Can be overridden for an optimized way (a.k.a. MONGO)
    def find_data_with_id(self, filters, log_error=True):
        # type: (Any, bool) -> List[Tuple[str, Any]]
        return self.read_all_with_id(self.find_data_id(filters), log_error=log_error)
    
    
    @abstractmethod
    def is_data_correct(self, data_id):
        # type: (str) -> bool
        raise NotImplementedError()
    
    
    @abstractmethod
    def lock_context(self, data_id):
        # type: (str) -> ContextManager[Any]
        raise NotImplementedError()
    
    
    @abstractmethod
    def write(self, data_id, data):
        # type: (str, Any) -> None
        raise NotImplementedError()
    
    
    @abstractmethod
    def write_raw(self, data_id, data):
        # type: (str, Any) -> None
        raise NotImplementedError()
    
    
    @abstractmethod
    def read(self, data_id, log_error=True):
        # type: (str, bool) -> Any
        raise NotImplementedError()
    
    
    # Default implementation to read all data by id
    # Can be overridden for an optimized way (a.k.a. MONGO)
    def read_all(self, data_id_list, log_error=True):
        # type: (List[str], bool) -> List[Any]
        return [self.read(data_id, log_error=log_error) for data_id in data_id_list]
    
    
    # Default implementation to read all data by id, returning a pair of (id, data)
    # Can be overridden for an optimized way (a.k.a. MONGO)
    def read_all_with_id(self, data_id_list, log_error=True):
        # type: (List[str], bool) -> List[Tuple[str, Any]]
        return [(data_id, self.read(data_id, log_error=log_error)) for data_id in data_id_list]
    
    
    @abstractmethod
    def read_raw(self, data_id, log_error=True):
        # type: (str, bool) -> Any
        raise NotImplementedError()
    
    
    def read_and_get_last_modification_date(self, data_id, log_error=True):
        # type: (str, bool) -> Tuple[Any, int]
        return self.read(data_id, log_error), self.get_last_modification_date(data_id)
    
    
    @abstractmethod
    def remove(self, data_id):
        # type: (str) -> None
        raise NotImplementedError()
    
    
    def hard_remove(self, data_id):
        # type: (str) -> None
        self.remove(data_id)
    
    
    @abstractmethod
    def get_last_modification_date(self, data_id):
        # type: (str) -> int
        raise NotImplementedError()
    
    
    def restore(self, data_id, data):
        # type: (str, Any) -> None
        self.write_raw(data_id, data)
    
    
    def sanitize(self, data_id, data):
        # type: (str, Any) -> None
        self.write(data_id, data)
    
    
    @abstractmethod
    def destroy(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    @abstractmethod
    def stop(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    @abstractmethod
    def get_total_size(self):
        # type: () -> int
        raise NotImplementedError()
    
    
    @abstractmethod
    def get_size_of(self, data_id):
        # type: (str) -> int
        raise NotImplementedError()
    
    
    def check_right_access_issues_then_fix_them(self):
        # type: () -> None
        pass
    
    
    def force_shinken_rights(self, data_id):
        # type: (str) -> None
        pass
