#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.
import sys
import time

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import to_bool
from shinkensolutions.system_tools import check_right_on_file

if TYPE_CHECKING:
    from typing import Optional, Dict, Any
    from logging import Logger
    from shinken.log import PartLogger
    from shinkensolutions.logs.create_user_log_file import CreateUserLogFile


class LoggerAbstract:
    
    KEY_LOG_USERS__ENABLED = 'log_users__enabled'
    KEY_LOG_USERS__FILE_PATH = 'log_users__file_path'
    KEY_LOG_USERS__ADD_USER_NAME = 'log_users__add_user_name'
    
    
    def __init__(self, requester_prefix, conf, create_user_log_file, logger_error=None):
        #  type: (str, Any, CreateUserLogFile,Optional[PartLogger]) -> None
        self.log_enable = to_bool(getattr(conf, '%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ENABLED), False))
        
        _default_path = '/var/log/shinken/%s/log_users.log' % requester_prefix
        self.log_file = getattr(conf, '%s__%s' % (requester_prefix, self.KEY_LOG_USERS__FILE_PATH), _default_path)
        if not self.log_file or not isinstance(self.log_file, str):
            self.log_file = _default_path
        
        self.add_user_name = to_bool(getattr(conf, '%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ADD_USER_NAME), False))
        # This logger are use for log errors of this logger,
        # classically use the default logger of the module or daemon
        self.logger_error = logger_error
        self.create_user_log_file = create_user_log_file
        self._logger = None  # type: Optional[Logger]
        
        if self.log_enable and not self._logger:
            self._logger = create_user_log_file.init_internal_logger()
    
    
    def _build_common_log_data(self,
                               user_uuid=None,
                               user_name=None,
                               requester_ip=None,
                               run_time=None,
                               result_code=None,
                               requester_module=None,
                               auth_module=None,
                               session_message=None,
                               error=None
                               ):
        # type: (Optional[str], Optional[str], Optional[str], Optional[float], Optional[int], Optional[str], Optional[str], Optional[str], Optional[str]) -> Dict
        _run_time = None
        if run_time:
            _run_time = run_time * 1000  # we want ms
        return {
            'log_time'        : time.strftime('%Y-%m-%d %H:%M:%S') if sys.platform.startswith('win') else time.strftime('%Y-%m-%d %T') ,
            'user_info'       : self._build_user_info(user_uuid, user_name),
            'mode'            : 'READ',
            'run_time'        : _run_time,
            'result_code'     : result_code,
            'requester_ip'    : requester_ip,
            'requester_module': requester_module,
            'auth_module'     : auth_module,
            'session_message' : session_message,
            'error'           : error
        }
    
    
    def _build_user_info(self, user_uuid, user_name):
        # type: (Optional[str], Optional[str]) -> Optional[str]
        
        user_info = user_uuid
        if self.add_user_name and user_name:
            if user_info:
                user_info += '/%-6s' % user_name
            else:
                user_info = user_name
        if user_info:
            return user_info
        return None
    
    
    @staticmethod
    def _build_message(log_data, formats):
        # type: (Dict, [str]) -> str
        # We compare with None to keep the empty elements.
        res = (_format % log_data for key_data, _format in formats if log_data.get(key_data, None) is not None)
        return ' '.join(res)
    
    
    def _log(self, txt):
        try:
            check_right_on_file(self.log_file, 'write')
            self._logger.info(txt)
        except OSError:
            self.create_user_log_file.create_path_and_add_right()
            try:
                self._logger.info(txt)
            except Exception as e:
                if self.logger_error:
                    self.logger_error.error('Could not write in file [%s] -> Error [%s]' % (self.log_file, e))
