#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023
# This file is part of Shinken Enterprise, all rights reserved.

import codecs
import json
import os
import shutil
import ssl
import threading
import time

from graphite.logger import log
from shinkensolutions.http_helper import urlopen, URLError


# ################ IMPORTANT #################
# THIS FILE IS LOADED IN APACHE WORKER PROCESS
# ############################################


class InApacheHTTPInventoryReader:
    def __init__(self, elements_mapping, elements_mapping_lock, log_about_f, graphite_banner, inter_process_lock, cache_dir):
        self._elements_mapping = elements_mapping
        self._elements_mapping_lock = elements_mapping_lock
        self._log_about = log_about_f
        self._graphite_banner = graphite_banner
        
        self._shinken_inventory_uri = 'http://localhost:52000/inventory/'
        self._shinken_inventory_timeout = 10
        self._shinken_inventory_configuration_file = '/opt/graphite/conf/shinken_inventory.conf'
        self._inventory_update_chapter = '%s [ %%s ] [ INVENTORY CACHE UPDATE ]' % graphite_banner
        
        self._inter_process_lock = inter_process_lock
        
        self._inventory_cache_file = '%s/%s' % (cache_dir, '.shinken_apache__daemons__httpd_graphite__data__shinken_inventory.json')
        self._sup_data_cache_file = '%s/%s' % (cache_dir, '.shinken_apache__daemons__httpd_graphite__data__sup_data.json')
        self._inventory_fetching_file = '%s/%s' % (cache_dir, '.shinken_apache__daemons__httpd_graphite__inventory_fetching_in_progress')
    
    
    @property
    def configuration_file_name(self):
        return self._shinken_inventory_configuration_file
    
    
    @configuration_file_name.setter
    def configuration_file_name(self, file_name):
        self._shinken_inventory_configuration_file = file_name
    
    
    def get_inventory_file_name(self):
        return self._inventory_cache_file
    
    
    def log_info(self, message):
        log.info('%s %s' % (self._inventory_update_chapter % threading.current_thread().name, message))
    
    
    def log_fatal(self, message):
        message = '%s %s' % (self._inventory_update_chapter % threading.current_thread().name, message)
        log.info(message)
        log.exception(message)
    
    
    # We do not care about cache reset as we dump full every time
    def cache_was_reset(self):
        pass
    
    
    @staticmethod
    def _save_into_file(fpath, data):
        with codecs.open(fpath + '.tmp', 'wb', 'utf8') as f:
            buf = json.dumps(data)
            f.write(buf)
        try:
            shutil.move(fpath + '.tmp', fpath)
        except:  # noqa : maybe another worker did just do it, not a problem, we accept here
            pass
    
    
    def inventory_data_from_shinken_lookup(self, cache_invalidation_date):
        with self._inter_process_lock:
            if os.path.exists(self._inventory_cache_file):
                try:
                    data_last_update = os.stat(self._inventory_cache_file).st_mtime
                except:  # noqa : maybe did move between test, not a real problem
                    data_last_update = 0
                
                start_time = time.time()
                with codecs.open(self._inventory_cache_file, 'rb', 'utf8') as f:
                    buf = f.read()
                read_time = time.time()
                data = json.loads(buf)
                json_time = time.time()
                self._elements_mapping.clear()
                self._elements_mapping.update(data)
                end_time = time.time()
                self.log_info('Update host/check mapping:: the inventory was loaded from memory cache file in %.3fs ( including file read in %.3fs, json parse in %.3fs, dict update in %.3fs )' % (
                    end_time - start_time,
                    read_time - start_time,
                    json_time - read_time,
                    end_time - json_time))
                if data_last_update >= cache_invalidation_date:
                    return True
            
            return self._inventory_data_from_shinken_lookup()
    
    
    def _touch_file(self, file_name):
        now = time.time()
        try:
            os.utime(file_name, (now, now))
        except Exception as e:
            self.log_fatal('File [ %s ] touch failed with error %s' % (file_name, e))
    
    
    def _remove_cache_fetching_status_file(self):
        try:
            os.unlink(self._inventory_fetching_file)
        except Exception as e:
            self.log_fatal('File [ %s ] removal failed with error %s' % (self._inventory_fetching_file, e))
    
    
    def _inventory_data_from_shinken_lookup(self):
        # type: () -> bool
        t0 = time.time()
        if os.path.isfile(self._shinken_inventory_configuration_file):
            # Beware: ConfigParser is not at the same place in PY2 and PY3
            from shinken.compat import ConfigParser as ConfigParserLib
            parser = ConfigParserLib.ConfigParser()
            parser.read(self._shinken_inventory_configuration_file)
            
            try:
                enable = parser.get('inventory', 'ENABLE') != '0'
            except ConfigParserLib.NoOptionError:
                enable = True
            
            if not enable:
                return True
            
            try:
                self._shinken_inventory_uri = parser.get('inventory', 'URI').split(',')
            except ConfigParserLib.NoOptionError:
                pass
            
            try:
                self._shinken_inventory_timeout = float(parser.get('inventory', 'TIMEOUT'))
            except (ConfigParserLib.NoOptionError, ValueError):
                pass
        
        # Whatever we have as conf, we are working on a list of URLs
        if not isinstance(self._shinken_inventory_uri, list):
            self._shinken_inventory_uri = [self._shinken_inventory_uri]
        
        if os.path.exists(self._inventory_fetching_file):
            stats = os.stat(self._inventory_fetching_file)
            if stats.st_mtime > (time.time() - 2 * self._shinken_inventory_timeout):
                # Another worker is fetching inventory data
                return True
            # File has not been updated for too long, other worker must be dead, we take its place
            self._touch_file(self._inventory_fetching_file)
        
        else:
            # we create the file.
            try:
                with open(self._inventory_fetching_file, 'w'):
                    pass
            except Exception as e:
                self.log_fatal('File [ %s ] creation failed with error %s' % (self._inventory_fetching_file, e))
        
        self.log_info('parameters : URI %s, timeout [%s]' % (self._shinken_inventory_uri, self._shinken_inventory_timeout))
        
        success = False
        sup = {}
        _new_elements_mapping = {}
        
        if os.path.exists(self._inventory_cache_file):
            cache_stats = os.stat(self._inventory_cache_file)
            # if cache file does not contain any UUID (32 bytes), consider it as empty
            if cache_stats.st_size <= 32:
                first_startup_mode = True
            else:
                first_startup_mode = False
        else:
            first_startup_mode = True
        
        # Release inter process lock while fetching inventory to avoid locking all Graphite workers
        try:
            self._inter_process_lock.release()
            self._elements_mapping_lock.release()
            
            for url in self._shinken_inventory_uri:
                if self._get_inventory_data_from_shinken(url, sup, _new_elements_mapping):
                    success = True
                    if first_startup_mode:
                        # As we have no data in cache, fill the cache as soon as we can for other workers
                        with self._inter_process_lock:
                            self._save_into_file(self._inventory_cache_file, _new_elements_mapping)
                
                self._touch_file(self._inventory_fetching_file)
            if sup:
                self._save_into_file(self._sup_data_cache_file, sup)
        finally:
            self._elements_mapping_lock.acquire()
            self._inter_process_lock.acquire()
            self._remove_cache_fetching_status_file()
        
        
        self._elements_mapping.clear()
        self._elements_mapping.update(_new_elements_mapping)
        
        if not success:
            return False
        
        before = time.time()
        self._save_into_file(self._inventory_cache_file, _new_elements_mapping)
        elapsed = time.time() - before
        self.log_info('Update host/check mapping:: the inventory was stored in the memory cache file in %.3fs' % elapsed)
        
        nb_hosts = len(_new_elements_mapping)
        nb_checks = 0
        for host_entry in _new_elements_mapping.values():
            nb_checks += (len(host_entry['checks']) - 1)  # remove the __host__ entry
        self.log_info('Update host/check mapping:: Total number of hosts/checks in mapping cache: %s/%s, Cache update took: %.3fs' % (nb_hosts, nb_checks, time.time() - t0))
        _new_elements_mapping = {}
        return True
    
    
    def _get_inventory_data_from_shinken(self, url, sup, elements_mapping):
        start_time = time.time()
        url = url.strip()
        
        self.log_info('Updating inventory data from URL [ %s ]' % url)
        
        try:
            result = None
            while result is None:
                try:
                    if hasattr(ssl, '_create_unverified_context'):
                        context = ssl._create_unverified_context()  # noqa: no choice here
                        result = urlopen(url, timeout=self._shinken_inventory_timeout, context=context)
                    else:
                        result = urlopen(url, timeout=self._shinken_inventory_timeout)
                    break
                except URLError as error:
                    if getattr(error.reason, 'errno', 0) == 111 and (time.time() - start_time) < self._shinken_inventory_timeout:
                        time.sleep(0.5)
                        continue
                    raise
        except Exception as error:
            attempt_duration = time.time() - start_time
            self.log_info('Unable to access URL [ %s ] after %.3fs with error [ %s ]' % (url, attempt_duration, error))
            sup.update({url: (1, start_time, 'Unable to connect after .%3fs [ %s ]' % (attempt_duration, error))})
            return False
        
        # log.info('%s Requesting URI [ %s ] to fetch inventory data' % (graphite_banner, url))
        
        try:
            result = json.load(result)
        except Exception as error:
            self.log_info('Unable to decode received json data from URL [ %s ] with error [ %s ]' % (url, error))
            sup.update({url: (1, start_time, 'Unable to decode data [ %s ]' % error)})
            return False
        
        if 'inventory' not in result:
            self.log_info('missing inventory data in reply from URL [ %s ]' % url)
            sup.update({url: (1, start_time, 'missing inventory data in server reply')})
            return False
        nb_check = nb_host = 0
        for element in result['inventory']:
            name, uuid, status, last_update = element
            host_uuid, check_uuid = uuid.split('.', 1)
            host_name, check_name = name.split('.', 1)
            host_entry = elements_mapping.get(host_name, None)
            if host_entry is None:
                host_entry = {'uuid': host_uuid, 'checks': {}}
                elements_mapping[host_name] = host_entry
                self._log_about(host_name, '%s Update host/check mapping:: new host detected: %s, uuid=%s' % (self._graphite_banner, host_name, host_uuid))
            if host_uuid != host_entry['uuid']:
                self._log_about(host_name, '%s Update host/check mapping:: Old check name is detected, skipping it: %s/%s %s %s' % (self._graphite_banner, host_name, check_name, host_uuid, check_uuid))
                continue
            
            check_entry = {'uuid': check_uuid}
            host_entry['checks'][check_name] = check_entry
            if check_uuid != '__HOST__':
                nb_check = nb_check + 1
                self._log_about(host_name, '%s Update host/check mapping:: New check add: %s/%s, uuid=%s-%s' % (self._graphite_banner, host_name, check_name, host_uuid, check_uuid))
            else:
                nb_host = nb_host + 1
        self.log_info('successfully updated inventory data with %s hosts / %s checks from URL [ %s ] in %.3fs' % (nb_host, nb_check, url, time.time() - start_time))
        sup.update({url: (0, start_time, 'Inventory fetched in %.3fs' % (time.time() - start_time))})
        return True
