#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

# ################ IMPORTANT #################
# THIS FILE IS LOADED IN APACHE WORKER PROCESS
# ############################################

import fnmatch
import os
import shutil
import threading
import time

from graphite.logger import log
from shinkensolutions.locking.shinken_locking.shinken_interprocess_rlock import ShinkenInterProcessRLock, create_tree
from shinkensolutions.metrology.graphite.in_apache.http_inventory_reader import InApacheHTTPInventoryReader
from shinkensolutions.metrology.graphite.in_apache.mongodb_inventory_reader import InApacheMongodbInventoryReader

graphite_banner = '[ APACHE(pid=%6d) / GRAPHITE ]' % os.getpid()

LOG_FLAG_FILE = '/opt/graphite/storage/whisper/.apache_graphite_host_filter_log'
LOG_FLAG_FILE_LAST_CHECK = 0
LOG_FLAG_FILE_FILTER = '--no-filter--'


def log_about(host_name, line):
    global LOG_FLAG_FILE_LAST_CHECK, LOG_FLAG_FILE_FILTER
    if not os.path.exists(LOG_FLAG_FILE):
        return
    now = int(time.time())
    if now > LOG_FLAG_FILE_LAST_CHECK + 10:  # re-read the filter log file every 10s
        with open(LOG_FLAG_FILE, 'r') as f:
            old_filter = LOG_FLAG_FILE_FILTER
            new_filter = f.read().strip()
            if new_filter != '' and old_filter != new_filter:
                log.info('%s Updating log filter from %s to %s' % (graphite_banner, old_filter, LOG_FLAG_FILE_FILTER))
                LOG_FLAG_FILE_FILTER = new_filter
            LOG_FLAG_FILE_LAST_CHECK = now
    if LOG_FLAG_FILE_FILTER in host_name:
        try:
            log.info(('[ FILTER=%s ] ' % LOG_FLAG_FILE_FILTER) + line)
        except UnicodeError:
            log.error(('Fail to log of the hostname with the current filter %s' % LOG_FLAG_FILE_FILTER))


class InApacheGraphiteMappingInventory:
    # noinspection SpellCheckingInspection
    _cache_invalidation_file = '/opt/graphite/storage/whisper/.cacheinvalidation'
    
    
    def __init__(self):
        # ELEMENTS_MAPPING:
        #   host_name:
        #      'uuid' => uuid of the host
        #      'checks':
        #          'service description':
        #             'uuid' => uuid of the Check
        self._elements_mapping = {
            # EXAMPLE/
            # 'serveur_linux_shinken': {
            #     'uuid'  : 'd41ed7d463e011e88ecd080027f6d105',
            #     'checks': {
            #         'Kernel_Stats': {
            #             'uuid': 'c2971d2c5ad911e58cc5080027f08538',
            #         },
            #         'Load_Average': {
            #             'uuid': 'c297260a5ad911e58cc5080027f08538',
            #         },
            #     },
            # },
        }
        
        self._elements_mapping_lock = threading.RLock()
        self._inter_process_lock = ShinkenInterProcessRLock('%s.lock' % self._cache_invalidation_file)
        
        self._httpd_start_time = self._get_parent_proc_start_time()
        self._cache_dir = '/dev/shm/shinken_apache'
        create_tree(self._cache_dir)
        self._cache_invalidation_date = self._httpd_start_time  # We will reload inventory each time Apache is restarted
        
        self._mongodb_inventory_reader = InApacheMongodbInventoryReader(self._elements_mapping, self._elements_mapping_lock, log_about, graphite_banner)
        self._http_inventory_reader = InApacheHTTPInventoryReader(self._elements_mapping, self._elements_mapping_lock, log_about, graphite_banner, self._inter_process_lock, self._cache_dir)
        
        self._mongodb_inventory_reader.configuration_file_name = self._copy_configuration_file_in_cache(self._mongodb_inventory_reader.configuration_file_name)
        self._http_inventory_reader.configuration_file_name = self._copy_configuration_file_in_cache(self._http_inventory_reader.configuration_file_name)
        
        self._last_update_time = 0  # do not query inventory server too much
        self._log_about = log_about
        self._graphite_banner = graphite_banner
        self._graphite_cache_banner = '%s [ %%s ] [ INVENTORY CACHE UPDATE ]' % graphite_banner
    
    
    @staticmethod
    def _get_parent_proc_start_time():
        try:
            import psutil
            
            return psutil.Process().parent().create_time()
        except:
            hz = os.sysconf(os.sysconf_names['SC_CLK_TCK'])
            boot_timestamp = 0
            with open('/proc/stat') as proc_file:
                for line in proc_file.readlines():
                    # noinspection SpellCheckingInspection
                    if line.startswith('btime'):
                        boot_timestamp = int(line.split()[1])
                        break
            
            with open('/proc/%s/stat' % os.getppid()) as proc_file:
                process_stats = proc_file.read().split()
            age_from_boot_jiffies = int(process_stats[21])
            age_from_boot_timestamp = age_from_boot_jiffies / hz
            age_timestamp = boot_timestamp + age_from_boot_timestamp
            return age_timestamp
    
    
    def _copy_configuration_file_in_cache(self, file_name):
        base_name = os.path.basename(file_name)
        cache_name = os.path.join(self._cache_dir, base_name)
        
        if not os.path.exists(file_name):
            return file_name
        
        with self._inter_process_lock:
            if os.path.exists(cache_name):
                cache_last_update = os.stat(cache_name).st_mtime
                if cache_last_update >= self._httpd_start_time:
                    return cache_name
            
            shutil.copy(file_name, cache_name)
            os.utime(cache_name, (time.time(), time.time()))  # <- this may be useless, just to be sure modification is NOW
        return cache_name
    
    
    @classmethod
    def get_cache_invalidation_file(cls):
        return cls._cache_invalidation_file
    
    
    def _reset_cache(self):
        with self._elements_mapping_lock:
            self._elements_mapping.clear()
            self._last_update_time = 0
            # Let the readers know we did reset the cache
            self._mongodb_inventory_reader.cache_was_reset()
            self._http_inventory_reader.cache_was_reset()
    
    
    def check_for_file_update(self, file_name, file_type='configuration'):
        if not os.path.exists(file_name):
            return
        stats = os.stat(file_name)
        file_last_update = stats.st_mtime
        if file_last_update > self._cache_invalidation_date:
            self._cache_invalidation_date = file_last_update
            log.info('%s Resetting the cache because %s file %s has unread data' % (self._graphite_cache_banner % threading.current_thread().name, file_type, file_name))
            self._reset_cache()
    
    
    def _look_at_cache_invalidation(self):
        
        self.check_for_file_update(self._http_inventory_reader.get_inventory_file_name(), 'data')
        
        # If the path does not exist, do nothing more
        if not os.path.exists(self._cache_invalidation_file):
            return
        
        stats = os.stat(self._cache_invalidation_file)
        last_modification_time = stats.st_mtime
        if last_modification_time > self._cache_invalidation_date:
            self._cache_invalidation_date = last_modification_time
            log.info('%s Resetting the cache because the invalidation file %s has changed, meaning a new configuration was pushed.' % (self._graphite_cache_banner % threading.current_thread().name, self._cache_invalidation_file))
            self._reset_cache()
    
    
    def update_hosts_checks_mapping(self):
        with self._elements_mapping_lock:
            self._do_update_hosts_checks_mapping()
    
    
    # same but with lock
    def _do_update_hosts_checks_mapping(self):
        now = int(time.time())
        
        # Check if the cache should be invalidated (new conf)
        self._look_at_cache_invalidation()
        
        # We get back in time? reset all
        if now < self._last_update_time:
            self._last_update_time = 0
        
        # Data is present, no need to reload
        if 0 < self._last_update_time and self._elements_mapping:
            return
        
        self._last_update_time = now
        
        if self._http_inventory_reader.inventory_data_from_shinken_lookup(self._cache_invalidation_date):
            return
        
        log.info('%s Could not join Shinken to get inventory data, fallback to MongoDB' % graphite_banner)
        try:
            self._mongodb_inventory_reader.inventory_data_from_shinken_lookup(self._cache_invalidation_date)
        except:
            self._last_update_time = 0
    
    
    def find_all_hosts_matching(self, server_expr, absolute_root):
        # t0 = time.time()
        self.update_hosts_checks_mapping()
        
        res = set()
        with self._elements_mapping_lock:
            matching_hosts = fnmatch.filter(self._elements_mapping.keys(), server_expr)
            self._log_about(server_expr, '%s [CACHE READ] Looking for host that match %s => %s' % (self._graphite_banner, server_expr, ', '.join(matching_hosts)))
            for name in matching_hosts:
                host_uuid = self._elements_mapping[name]['uuid']
                # If the uuid directory is missing, means that this host got no metrics
                pth = os.path.join(absolute_root, host_uuid)
                if not os.path.exists(pth):
                    continue
                res.add((name, host_uuid))
        # log.info('%s [PERF] Look at all host matching a pattern: %.3fs' % (self._graphite_banner, time.time() - t0))
        return res
    
    
    def find_matching_host_check_paths(self, matching_hosts, check_pattern, absolute_root):
        # t0 = time.time()
        res = []
        with self._elements_mapping_lock:
            for (host_name, host_uuid) in matching_hosts:
                checks = self._elements_mapping[host_name]['checks']
                self._log_about(host_name, '%s [CACHE READ] In cache checks for host:: %s => %s' % (self._graphite_banner, host_name, ', '.join(checks.keys())))
                matching_checks_names = fnmatch.filter(checks.keys(), check_pattern)
                self._log_about(host_name, '%s [CACHE READ] In cache check that match pattern %s for host %s => %s' % (self._graphite_banner, check_pattern, host_name, ', '.join(matching_checks_names)))
                for matching_check_name in matching_checks_names:
                    check_uuid = checks[matching_check_name]['uuid']
                    pth = os.path.join(host_uuid, check_uuid)
                    
                    # If this check has no metric, we are not interested about it
                    dir_path = os.path.join(absolute_root, pth)
                    if not os.path.exists(dir_path):
                        continue
                    res.append((host_name, matching_check_name, pth))
        # log.info('_find_matching_host_check_paths:: time: %.3f' % (time.time() - t0))
        return res
