#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

# It was named aaaa for be the first in automatic import

import copy
import collections
import socket
import ssl
import sys
import queue

from shinken.thread_helper import patch_thread_name

try:
    # noinspection PyPackageRequirements
    from cheroot.workers.threadpool import ThreadPool, WorkerThread
    # noinspection PyPackageRequirements, PyProtectedMember
    from cheroot._compat import IS_ABOVE_OPENSSL10
    # noinspection PyPackageRequirements
    from cheroot import errors
    # noinspection PyPackageRequirements, PyProtectedMember
    from cheroot.ssl.builtin import BuiltinSSLAdapter, _assert_ssl_exc_contains
    # noinspection PyPackageRequirements
    from cheroot.server import HTTPServer, HTTPConnection
    
    
    # noinspection PyUnusedLocal
    def get_environ(self, sock):
        """Create WSGI environ entries to be merged into each request."""
        cipher = sock.cipher()
        # noinspection SpellCheckingInspection
        ssl_environ = {
            'wsgi.url_scheme'      : 'https',
            'HTTPS'                : 'on',
            'SSL_PROTOCOL'         : lambda x: sock.cipher()[1],
            'SSL_CIPHER'           : lambda x: sock.cipher()[0],
            'SSL_CIPHER_EXPORT'    : '',
            'SSL_CIPHER_USEKEYSIZE': lambda x: sock.cipher()[2],
            'SSL_VERSION_INTERFACE': '%s Python/%s' % (
                HTTPServer.version, sys.version,
            ),
            'SSL_VERSION_LIBRARY'  : ssl.OPENSSL_VERSION,
            'SSL_CLIENT_VERIFY'    : 'NONE',
            # 'NONE' - client did not provide a cert (overridden below)
        }
        return ssl_environ
    
    
    def wrap(self, sock):
        """Wrap and return the given socket, plus WSGI environ entries."""
        try:
            s = self.context.wrap_socket(
                sock, do_handshake_on_connect=False, server_side=True,
            )
        except (
                ssl.SSLEOFError,
                ssl.SSLZeroReturnError,
        ) as tls_connection_drop_error:
            raise errors.FatalSSLAlert(
                *tls_connection_drop_error.args,
            ) from tls_connection_drop_error
        except ssl.SSLError as generic_tls_error:
            peer_speaks_plain_http_over_https = (
                    generic_tls_error.errno == ssl.SSL_ERROR_SSL and
                    _assert_ssl_exc_contains(generic_tls_error, 'http request')
            )
            if peer_speaks_plain_http_over_https:
                reraised_connection_drop_exc_cls = errors.NoSSLError
            else:
                reraised_connection_drop_exc_cls = errors.FatalSSLAlert
            
            raise reraised_connection_drop_exc_cls(
                *generic_tls_error.args,
            ) from generic_tls_error
        except OSError as tcp_connection_drop_error:
            raise errors.FatalSSLAlert(
                *tcp_connection_drop_error.args,
            ) from tcp_connection_drop_error
        
        return s, self.get_environ(s)
    
    
    def threadpool__init__(self, server, min=10, max=-1, accepted_queue_size=-1, accepted_queue_timeout=10, ):
        """Initialize HTTP requests queue instance.

        Args:
            server (cheroot.server.HTTPServer): web server object
                receiving this request
            min (int): minimum number of worker threads
            max (int): maximum number of worker threads (-1/inf for no max)
            accepted_queue_size (int): maximum number of active
                requests in queue
            accepted_queue_timeout (int): timeout for putting request
                into queue

        :raises ValueError: if the min/max values are invalid
        :raises TypeError: if the max is not an integer or inf
        """
        if min < 1:
            raise ValueError(f'min={min!s} must be > 0')
        
        if max == float('inf'):
            pass
        elif not isinstance(max, int) or max == 0:
            raise TypeError(
                'Expected an integer or the infinity value for the `max` '
                f'argument but got {max!r}.',
            )
        elif max < 0:
            max = float('inf')
        
        if max < min:
            raise ValueError(
                f'max={max!s} must be > min={min!s} (or infinity for no max)',
            )
        
        self.server = server
        self.min = min
        self.max = max
        self._threads = []
        self._queue = queue.Queue(maxsize=accepted_queue_size)
        self._queue_put_timeout = accepted_queue_timeout
        self._pending_shutdowns = collections.deque()
    
    
    def threadpool_get(self):
        conn: 'HTTPConnection' = self._queue.get()
        if conn and hasattr(conn, 'socket') and hasattr(conn.socket, 'do_handshake') and not getattr(conn.socket, 'shinken_handshake_done', False):
            conn.socket.shinken_handshake_done = True
            try:
                conn.socket.do_handshake()
                do_shutdown = False
            except ssl.SSLError as cheroot_ssl_error:
                do_shutdown = True
                self.server.error_log(f'''SSL handshake for {getattr(conn, 'remote_addr', '---')}:{getattr(conn, 'remote_port', '-')} failed with SSL error:{cheroot_ssl_error}''')
                
                # Try to send HTTP 400 status for plain text queries
                peer_speaks_plain_http_over_https = (cheroot_ssl_error.errno == ssl.SSL_ERROR_SSL and _assert_ssl_exc_contains(cheroot_ssl_error, 'http request'))
                if peer_speaks_plain_http_over_https:
                    msg = f'''The client {getattr(conn, 'remote_addr', '---')}:{getattr(conn, 'remote_port', '-')} sent a plain HTTP request, but this server only speaks HTTPS on this port.'''
                    buf = [
                        '%s 400 Bad Request\r\n' % self.server.protocol,
                        'Content-Length: %s\r\n' % len(msg),
                        'Content-Type: text/plain\r\n\r\n',
                        msg,
                    ]
                    
                    try:
                        # - writing on conn.socket attempt to use non-initialized SSL layer and raises exception: EOF occurred in violation of protocol (_ssl.c:2427)
                        # - conn.socket.unwrap raises exception: [SSL: SHUTDOWN_WHILE_IN_INIT] shutdown while in init (_ssl.c:2706)
                        # - this horrible hack returns a non SSL socket
                        conn.socket = socket.socket(fileno=conn.socket.detach())
                        conn.socket.sendall(''.join(buf).encode('ISO-8859-1'))
                    except Exception as write_exception:
                        self.server.error_log(f'{msg} Following error occurred while replying:{write_exception}')
            
            except Exception as cheroot_error:
                do_shutdown = True
                self.server.error_log(f'''SSL handshake for {getattr(conn, 'remote_addr', '---')}:{getattr(conn, 'remote_port', '-')} failed with error:{cheroot_error}''')
            
            if do_shutdown:
                try:
                    conn.socket.shutdown(socket.SHUT_RDWR)
                except:
                    pass
        return conn
    
    
    def threadpool_spawn_worker(self):
        worker = WorkerThread(self.server)
        worker.name = (
            'CP {worker_name!s}'.
            format(worker_name=worker.name)
        )
        worker.start()
        return worker
    
    
    BuiltinSSLAdapter.get_environ = get_environ
    BuiltinSSLAdapter.wrap = wrap
    ThreadPool.__init__ = threadpool__init__
    ThreadPool.get = threadpool_get
    ThreadPool._spawn_worker = threadpool_spawn_worker
except ImportError:
    ThreadPool = None
    WorkerThread = None
    errors = None
    BuiltinSSLAdapter = None
    HTTPServer = None
    HTTPConnection = None
    MakeFile = None
    
    
    def _assert_ssl_exc_contains(*_args):
        return None

try:
    # noinspection PyProtectedMember, PyPackageRequirements
    from cheroot import _compat
    
    # noinspection SpellCheckingInspection
    orig_ntob = _compat.ntob
    
    
    # noinspection SpellCheckingInspection
    def ntob(n, encoding='UTF-8'):
        return orig_ntob(n, encoding)
    
    
    # noinspection SpellCheckingInspection
    _compat.ntob = ntob
except:
    _compat = None

try:
    # SEF-11578 - cheroot ate all CPU and stopped responding in Synchronizer main daemon
    import selectors
    import threading
    import shinkensolutions.subprocess_helper.fork_setup
    
    # noinspection PyProtectedMember,PyUnresolvedReferences,PyPackageRequirements
    from cheroot.connections import _ThreadsafeSelector
    
    
    def patch_cheroot_selector():
        original_init = _ThreadsafeSelector.__init__
        
        
        def patched_init(self):
            original_init(self)
            shinkensolutions.subprocess_helper.fork_setup.cheroot_selector_list.append(self)
        
        
        _ThreadsafeSelector.__init__ = patched_init
    
    
    patch_cheroot_selector()
except:
    pass

try:
    # SEF-11985 - Fix circular reference on HTTPConnection instantiation.
    import contextlib
    from cheroot.server import HTTPConnection
    
    
    def patch_cheroot_http_connection():
        # noinspection PyTypeChecker
        original_init = HTTPConnection.__init__
        
        
        def patched_init(self: 'HTTPConnection', *args, **kwargs) -> None:
            original_init(self, *args, **kwargs)
            # Delete method re-assignment.
            # NOTE: peer creds is a feature of UNIX sockets. We use cheroot only with IPv4/v6 sockets.
            with contextlib.suppress(AttributeError):
                del self.get_peer_creds
            with contextlib.suppress(AttributeError):
                del self.resolve_peer_creds
        
        
        HTTPConnection.__init__ = patched_init
    
    
    patch_cheroot_http_connection()
except:
    pass

try:
    # SEF-12122 - We may have "SystemError: Objects/tupleobject.c:927: bad argument to internal function" here.
    # There is a race condition between a tuple object creation and the call of gc.get_objects() made by MemoryStats.query_memory_stats().
    def patch_deepcopy(x, memo=None, *, _original_deepcopy=copy.deepcopy):
        try:
            return _original_deepcopy(x, memo)
        except SystemError:
            # Retry operation only once.
            return _original_deepcopy(x, memo)
    
    
    copy.deepcopy = patch_deepcopy
except:
    pass

patch_thread_name()
