#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import threading
import time
import traceback
from collections import deque

from .log import LoggerFactory
from .misc.type_hint import TYPE_CHECKING
from .runtime_stats.cpu_stats import cpu_stats_helper

if TYPE_CHECKING:
    from .brok import Brok
    from .scheduler import Scheduler

logger = LoggerFactory.get_logger('INITIAL BROKS')
logger_registering = logger.get_sub_part('REGISTERING')
logger_generating = logger.get_sub_part('GENERATING')
logger_start = logger_generating.get_sub_part('START')
logger_done = logger_generating.get_sub_part('DONE')


# Class to generate initial broks, if possible only once for numerous brokers
# if they are asking them at the same time
# NOTE: we are NOT stocking Broks, only ask Scheduler, and push them to Brokers objects, then
#       all the memory is lost for us
class InitialBroksFactory:
    def __init__(self, scheduler):
        # type: (Scheduler) -> None
        self._scheduler = scheduler
        self._lock = threading.Condition(threading.RLock())
        self._requests = set()
        self._completed = set()
        self._broks = deque()  # type: deque[Brok]
        self._thread = None
    
    
    def register_for_generation(self, broker_name):
        # type: (str) -> None
        with self._lock:
            # If already asked in the past, remove the finish, and start a new request
            if broker_name in self._completed:
                self._completed.remove(broker_name)
            if broker_name not in self._requests:
                self._requests.add(broker_name)
                self._lock.notify_all()
                logger_registering.info('[ %s ] The Broker is registering for initial broks generation. ( Currently %d registered )' % (broker_name, len(self._requests)))
    
    
    def is_completed(self, broker_name):
        # type: (str) -> bool
        with self._lock:
            r = broker_name in self._completed
            if r:
                logger_done.info('[ %s ] The broker is warned that the generation is done and can be GET.' % broker_name)
            return r
    
    
    def wait_for_completion_for_broker(self, broker_name: 'str') -> None:
        with self._lock:
            while broker_name not in self._completed:
                self._lock.wait()
        logger_done.info(f'[ {broker_name} ] The broker is warned that initial broks generation has completed.')
    
    
    def _have_requests(self):
        # type () -> bool
        with self._lock:
            return len(self._requests) != 0
    
    
    def _wait_for_requests(self):
        with self._lock:
            while not self._requests:
                self._lock.wait()
    
    
    # We already know there is requests, so launch Brok generation
    def _launch_generation(self):
        # type () -> None
        
        # Note our thread current consumption CPU
        start_snap = cpu_stats_helper.get_thread_cpu_snapshot()
        
        logger_start.info('Starting initial broks generation')
        
        start = time.time()
        
        # NOTE: all of this is LOCAL so new brokers can register during this time
        with self._scheduler.sched_daemon.satellite_lock:
            # Maybe the scheduler just got disabled, then retry until it's back again
            broks = []
            while not broks:
                try:
                    broks = self._scheduler.get_initial_broks()
                except Exception as exp:
                    logger_generating.warning('We are generating Broks but the scheduler seems to be unavailable, retrying: %s' % exp)
                    self._scheduler.sched_daemon.satellite_lock.wait(1.0)
            
            # Fast switch for requesters
            with self._lock:
                brokers_to_give = self._requests
                self._requests = set()  # reset it
            
            # Assert we have the brokers declared or have new broks ready
            # IMPORTANT: we MUST have a deque() by brokers, because each one
            #            will pop() it independently
            for broker_name in brokers_to_give:
                self._scheduler.reset_broker_entry_for_initial_broks(broker_name)
            
            # Now give back the broks to all our brokers
            # NOTE: we are handling the sat lock BEFORE the broker locks
            # NOTE2: out of the
            for brok in broks:
                for broker_name in brokers_to_give:
                    # NOTE: we have the sat lock, so broker MUST exist here
                    self._scheduler.add_Brok(brok, broker_name)
            
            # We have finished, so let our requesters know it
            with self._lock:
                self._completed = brokers_to_give
                self._lock.notify_all()
                
                elapsed_time = time.time() - start
                logger_done.info('[ Elapsed time=%.3fs ] %d initial broks are generated for %s brokers:' % (
                    elapsed_time, len(broks), len(self._completed)))
                for broker_name in self._completed:
                    logger_done.info(' - %s' % broker_name)
                # NOTE: the cpu consumption log is hard to be understood by the end user, so we are putting it in DEBUG level
                #       for support only
                
                logger_done.debug('[PERF] %s [ %d initial broks ]' % (start_snap.get_diff(), len(broks)))
    
    
    def launch_main_thread(self):
        # Do not launch twice
        # NOTE: if dead, will kill the daemon anyway
        if self._thread is not None:
            return
        self._thread = threading.Thread(None, target=self._main, name='InitialBroksFactory')
        self._thread.daemon = True
        self._thread.start()
    
    
    def _main(self):
        try:
            # Simple: if we have requests, launch the generation, new requests can be stacked while
            #         we are generating these
            while True:
                self._wait_for_requests()
                # We cheat a bit here: let a 1s sleep, so another brokers can have enough time to register
                # even on very small configuration
                time.sleep(1.0)
                self._launch_generation()
        
        except Exception:  # noqa: yes we want all
            logger.error('The thread for initial broks generation did crash with a FATAL error. Stopping the daemon: %s' % traceback.format_exc())
            os._exit(2)  # noqa: kill all others threads too
