#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.


import contextlib
import os
import shutil
import sys
import time
import uuid

from shinken.compat import cPickle
from shinken.misc.os_utils import safe_write_binary_file_and_force_mtime, make_file_hidden
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.locking.shinken_locking.shinken_interprocess_rlock import ShinkenInterProcessRLock, set_ownership

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional, Any

INTERNAL_KEY = ('_lock_ex', '_lock_shared', '_data', '_data_dir_name', '_data_file_name', '_last_load_time', '_in_update_context')


def get_share_item_file_path() -> 'str':
    if sys.platform.startswith('win'):
        import tempfile
        
        tempfile.tempdir = r'c:\shinken\var\temp'
        return os.path.join(tempfile.gettempdir(), 'share_item')
    
    else:
        import pwd
        
        cur_name = None
        
        # TODO: refactor this code duplication from log.py, share_item.py and os_helper.py
        # Try to workaround strange bug with pwd module raising KeyError even with existing user
        for workaround_try_nb in range(5):
            try:
                cur_name = pwd.getpwuid(os.getuid()).pw_name
                break
            except:
                if workaround_try_nb == 4:
                    raise
        if os.getuid() != 0 and cur_name != 'shinken':
            return f'/dev/shm/shinken_{pwd.getpwuid(os.getuid()).pw_name}/share_item'
        else:
            return '/dev/shm/shinken/share_item'


SHARE_ITEM_SYNC_FILE_PATH = get_share_item_file_path()


class MockShareItem:
    @staticmethod
    def acquire():
        return
    
    
    @staticmethod
    def release():
        return
    
    
    def __enter__(self):
        self.acquire()
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.release()


class ShareItem:
    @staticmethod
    def _build_dirname_for_daemon(daemon_type: 'str|None' = None, daemon_id: 'int|str|None' = None, module_name: 'str|None' = None) -> 'str':
        dir_name = ''
        if daemon_type:
            dir_name = os.path.join('daemons', f'{daemon_type}')
            if daemon_id is not None and daemon_id != '':
                dir_name = os.path.join(dir_name, f'{daemon_id}')
            if module_name:
                dir_name = os.path.join(dir_name, 'modules', f'{module_name}')
        if dir_name:
            return os.path.join(SHARE_ITEM_SYNC_FILE_PATH, dir_name)
        else:
            return ''
    
    
    def get_dir_path(self):
        return self._data_dir_name
    
    
    def delete_dir(self):
        folder_path = self.get_dir_path()
        if os.path.exists(folder_path):
            shutil.rmtree(folder_path)
    
    
    def destroy(self):
        try:
            self._lock_ex.destroy()
        except:
            pass
        try:
            self._lock_shared.destroy()
        except:
            pass
        try:
            os.unlink(self._data_file_name)
        except:
            pass
        try:
            os.rmdir(os.path.dirname(self._data_file_name))
        except:
            pass
    
    
    def __init__(self, key_name: 'Optional[str]' = None, reinit: 'bool' = False, dir_name: 'Optional[str]' = None, daemon_type: 'str|None' = None, daemon_id: 'int|str|None' = None, module_name: 'str|None' = None) -> None:
        if key_name:
            key_name = key_name.replace(os.sep, '')
            key_name = key_name.replace('/', '')
            key_name = os.path.splitdrive(key_name)[1]
        if not key_name:
            key_name = '%s.%s-%s' % (type(self).__module__, type(self).__name__, uuid.uuid4().hex)
        if dir_name:
            dir_name = os.path.splitdrive(dir_name)[1]
            if not dir_name or dir_name[1] != os.sep:
                dir_name = None
        if not dir_name:
            dir_name = self._build_dirname_for_daemon(daemon_type=daemon_type, daemon_id=daemon_id, module_name=module_name)
        if not dir_name:
            dir_name = os.path.join(SHARE_ITEM_SYNC_FILE_PATH, type(self).__name__)
        self._data_dir_name = dir_name
        self._data = {}
        self._data_file_name = os.path.join(self._data_dir_name, key_name)
        self._last_load_time = 0
        self._in_update_context = False
        # self.dir_create()
        self._lock_ex, self._lock_shared = self._create_lock(self._data_file_name)
        if reinit:
            self.data_file_remove()
        self.init_data_file()
    
    
    @staticmethod
    def _create_lock(data_file_name):
        lock_filename = make_file_hidden('%s.lock' % data_file_name)
        return ShinkenInterProcessRLock(lock_filename), ShinkenInterProcessRLock(lock_filename, True)
    
    
    def init_data_file(self):
        # type: () -> None
        if not os.path.exists(self._data_file_name):
            self._save()
    
    
    def get_lock(self, shared=False):
        return self._lock_shared if shared else self._lock_ex
    
    
    def data_file_remove(self):
        # type: () -> None
        try:
            with self.get_lock():
                # On Windows the clock is not enough precise, so we must wait here to force a change on file
                if os.name == 'nt':
                    time.sleep(0.01)
                
                # the open / close as wb will clean the file.
                with open(self._data_file_name, 'wb'):
                    now = time.time()
                # Forcing the modification time to now, this was benched, and we lost ~4% process time for 1M executions
                # The loss is acceptable because without this, the share_item would have update issues when really fast accesses
                os.utime(self._data_file_name, (now, now))
        except:
            pass
    
    
    def __str__(self):
        # type: () -> str
        return 'ShareItem %s/%s: [%s]' % (type(self).__name__, self._data_file_name, self._data)
    
    
    def __repr__(self):
        # type: () -> str
        return self.__str__()
    
    
    def __getattr__(self, key):
        # type: (str) -> Any
        if key in INTERNAL_KEY:
            return super(ShareItem, self).__getattribute__(key)
        self._reload(need_lock=True, shared_lock=True)
        try:
            return self._data[key]
        except KeyError as exc:
            raise AttributeError('%s object has no attribute [%s]' % (type(self).__name__, key)) from exc
    
    
    def __setattr__(self, key, value):
        # type: (str, str) -> None
        if key in INTERNAL_KEY:
            return super(ShareItem, self).__setattr__(key, value)
        with self.get_lock():
            self._reload()
            self._data[key] = value
            if not self._in_update_context:
                self._save()
    
    
    def __enter__(self):
        # type: () -> None
        self.get_lock().acquire()
        self._in_update_context = True
    
    
    def __exit__(self, _type, value, traceback):
        # type: (str, Any, str) -> None
        self._in_update_context = False
        self._save()
        self.get_lock().release()
    
    
    def get_all_attr_read_only(self):
        # type: () -> Dict
        with self.get_lock(shared=True):
            self._reload()
            return self._data.copy()
    
    
    def clear_all_attr(self):
        # type: () -> None
        with self.get_lock():
            self._data.clear()
            self._save()
    
    
    def _reload(self, need_lock=False, shared_lock=False):
        # type: (bool, bool) -> None
        last_modification_time = os.stat(self._data_file_name).st_mtime
        if last_modification_time == self._last_load_time:
            return
        with contextlib.ExitStack() as exit_stack:
            if need_lock:
                exit_stack.enter_context(self.get_lock(shared=shared_lock))
            # Be careful, on Windows the modification date on files is not precise enough and this condition could fail
            # when accessing the same file too quickly (like in unit tests)
            if self._last_load_time < last_modification_time:
                if os.stat(self._data_file_name).st_size > 0:
                    with open(self._data_file_name, 'rb') as data_file:
                        self._deserialize(data_file)
                else:
                    self._data = {}
                self._last_load_time = last_modification_time
    
    
    def _deserialize(self, file_descriptor):
        self._data = cPickle.load(file_descriptor)
    
    
    def _serialize(self):
        # type: () -> bytes
        return cPickle.dumps(self._data, cPickle.HIGHEST_PROTOCOL)
    
    
    def _save(self):
        # type: () -> None
        # self.dir_create()
        with self.get_lock():
            file_content = self._serialize()
            safe_write_binary_file_and_force_mtime(self._data_file_name, file_content)
            set_ownership(self._data_file_name)
            self._last_load_time = os.stat(self._data_file_name).st_mtime
