#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2022:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import copy
import imp
import inspect
import itertools
import json
import os
import sys
import time
from threading import RLock, Thread

from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.autoloaded_module import AutoLoadedModule
from shinken.modules.base_module.basemodule import BaseModule, ModuleState, INSTANCE_INIT_STATE
from shinken.modules.modules_stats import NO_MODULES_WANTED
from shinken.objects.module import Modules, Module as ModuleConfiguration, FIRST_MODULE_ID_FOR_MODULES_NOT_INSTANTIATED_BY_ARBITER
from shinken.subprocess_helper.after_fork_cleanup import AFTER_FORK_CLEANUP_PART_LOGGER, AFTER_FORK_CLEANUP_METHOD
from shinken.withworkersandinventorymodule import WithWorkersAndInventoryModule

if TYPE_CHECKING:
    import types
    from shinken.ipc.shinken_queue.shinken_queue import ShinkenQueue
    from shinken.misc.type_hint import Callable, Optional, Any
    from shinken.inter_daemon_message import InterDaemonMessage
    from shinken.log import PartLogger


# We need to manage pre-2.0 module types with _ into the new 2.0 - mode
def uniform_module_type(s: str) -> str:
    return s.replace('_', '-')


class StateCleaner(Thread):
    # Used to clean up the modules state (self.last_restarts)
    
    CLEAN_STEP = 60  # every 60 seconds
    
    
    def __init__(self, manager: 'ModulesManager') -> None:
        Thread.__init__(self)
        self.daemon = True  # Will be killed at application close
        self._manager = manager
    
    
    def run(self):
        while True:
            time.sleep(StateCleaner.CLEAN_STEP)
            self._manager.clean_modules_states()


class ModulesManager:
    # This class is used to manage modules and call callback
    
    def __init__(self, modules_type: str, modules_path: str, modules: 'Optional[Modules]', daemon_display_name: str = 'UNSET', with_thread: bool = True, *, keep_modules_info: bool = False) -> None:
        self.logger = LoggerFactory.get_logger('MODULES-MANAGER')
        self.logger_module_code_loading = self.logger.get_sub_part('CODE-LOADING')
        self.logger_module_instance_creation = self.logger.get_sub_part('CREATION')
        self.logger_modules_instance_crash = self.logger.get_sub_part('CRASH')
        self.logger_modules_change = self.logger.get_sub_part('UPDATE')
        self.logger_modules_stop = self.logger.get_sub_part('SHUTDOWN')
        self._modules: 'Optional[Modules]' = None
        self._auto_loaded_modules_conf: 'dict[str,ModuleConfiguration]' = {}
        self._auto_loaded_modules_auto_conf: 'dict[str,ModuleConfiguration]' = {}
        self.modules_path = modules_path
        self.modules_type = modules_type
        self.modules = modules
        self.allowed_types = [uniform_module_type(plug.module_type) for plug in modules if hasattr(plug, 'module_type')] if modules else []
        self.imported_modules: 'list[types.ModuleType]' = []
        self.modules_assoc = {}
        self.instances: 'dict[str,BaseModule]' = {}
        self.to_restart = {}
        self.last_restarts = {}
        self.last_restarts_lock = RLock()
        self.instances_lock = RLock()
        self.last_restarts_keep = 86400  # lasts 24h (86400 seconds) of restarts
        self.max_queue_size = 0
        self.errors = []
        self.daemon_display_name = daemon_display_name
        self.daemon_name = None
        self.daemon_id: 'Optional[int]' = None
        self._all_modules_info: 'dict[str, dict]' = {}
        self._keep_modules_info = keep_modules_info
        if with_thread:
            StateCleaner(self).start()
    
    
    @property
    def modules(self):
        return self._modules
    
    
    @modules.setter
    def modules(self, modules: 'Optional[Modules]'):
        self._modules = self._add_auto_loaded_modules_to_modules(modules)
        self._set_modules_ordering_indexes()
    
    
    def _add_auto_loaded_modules_to_modules(self, modules: 'Optional[Modules]', reset_auto_loaded_modules_configuration: 'bool' = False) -> 'Optional[Modules]':
        if self._auto_loaded_modules_conf:
            if modules:
                auto_loaded_modules_types = {m_type for m_type in self._auto_loaded_modules_conf.keys()}
                conf_modules_types = {m.get_type() for m in modules}
                to_remove_from_auto_load = auto_loaded_modules_types.intersection(conf_modules_types)
                if reset_auto_loaded_modules_configuration:
                    still_auto_loaded_modules = [m_conf for m_type, m_conf in self._auto_loaded_modules_auto_conf.items() if m_type not in to_remove_from_auto_load]
                else:
                    still_auto_loaded_modules = [m_conf for m_type, m_conf in self._auto_loaded_modules_conf.items() if m_type not in to_remove_from_auto_load]
                if still_auto_loaded_modules:
                    modules = Modules([m for m in modules] + [m for m in still_auto_loaded_modules])
            else:
                if reset_auto_loaded_modules_configuration:
                    modules = Modules(self._auto_loaded_modules_auto_conf.values())
                else:
                    modules = Modules(self._auto_loaded_modules_conf.values())
        return modules
    
    
    def _set_modules_ordering_indexes(self):
        self._idx_modules_order_by_name = {}
        if self._modules:
            for index, module in enumerate(self._modules):
                self._idx_modules_order_by_name[module.module_name] = index
    
    
    def set_properties_modules_configuration(self, conf_modules: 'Modules') -> None:
        if not self._keep_modules_info:
            raise Exception(f'You cannot use method "set_properties_modules_configuration" without "keep_modules_info" option.')
        for module in conf_modules:
            module_properties = self._all_modules_info.get(uniform_module_type(module.module_type), {})
            module.properties = module_properties.copy()
    
    
    def unload_all_modules_info(self) -> None:
        self._all_modules_info = {}
    
    
    # Set the modules requested for this manager
    def set_modules(self, conf_modules: 'Modules') -> None:
        self.modules = conf_modules
        self.errors = []
        self.allowed_types = [uniform_module_type(mod.module_type) for mod in conf_modules if hasattr(mod, 'module_type')]
        updated_modules_names = []
        if self._auto_loaded_modules_conf:
            with self.instances_lock:
                if self.instances:
                    for auto_loaded_module_type, auto_loaded_module_conf in self._auto_loaded_modules_conf.items():
                        auto_loaded_module_is_already_up_to_date = False
                        auto_loaded_module_hash = auto_loaded_module_conf.hash
                        for instance in self.instances.values():
                            if instance.get_module_type() == auto_loaded_module_type and instance.get_module_configuration_hash() == auto_loaded_module_hash:
                                auto_loaded_module_is_already_up_to_date = True
                                break
                        if not auto_loaded_module_is_already_up_to_date:
                            updated_modules_names.append(auto_loaded_module_conf.get_name())
                else:
                    updated_modules_names = [module.get_name() for module in self._auto_loaded_modules_conf.values()]
        
        self._display_module_tree(added_modules_names=[module.get_name() for module in conf_modules], updated_modules_names=updated_modules_names, deleted_modules_names=[])
    
    
    def _find_module_by_name(self, module_name: str) -> 'Optional[ModuleConfiguration]':
        for module in self.modules:
            if module.get_name() == module_name:
                return module
        return None
    
    
    # When module has changed in a way, we print what we currently have as module
    def _display_module_tree(self, added_modules_names, updated_modules_names, deleted_modules_names):
        # If no modules, don't need to display anything
        if len(self.modules) == 0 and len(deleted_modules_names) == 0:
            self.logger_modules_change.info('No module')
            return
        
        # Maybe there was no change in modules
        if len(added_modules_names) == 0 and len(updated_modules_names) == 0 and len(deleted_modules_names) == 0:
            self.logger_modules_change.info('No module has changed')
            return
        
        # Ok something has changed
        all_module_names = [module.get_name() for module in self.modules]
        all_module_names.extend(deleted_modules_names)  # all = current have + deleted
        all_module_names = sorted(set(all_module_names))
        self.logger_modules_change.info('Module modifications:')
        for module_name in all_module_names:
            module = self._find_module_by_name(module_name)
            if module is None:
                what = '-> stopped   ( removed )'
            elif module_name in added_modules_names:
                what = '-> started   ( new )'
            elif module_name in updated_modules_names:
                what = '-> restarted ( configuration change )'
            else:  # untouched
                what = ''
            self.logger_modules_change.info(f'   - {module_name:<20s} {what}')
            if module is None:  # was deleted, don't print submodules
                continue
            # Maybe there is no submodules to print
            if len(module.modules) == 0:
                continue
            
            # IMPORTANT: DO NOT SORT submodules, because for webui for example the
            #            order IS important (for auth modules for example)
            # NOTE: see this is only useful in DEBUG phase
            for sub_module in module.modules:
                self.logger_modules_change.debug(f'      * {sub_module.get_name()}')
    
    
    def get_module_by_name(self, module_name: str) -> 'BaseModule':
        inst = self.instances.get(module_name, None)
        return inst
    
    
    # We get some new modules, and we must look which one are:
    # * no more there and should be stopped
    # * the new ones, we need to start
    # * the one that has changed (look at hash property) that we need to restart
    def update_modules(self, modules: Modules) -> None:
        with self.instances_lock:
            changed_to_restart = []
            self.errors = []
            
            modules = self._add_auto_loaded_modules_to_modules(modules, reset_auto_loaded_modules_configuration=True)
            
            # First try to look which one are no more need
            new_modules_name = {m.get_name() for m in modules}
            old_modules_name = {m.get_name() for m in self.modules} if self.modules else set()
            
            old_modules_to_delete_name = old_modules_name - new_modules_name
            new_modules_to_start_name = new_modules_name - old_modules_name
            
            # In the common ones, look at the one that has changed, and so we will have to delete
            module_old_and_new_name = old_modules_name & new_modules_name
            
            for _current_module_name in module_old_and_new_name:
                _logger = self.logger_modules_change.get_sub_part(_current_module_name, register=False)
                old_one = self._find_module_by_name(_current_module_name)
                new_one = next((m for m in modules if m.get_name() == _current_module_name), None)
                
                if old_one is None or new_one is None:
                    _logger.error(f'''Cannot update '{_current_module_name}' module. Please contact your Support.''')
                    continue
                # we compare hash to know if something did change, if so we will need to kill/restart it
                if old_one.hash != new_one.hash:
                    _logger.info('Restarting module because its configuration has changed.')
                    changed_to_restart.append(new_one)
            
            for _current_module_name in old_modules_to_delete_name:
                _logger = self.logger_modules_change.get_sub_part(_current_module_name, register=False)
                _logger.info('Removing module because it has been removed from configuration.')
                self.__remove_instance(_current_module_name, self.logger_modules_change)
            
            for _current_module_name in new_modules_to_start_name:
                _logger = self.logger_modules_change.get_sub_part(_current_module_name, register=False)
                _logger.info('Launched as a NEW module as it was added to configuration.')
                mod_conf = next((m for m in modules if m.get_name() == _current_module_name), None)
                # Got a new module configuration, need to start it
                inst = self.__get_instance_from_modconf(mod_conf)
                if inst is None:
                    continue
                updated_instances = self.instances.copy()
                updated_instances[_current_module_name] = inst
                self.instances = updated_instances
                # Set this module to be restart so it will start
                self.set_to_restart(inst, is_an_error=False)  # here it's not a problem, so don't log it as AT RISK in health check
            
            for new_instance in changed_to_restart:
                _current_module_name = new_instance.get_name()
                _logger = self.logger_modules_change.get_sub_part(_current_module_name, register=False)
                _logger.debug('Restarting module (configuration change)')
                self.__remove_instance(_current_module_name, self.logger_modules_change)
                mod_conf = next((m for m in modules if m.get_name() == _current_module_name), None)
                # Got a new module configuration, need to start it
                inst = self.__get_instance_from_modconf(mod_conf)
                if inst is None:
                    continue
                updated_instances = self.instances.copy()
                updated_instances[_current_module_name] = inst
                self.instances = updated_instances
                if inst.is_external:
                    self.start_external_instance(inst)
                elif isinstance(inst, WithWorkersAndInventoryModule):
                    self.start_worker_based_instance(inst)
                else:
                    self.start_internal_instance(inst)
            # save the new modules value
            self.modules = modules
            
            # Now display what changed
            updated_instances_names = [module.get_name() for module in changed_to_restart]
            self._display_module_tree(added_modules_names=new_modules_to_start_name, updated_modules_names=updated_instances_names, deleted_modules_names=old_modules_to_delete_name)
    
    
    def set_max_queue_size(self, max_queue_size: int) -> None:
        self.max_queue_size = max_queue_size
    
    
    # Import, instantiate & "init" the modules we have been requested
    def load_and_init(self) -> bool:
        self.load()
        all_was_started = self._get_instances()
        return all_was_started
    
    
    def load(self) -> None:
        self._load_python_modules()
    
    
    # We want to know which sys.modules are new after the modules load, but beware of duplicate ones :)
    @staticmethod
    def _get_only_new_sys_modules(sys_modules_names_before, sys_modules_code_before):
        # Look at sys.modules and look which modules are now loaded when they were not before:
        # WARNING: there will be duplicate: entries that match the same lib object
        sys_modules_names_after = set(sys.modules.keys())
        sys_modules_names_added = sorted(sys_modules_names_after - sys_modules_names_before)
        
        for sys_module_name in list(sys_modules_names_added):  # copy because we will remove elements inside
            sys_module = sys.modules.get(sys_module_name, None)  # maybe a thread did remove it?
            if sys_module in sys_modules_code_before:  # was already loaded in fact
                sys_modules_names_added.remove(sys_module_name)
        return sys_modules_names_added
    
    
    # Try to import the requested modules ; put the imported modules in self.imported_modules.
    def _load_python_modules(self) -> None:
        # Was already done, don't load it again
        if self.imported_modules:
            return
        
        # And directories
        modules_dirs = sorted(module_dir_name for module_dir_name in os.listdir(self.modules_path) if os.path.isdir(os.path.join(self.modules_path, module_dir_name)))
        
        # Now we try to load them
        # So first we add their dir into the sys.path
        if self.modules_path not in sys.path:
            sys.path.append(self.modules_path)
        
        # Automatically loaded modules would have a conflicting ID with the modules sent by the Arbiter.
        ModuleConfiguration.reset_new_first_id(FIRST_MODULE_ID_FOR_MODULES_NOT_INSTANTIATED_BY_ARBITER)
        
        before = time.time()
        failed_modules_dir = []
        available_modules_names = []
        # We try to import them, but we keep only the one of
        # our type
        for module_dir_name in modules_dirs:
            _module_loading_logger = self.logger_module_code_loading.get_sub_part(f'directory={module_dir_name}', register=False)
            module_start = time.time()
            sys_modules_names_before = set(sys.modules.keys())
            sys_modules_code_before = set(sys.modules.values())
            module_info = None
            module_info_path = os.path.join(self.modules_path, module_dir_name, 'module_info.json')
            try:
                if os.path.exists(module_info_path):
                    with open(module_info_path) as json_file:
                        module_info = json.load(json_file)
                        module_type = module_info.get('type')
                        if module_type and self._keep_modules_info:
                            self._all_modules_info[uniform_module_type(module_type)] = module_info
                        if module_info.get('module_types_by_daemons', None):
                            modules_matches = [m for m in module_info['module_types_by_daemons'] if m['daemon'] == self.modules_type]
                            if not modules_matches:
                                continue
                        elif module_info.get('daemons'):
                            if self.modules_type not in module_info['daemons']:
                                continue
                        else:
                            _module_loading_logger.warning(f'''Missing 'daemons' key in the 'module_info.json' in directory '{module_info_path}'. Module code will be loaded anyway.''')
                else:
                    _module_loading_logger.warning(f'''Missing 'module_info.json' in directory '{module_info_path}'. Module code will be loaded anyway.''')
            except ValueError:
                _module_loading_logger.warning(f''''module_info.json' file in directory '{module_info_path}' is malformed. Module code will be loaded anyway.''')
            except IOError:
                _module_loading_logger.warning(f'''Read permission denied to file 'module_info.json' in directory '{module_info_path}'. Module code will be loaded anyway.''')
            
            module_dir = os.path.join(self.modules_path, module_dir_name)
            init_path_file = os.path.join(module_dir, '__init__.py')
            
            try:
                _module_loading_logger.info(f'Starting to load the module code from directory {module_dir}.')
                
                sys.path.append(module_dir)
                _module_loading_logger.info(f'Try to import code from {module_dir} as python module.')
                try:
                    imp.load_module(module_dir_name, *imp.find_module(module_dir_name, [self.modules_path]))
                except Exception:
                    _module_loading_logger.info(f'''Failed to import the directory {module_dir} as a python module. If this is not a python module, this is not a problem.''')
                else:
                    _module_loading_logger.info(f'Successfully imported the directory {module_dir} as a python module.')
                
                # Then we load the module.py inside this directory
                mod_file = os.path.abspath(os.path.join(self.modules_path, module_dir_name, 'module.py'))
                mod_dir = os.path.dirname(mod_file)
                # We add this dir to sys.path so the module can load local files too
                sys.path.append(mod_dir)
                if not os.path.exists(mod_file):
                    mod_file = os.path.abspath(os.path.join(self.modules_path, module_dir_name, 'module.pyc'))
                
                if mod_file.endswith('.py'):
                    _module_loading_logger.info(f'Loading the module source file {mod_file}.')
                    # important, equivalent to import file_name from module.py
                    py_module = imp.load_source(module_dir_name, mod_file)
                else:
                    _module_loading_logger.info(f'Loading the module compiled file {mod_file}.')
                    py_module = imp.load_compiled(module_dir_name, mod_file)
                
                if module_info:
                    py_module.properties = module_info
                # Look if it's a valid module
                elif hasattr(py_module, 'properties'):
                    module_info = py_module.properties
                    
                    module_type = module_info.get('type')
                    if module_type and self._keep_modules_info:
                        self._all_modules_info[uniform_module_type(module_type)] = module_info
                
                else:
                    _module_loading_logger.error(f'''Missing properties dict in module {mod_file}. The module won't be loaded.''')
                    failed_modules_dir.append(module_dir_name)
                    continue
                
                py_module.properties = module_info
                
                all_modules_info_for_this_daemon: 'list[dict[str, Any]]'
                if 'module_types_by_daemons' in module_info:
                    all_modules_info_for_this_daemon = [m for m in module_info['module_types_by_daemons'] if m['daemon'] == self.modules_type]
                    if not all_modules_info_for_this_daemon:
                        _module_loading_logger.warning(f'Bad module file for {mod_file} : not any daemon matches in "module_types_by_daemons"')
                        failed_modules_dir.append(module_dir_name)
                        continue
                elif 'type' in module_info:
                    all_modules_info_for_this_daemon = [module_info]
                else:
                    _module_loading_logger.warning(f'Bad module file for {mod_file} : missing type entry in properties dict')
                    failed_modules_dir.append(module_dir_name)
                    continue
                
                for daemon_module_info in [m for m in all_modules_info_for_this_daemon if m.get('autoload', False)]:
                    module_default_conf = None
                    module_type = daemon_module_info.get('type', daemon_module_info.get('module_type', ''))
                    
                    arg_spec = inspect.getfullargspec(py_module.get_autoload_class)
                    module_class: 'type[AutoLoadedModule]'
                    params = {}
                    if 'module_on' in arg_spec.args:
                        params['module_on'] = self.modules_type
                    if 'module_type' in arg_spec.args:
                        params['module_type'] = module_type
                    module_class = py_module.get_autoload_class(**params)
                    
                    if inspect.isclass(module_class) and issubclass(module_class, AutoLoadedModule) and module_class is not AutoLoadedModule:
                        _module_loading_logger.debug('autoload flag enabled')
                        # Module cannot change its own name when it has already been instantiated, we must load its saved name here
                        try:
                            module_default_conf = module_class.get_autoload_configuration(self.modules_type, self.daemon_id, self.daemon_display_name, module_type, logger=_module_loading_logger)
                        except Exception as e:
                            _module_loading_logger.error(str(e))
                            _module_loading_logger.print_stack()
                        # Required if we need to reset configuration later (when removed from Arbiter)
                        try:
                            self._auto_loaded_modules_auto_conf[module_type] = module_class.get_autoload_auto_configuration(self.daemon_display_name, module_type, _module_loading_logger)
                        except Exception as e:
                            _module_loading_logger.error(str(e))
                            _module_loading_logger.print_stack()
                        
                        if not module_default_conf and module_type in self._auto_loaded_modules_auto_conf:
                            module_default_conf = self._auto_loaded_modules_auto_conf[module_type]
                    else:
                        _module_loading_logger.error('Cannot auto load module (unable to detect main class of module)')
                    if module_default_conf:
                        self._auto_loaded_modules_conf[module_type] = module_default_conf
                    else:
                        _module_loading_logger.debug('No default configuration found for auto loaded module')
                
                # Look at sys.modules and look which modules are now loaded when they were not before:
                # WARNING: there will be duplicate: entries that match the same lib object
                sys_modules_names_added = self._get_only_new_sys_modules(sys_modules_names_before, sys_modules_code_before)
                
                if _module_loading_logger.is_debug():
                    imported_libs_log = ''
                    if sys_modules_names_added:
                        imported_libs_log = f''' Imported {len(sys_modules_names_added)} new python libraries ({','.join(sys_modules_names_added)}).'''
                    _module_loading_logger.debug(f'[{time.time() - module_start:.3f}s] Module code was loaded.{imported_libs_log}')
                
                self.imported_modules.append(py_module)
                for daemon_module_info in all_modules_info_for_this_daemon:
                    module_type = uniform_module_type(daemon_module_info.get('type', daemon_module_info.get('module_type', '')))
                    available_modules_names.append(module_type)
            
            except ImportError as exp:
                if os.path.exists(init_path_file):
                    _module_loading_logger.error(f'Import module [{module_dir_name}] failed: {exp}.')
                    _module_loading_logger.print_stack()
                else:
                    _module_loading_logger.error(f'''Failed to import the directory {module_dir} as a python module. Python code won't be loaded.''')
                    _module_loading_logger.error(f' - because of missing file : {init_path_file}.')
                    _module_loading_logger.error(f''' - The line 'import {module_dir_name}.my_submodule' won't work in module.py file.''')
            
            except Exception as exp:
                # Oops, something went wrong here... added in the daemon configuration.
                _module_loading_logger.error(f'Import module [{module_dir_name}] failed: {exp}.')
                _module_loading_logger.print_stack()
                failed_modules_dir.append(module_dir_name)
        if self.imported_modules:
            available_modules_names.sort()
            self.logger_module_code_loading.info('A total of %s Shinken Enterprise modules are available for this daemon/module ( %s ): %s ( on a total of %s, loaded in %.3fs )' % (
                len(self.imported_modules), self.modules_type, ', '.join(available_modules_names), len(modules_dirs), time.time() - before))
        
        if self._auto_loaded_modules_conf:
            # Ensure the automatically loaded modules are in the modules list before proceed.
            # Can happen if set_modules() has been called BEFORE load()
            # This is the case on external modules with submodules (e.g. WebUI)
            self.modules = self._modules
    
    
    # For a specific module definition, we want to get the pymodule that match us
    def __get_pymodule_from_mod_conf(self, mod_conf: ModuleConfiguration) -> 'Optional[types.ModuleType]':
        if not hasattr(mod_conf, 'module_type'):
            return None
        py_module = self.get_module_code_with_module_type(mod_conf.module_type)
        if py_module:
            return py_module
        _error = f'The module {mod_conf.get_name()} of type {mod_conf.module_type} is not available for the daemon/module {self.modules_type}.'
        self.errors.append(_error)
        self.logger_module_instance_creation.warning(_error)
        return None
    
    
    def get_module_code_with_module_type(self, module_type: str) -> 'Optional[types.ModuleType]':
        module_type = uniform_module_type(module_type)
        
        for py_module in self.imported_modules:
            if py_module.properties.get('module_types_by_daemons', None):
                for module_info in [m for m in py_module.properties['module_types_by_daemons'] if m.get('daemon') == self.modules_type]:
                    if module_type == uniform_module_type(module_info.get('module_type', 'unknown module type')):
                        return py_module
            elif 'type' in py_module.properties and uniform_module_type(py_module.properties['type']) == module_type:
                return py_module
        
        return None
    
    
    # Try to "init" the given module instance.
    # If late_start, don't look for last_init_try
    # If force_start, don't look at restart timing, and just try to start
    # Returns: state of the module
    def _try_instance_init(self, inst: 'BaseModule', late_start: bool = False, force_start: bool = False, start_logger: 'Optional[PartLogger]' = None):
        # Someone asks us to force a restart of the module, without looking at timing
        if force_start:
            state = INSTANCE_INIT_STATE.INIT_READY
        else:  # Look if we didn't already try to init since not long
            state = inst.is_ready_to_init(self.logger_module_instance_creation, late_start)
        if state == INSTANCE_INIT_STATE.INIT_READY:
            if start_logger:
                start_logger.info(f'''Start to create {'an external' if inst.is_external else ('a worker based' if inst.is_worker_based else 'an internal')} module''')
            if inst.is_external or inst.is_worker_based:
                inst.create_queues()
            is_fail, exp_message = inst.do_init(self.logger_module_instance_creation)
            if is_fail:
                self.did_crash(inst, reason=exp_message, do_log=False)
                return INSTANCE_INIT_STATE.INIT_FAILED
            elif not getattr(inst, 'enabled', True) and not inst.is_external and not inst.is_worker_based:
                # Autoload module has been disabled by configuration
                self.clear_instances([inst])
            return INSTANCE_INIT_STATE.INIT_SUCCESSFUL
        else:
            return INSTANCE_INIT_STATE.INIT_SKIPPED
    
    
    # Request to "remove" the given instances list or all if not provided
    def clear_instances(self, instances=None):
        if instances is None:
            instances = self.get_all_instances()  # have to make a copy of the list
        for i in instances:
            auto_loaded_module_is_already_up_to_date = False
            if self._auto_loaded_modules_conf:
                module_type = i.get_module_type()
                module_hash = i.get_module_configuration_hash()
                for m_type, m_conf in self._auto_loaded_modules_conf.items():
                    if m_type == module_type and m_conf.hash == module_hash:
                        auto_loaded_module_is_already_up_to_date = True
                        break
            if auto_loaded_module_is_already_up_to_date and getattr(i, 'enabled', True):
                continue
            self.logger_modules_stop.info(f'Stopping module {i.get_name()}')
            self.__remove_instance(i.get_name(), self.logger_modules_stop)
    
    
    def clear_instances_by_module_on_type(self, module_on_type):
        modules_list = self.instances.copy()
        for key, module in modules_list.items():
            if module_on_type not in module.get_module_on_type():
                self.instances.pop(key)  # We can pop it's a copy
                self.to_restart.pop(key, None)
    
    
    def go_to_idle_mode(self, modules_configuration: 'list[ModuleConfiguration]|None'):
        if modules_configuration is None:
            auto_loaded_modules = self._auto_loaded_modules_auto_conf.values()
        else:
            auto_loaded_modules_type = set(list(self._auto_loaded_modules_auto_conf.keys()))
            auto_loaded_modules = [module for module in modules_configuration if module.module_type in auto_loaded_modules_type]
        if auto_loaded_modules:
            self.update_modules(Modules(auto_loaded_modules))
            if self.to_restart:
                self.try_to_restart_crashed(force_start=True)
        else:
            self.modules = None
            self.clear_instances()
    
    
    # A daemon did detect that a module instance did crash and so don't want it anymore.
    # We will need to log it, kill the subprocess if external and set to restart later
    def did_crash(self, inst, reason='', logger_with_chapter=None, do_log=True):
        if do_log:
            if logger_with_chapter and reason:
                logger_with_chapter.error(reason)
            elif reason:
                self.logger.error(reason)
        self.set_to_restart(inst)
        self.__register_module_restart(inst.get_name(), reason=reason)
    
    
    # Put an instance to the restart queue
    # By default if an instance is set to restart, it means it's an error,
    # but in some case it's not (like new modules)
    def set_to_restart(self, inst: BaseModule, is_an_error: bool = True):
        try:
            inst.stop_all()
        except:
            # If external, clean its queues
            if inst.is_external or inst.is_worker_based:
                inst.clear_queues()
                inst.kill_manager()
        if inst.get_internal_state() == ModuleState.FATAL and is_an_error:
            inst.fatal_error_has_been_managed = True
            inst.stop_all()
            return
        self.to_restart[inst.get_name()] = inst
    
    
    def __get_instance_from_modconf(self, mod_conf: 'ModuleConfiguration') -> 'Optional[BaseModule]':
        mod_name = mod_conf.get_name()
        mod_type = mod_conf.get_type()
        py_module = self.__get_pymodule_from_mod_conf(mod_conf)
        # if we cannot find a suitable code to launch, skip it, we already warn about it
        if not py_module:
            return None
        start = time.time()
        _logger = _instance_logger_subpart_from_name_and_type(self.logger_module_instance_creation, mod_name, mod_type)
        
        try:
            _logger.debug('Start to create the module instance.')
            if 'module_types_by_daemons' in py_module.properties:
                module_info: 'dict[str, Any]'
                module_info = next((md for md in py_module.properties['module_types_by_daemons'] if md['daemon'] == self.modules_type and uniform_module_type(md['module_type']) == uniform_module_type(mod_type)), {})
                if not module_info:
                    self.logger.warning(f'Could not find module info for {mod_type!r} module type.')
                    return None
                module_info = copy.deepcopy(module_info)
                module_info['type'] = module_info.pop('module_type')
                module_info['daemons'] = [module_info.pop('daemon')]
                mod_conf.properties = module_info
            else:
                mod_conf.properties = copy.deepcopy(py_module.properties)
            
            # Additional information needed for some modules.
            mod_conf.properties['daemon_id'] = self.daemon_id
            mod_conf.properties['daemon_type'] = self.modules_type
            mod_conf.properties['daemon_name'] = self.daemon_name if self.daemon_name else self.daemon_display_name
            
            arg_spec = inspect.getfullargspec(py_module.get_instance)
            if len(arg_spec.args) > 1:
                params = {}
                if 'module_on' in arg_spec.args:
                    params['module_on'] = self.modules_type
                if 'module_type' in arg_spec.args:
                    params['module_type'] = mod_type
                inst = py_module.get_instance(mod_conf, **params)
            else:
                inst = py_module.get_instance(mod_conf)
            if inst is None:  # None = Bad thing happened :)
                _logger.error("The module get_instance() call did not return any instance or does not exist.")
                return None
            assert (isinstance(inst, BaseModule))
            _logger.info(f'[{time.time() - start:.3f}s] The module is created.')
            return inst
        
        except Exception as exp:
            _logger.error(f'[{time.time() - start:.3f}s] The module creation has failed raising exception: {exp}. Will retry later.')
            _logger.print_stack()
            return None
    
    
    # Actually only arbiter call this method with start_external=False..
    # Create, init and then returns the list of module instances that the caller needs.
    # If an instance can't be created or inited then only log is done.
    # That instance is skipped. The previous modules instance(s), if any, are all cleaned.
    def _get_instances(self) -> bool:
        with self.instances_lock:
            
            if self.modules is None:
                return True
            
            self.clear_instances()
            
            all_was_start = True
            previously_started_instance = {(instance.get_module_type(), instance.get_module_configuration_hash()): instance for instance in self.instances.values()}
            for mod_conf in self.modules:
                if (mod_conf.get_type(), mod_conf.hash) in previously_started_instance.keys():
                    inst = previously_started_instance[(mod_conf.get_type(), mod_conf.hash)]
                else:
                    inst = self.__get_instance_from_modconf(mod_conf)
                if inst:
                    updated_instances = self.instances.copy()
                    updated_instances[mod_conf.get_name()] = inst
                    self.instances = updated_instances
                else:  # there was an error on this module
                    all_was_start = False
            
            # We should initialize the modules, but not the:
            # * external ones
            # * with workers based
            # because they can be crashed and so it must be done just before forking
            for inst in self.get_all_instances():
                if not inst.is_external and not inst.is_worker_based:
                    if getattr(inst, 'is_no_longer_supported', False):
                        _logger = _instance_logger_subpart(self.logger_module_instance_creation, inst)
                        _logger.error(inst.get_state()['output'])
                        continue
                    if (inst.get_module_type(), inst.get_module_configuration_hash()) in previously_started_instance.keys():
                        continue
                    did_init = self._try_instance_init(inst)
                    if did_init == INSTANCE_INIT_STATE.INIT_FAILED:
                        # If the init failed, we put in the restart queue
                        self.set_to_restart(inst)
                        all_was_start = False
                    elif did_init == INSTANCE_INIT_STATE.INIT_SKIPPED:
                        all_was_start = False
            
            return all_was_start
    
    
    def start_internal_instance(self, inst: 'BaseModule', late_start: bool = False) -> bool:
        _logger = _instance_logger_subpart(self.logger_module_instance_creation, inst)
        # Init of some instances may have failed, so bypass them for now
        init_state = self._try_instance_init(inst, late_start=late_start)
        if init_state == INSTANCE_INIT_STATE.INIT_FAILED:
            _logger.error('The module failed to init')
            self.set_to_restart(inst)
            return False
        if init_state == INSTANCE_INIT_STATE.INIT_SKIPPED:
            self.set_to_restart(inst, is_an_error=False)
            return False
        return True
    
    
    def start_external_instance(self, inst: 'BaseModule', late_start: bool = False) -> None:
        _logger = _instance_logger_subpart(self.logger_module_instance_creation, inst)
        
        # _logger.debug('Start to create the module instance')
        start = time.time()
        did_init = self._try_instance_init(inst, late_start=late_start, start_logger=_logger)
        if did_init == INSTANCE_INIT_STATE.INIT_FAILED:
            self.set_to_restart(inst)
            return
        if did_init == INSTANCE_INIT_STATE.INIT_SKIPPED:
            _logger.debug('Creation module instance was skipped')
            self.set_to_restart(inst, is_an_error=False)
            return
        try:
            inst.start(daemon_display_name=self.daemon_display_name)
            _logger.info(f'[{time.time() - start:.3f}s] The module is created.')
        except Exception as exp:
            _logger.error(f'[{time.time() - start:.3f}s] The module creation has failed raising exception: {exp}. Will retry later.')
            self.set_to_restart(inst)
    
    
    # Launch external instances that are load correctly
    def start_external_instances(self, late_start: bool = False) -> None:
        for inst in [inst for inst in self.get_all_instances() if inst.is_external]:
            self.start_external_instance(inst, late_start=late_start)
    
    
    def start_worker_based_instance(self, inst: 'WithWorkersAndInventoryModule', late_start: bool = False) -> bool:
        _logger = _instance_logger_subpart(self.logger_module_instance_creation, inst)
        _logger.info('Starting a worker based module.')
        did_init = self._try_instance_init(inst, late_start=late_start, start_logger=_logger)
        if did_init == INSTANCE_INIT_STATE.INIT_SKIPPED:
            self.set_to_restart(inst, is_an_error=False)
            return False
        if did_init == INSTANCE_INIT_STATE.INIT_FAILED:
            _logger.error('The worker based module failed to init. Will retry later.')
            self.set_to_restart(inst)
            return False
        
        # ok, init succeed
        _logger.info('The worker based module has started.')
        try:
            inst.start_workers(daemon_display_name=self.daemon_display_name)
            return True
        except Exception as e:
            _logger.error(f'The worker based module has failed to init raising exception: {e}. Will retry later.')
            self.did_crash(inst, reason=str(e))
            return False
    
    
    # Launch external instances that are load correctly
    def start_worker_based_instances(self, late_start: bool = False) -> None:
        for inst in [inst for inst in self.get_all_instances() if isinstance(inst, WithWorkersAndInventoryModule)]:
            self.start_worker_based_instance(inst, late_start=late_start)
    
    
    # Request to cleanly remove the given instance.
    # If instance is external also shutdown it cleanly
    def __remove_instance(self, instance_name: str, update_stop_logger: 'PartLogger') -> None:
        with self.instances_lock:
            inst = self.instances.get(instance_name, None)
            if inst is None:
                _logger = _instance_logger_subpart_from_name_and_type(update_stop_logger, instance_name, '')
                _logger.warning(f'Trying to remove the module {instance_name} but it was not in running modules list: {self.instances}')
                return
            
            _logger = _instance_logger_subpart(update_stop_logger, inst)
            try:
                inst.stop_all(_logger)
            except Exception as exp:
                _logger.error(f'The module {instance_name} has failed to stop raising exception: {exp}')
                _logger.print_stack()
            # Then do no more listen about it
            del self.instances[instance_name]
            # SEF-6521: do not forget to clean in all lists:
            if instance_name in self.to_restart:
                del self.to_restart[instance_name]
    
    
    def check_alive_instances(self, skip_external: bool = False) -> None:
        # Only for external
        for instance_name, inst in self.get_all_instances_with_name():
            # skip already to restart one
            _logger = _instance_logger_subpart(self.logger_modules_instance_crash, inst)
            
            if instance_name in self.to_restart or inst.fatal_error_has_been_managed:
                continue
            
            # The skip_external is used in a child process because there is no possible to set an external module to a subprocess/submodule
            if skip_external and inst.is_external:
                continue
            
            if getattr(inst, 'is_no_longer_supported', False):
                _logger.error(inst.get_state()['output'])
                inst.fatal_error_has_been_managed = True
                continue
            
            if not inst.is_alive():
                self.did_crash(inst, f'The module {instance_name} has gone down unexpectedly!', logger_with_chapter=_logger)
                inst.stop_process()
                continue
            
            if inst.is_worker_based and not inst.check_worker_processes():
                self.did_crash(inst, f'The module {instance_name} worker(s) has gone down unexpectedly!', logger_with_chapter=_logger)
                continue
            
            # Now look for man queue size. If above value, the module should get a huge problem
            # and so bailout. It's not a perfect solution, more a watchdog
            # If max_queue_size is 0, don't check this
            if self.max_queue_size == 0:
                continue
            # Ok, go launch the dog!
            queue_size = 0
            try:
                queue_size = inst.to_q.qsize()
            except Exception:
                pass
            if queue_size > self.max_queue_size:
                self.did_crash(inst, f'The external module {instance_name} queue size is too high ( {queue_size} > {self.max_queue_size} )!', logger_with_chapter=_logger)
    
    
    def is_instance_set_to_restart(self, instance: 'BaseModule') -> bool:
        return instance in self.to_restart.values()
    
    
    # We are looking at the
    def try_to_restart_crashed(self, force_start: bool = False) -> None:
        to_restart = self.to_restart
        self.to_restart = {}
        for instance_name, inst in list(to_restart.items()):
            if inst.is_external:
                self.start_external_instance(inst)
            elif inst.is_worker_based:
                self.start_worker_based_instance(inst)
            else:
                if not self._try_instance_init(inst, force_start=force_start) == INSTANCE_INIT_STATE.INIT_SUCCESSFUL:
                    self.set_to_restart(inst)
    
    
    def __register_module_restart(self, module_name: str, reason: str = '') -> None:
        with self.last_restarts_lock:
            if module_name not in self.last_restarts:
                self.last_restarts[module_name] = []
            
            self.last_restarts[module_name].append({'timestamp': time.time(), 'reason': reason.rstrip()})
    
    
    # Called from StateCleaner thread
    def clean_modules_states(self) -> None:
        with self.last_restarts_lock:
            now_ts = time.time()
            clean_ts = now_ts - self.last_restarts_keep
            
            for last_restarts in list(self.last_restarts.values()):
                delete_count = 0
                
                for restart_ts in last_restarts:
                    if restart_ts['timestamp'] < clean_ts:
                        delete_count += 1
                    else:
                        break
                
                del last_restarts[:delete_count]
    
    
    def get_modules_states(self, module_wanted: 'Optional[list]' = None) -> dict[str, list]:
        if module_wanted:
            actual_module = module_wanted[0]
            if actual_module == NO_MODULES_WANTED:
                return {}
        else:
            module_wanted = []
            actual_module = None
        states = []
        for name, inst in self.get_all_instances_with_name():
            if not actual_module or actual_module == name:
                start_time = time.time()
                status = {'restarts': self.last_restarts.get(name, []), 'name': name, 'type': inst.get_module_type()}
                status.update(self.get_module_state(inst, module_wanted[1:]))
                states.append(status)
                self.logger.log_perf(start_time, 'GET_MODULES_STATES', f'Module {name} state requested', min_time=1, warn_time=1.8)
        return {'modules': states, 'errors': self.errors}
    
    
    def get_modules_raw_stats(self, param: str = '', module_wanted: 'list[str]|None' = None) -> 'dict[str, Any]':
        if module_wanted:
            actual_module = module_wanted[0]
            if actual_module == NO_MODULES_WANTED:
                return {}
        else:
            module_wanted = []
            actual_module = None
        
        all_modules_stats = {}
        for name, instance in self.get_all_instances_with_name():
            # Always add module_type dictionary even if the module is not selected.
            module_stats = all_modules_stats.setdefault(instance.get_module_type(), {})
            if actual_module and actual_module != name:
                continue
            start_time = time.time()
            result = self.get_module_raw_stats(instance, param, module_wanted[1:])
            
            module_stats[name] = result
            self.logger.log_perf(start_time, 'GET_RAW_STATS', f'Module {name} stats requested', min_time=1, warn_time=1.8)
        
        return all_modules_stats
    
    
    def get_messages_to_send_at_arbiter(self) -> 'list[InterDaemonMessage]':
        return list(itertools.chain.from_iterable(self.get_messages_to_send_at_arbiter_from_module(instance) for instance in self.get_all_instances()))
    
    
    @classmethod
    def get_module_state(cls, instance: 'BaseModule', module_wanted: 'list|None' = None) -> 'dict[str, Any]':
        args_len = len(inspect.getfullargspec(instance.get_state).args)
        args: 'list[Any]' = []
        if args_len > 1:
            args.append(module_wanted)
        
        if instance.is_external:
            if not instance.commands_to_q:
                return {'status': ModuleState.CRITICAL, 'output': 'Module process is currently starting.'}
            if instance.process is None:
                return {'status': ModuleState.CRITICAL, 'output': 'Module process is currently starting.'}
            elif not instance.process.is_alive():
                return {'status': ModuleState.CRITICAL, 'output': 'Module process is currently down and is scheduled for a restart.'}
            
            try:
                return instance.send_command('get_state', args, raw_args=True)
            except Exception as e:
                return {'status': ModuleState.CRITICAL, 'output': str(e)}
        else:
            return instance.get_state(*args)
    
    
    @classmethod
    def get_module_raw_stats(cls, instance: 'BaseModule', param: str = '', module_wanted: 'list[str]|None' = None) -> 'dict[str, Any]':
        if module_wanted and module_wanted[0] != NO_MODULES_WANTED:
            # Only submodule stats is requested.
            return cls.get_submodule_raw_stats_on_module(instance, param=param, module_wanted=module_wanted)
        
        args_len = len(inspect.getfullargspec(instance.get_raw_stats).args)
        args: 'list[Any]' = []
        if args_len > 1:
            args.append(param)
        if args_len > 2:
            args.append(module_wanted)
        
        if instance.is_external:
            if not instance.commands_to_q:
                return {}
            try:
                return instance.send_command('get_raw_stats', args, raw_args=True)
            except Exception as e:
                instance.logger.error(f'Fail in get_raw_stats with error : [{e}]')
                return {}
        else:
            return instance.get_raw_stats(*args)
    
    
    @classmethod
    def get_submodule_raw_stats_on_module(cls, instance: 'BaseModule', param: str = '', module_wanted: 'list|None' = None) -> 'dict[str, Any]':
        if instance.is_external:
            if not instance.commands_to_q:
                return {}
            try:
                return instance.send_command('get_submodules_raw_stats', [param, module_wanted], raw_args=True)
            except Exception as e:
                instance.logger.error(f'Fail in get_submodules_raw_stats with error : [{e}]')
                return {}
        else:
            return instance.get_submodules_raw_stats(param, module_wanted)
    
    
    @classmethod
    def handle_messages_received_from_arbiter(cls, instance: 'BaseModule', push_message_by_arbiter: 'InterDaemonMessage') -> None:
        if instance.is_external:
            # Broker want to talk to external module.
            if not instance.commands_to_q:
                return
            try:
                instance.send_command('handle_messages_received_from_arbiter', args=[push_message_by_arbiter], raw_args=True)
            except Exception as e:
                instance.logger.error(f'Fail in handle_messages_received_from_arbiter with error : [{e}]')
        else:
            instance.handle_messages_received_from_arbiter(push_message_by_arbiter)
    
    
    @classmethod
    def get_messages_to_send_at_arbiter_from_module(cls, instance: 'BaseModule') -> 'list[InterDaemonMessage]':
        if instance.is_external:
            if not instance.commands_to_q:
                return []
            try:
                return instance.send_command('get_messages_to_send_at_arbiter')
            except Exception as e:
                instance.logger.error(f'Fail in get_messages_to_send_at_arbiter with error : [{e}]')
                return []
        else:
            return instance.get_messages_to_send_at_arbiter()
    
    
    def _get_instances_with_condition(self, condition: 'Callable[[BaseModule], bool]' = lambda inst: True) -> 'list[BaseModule]':
        return sorted(filter(condition, self.get_all_instances()), key=lambda inst: self._idx_modules_order_by_name[inst.get_name()])
    
    
    # Do not give to others inst that got problems
    def get_internal_instances(self, phase: str = None) -> 'list[BaseModule]':
        return self._get_instances_with_condition(lambda inst: not inst.is_external and phase in inst.phases and inst not in self.to_restart.values() and inst.get_internal_state() != ModuleState.FATAL)
    
    
    def get_external_instances(self, phase: str = None) -> 'list[BaseModule]':
        return self._get_instances_with_condition(lambda inst: inst.is_external and phase in inst.phases and inst not in self.to_restart.values() and inst.get_internal_state() != ModuleState.FATAL)
    
    
    def get_external_to_queues(self) -> list['ShinkenQueue']:
        return [inst.to_q for inst in self.get_all_instances() if inst.is_external and inst not in self.to_restart.values() and inst.get_internal_state() != ModuleState.FATAL]
    
    
    def get_external_modules_and_queues(self) -> list[tuple['BaseModule', 'ShinkenQueue']]:
        return [(inst, inst.to_q) for inst in self.get_all_instances() if inst.is_external and inst not in self.to_restart.values() and inst.get_internal_state() != ModuleState.FATAL]
    
    
    def get_external_modules_and_from_queues(self):
        return [(inst, inst.from_module_to_main_daemon_queue) for inst in self.get_all_alive_instances() if (inst.is_external or inst.is_worker_based)]
    
    
    def get_external_from_queues(self):
        return [inst.from_module_to_main_daemon_queue for inst in self.get_all_instances() if (inst.is_external or inst.is_worker_based) and inst not in self.to_restart.values() and inst.get_internal_state() != ModuleState.FATAL]
    
    
    def get_all_alive_instances(self) -> 'list[BaseModule]':
        with self.instances_lock:
            return self._get_instances_with_condition(lambda inst: inst not in self.to_restart.values() and inst.get_internal_state() != ModuleState.FATAL)
    
    
    def stop_all(self, update_stop_logger=None):
        if not update_stop_logger:
            update_stop_logger = self.logger_modules_stop
        update_stop_logger.info('Start to shutdown all modules.')
        # Ask internal to quit if they can
        for inst in self.get_internal_instances():
            if not inst.is_worker_based:
                inst.stop_all(update_stop_logger)
        
        # Clear/stop all external & worker based instances
        self.clear_instances([inst for inst in self.get_all_instances() if inst.is_external])
        self.clear_instances([inst for inst in self.get_all_instances() if inst.is_worker_based])
        
        update_stop_logger.info('Stopping all modules ended.')
    
    
    def get_all_instances(self):
        with self.instances_lock:
            return list(self.instances.values())
    
    
    def get_all_instances_name(self):
        with self.instances_lock:
            return list(self.instances.keys())
    
    
    def get_all_instances_with_name(self):
        with self.instances_lock:
            return list(self.instances.items())
    
    
    def is_activated(self, module_type):
        return any(mod.properties['type'] == module_type for mod in self.get_all_instances())
    
    
    def is_alive(self, module_type):
        return any(mod.properties['type'] == module_type for mod in self.get_all_alive_instances())
    
    
    def do_after_fork_cleanup(self, after_fork_new_top_instance):
        if hasattr(after_fork_new_top_instance, 'logger'):
            _logger = after_fork_new_top_instance.logger
        else:
            _logger = self.logger
        cleanup_logger = _logger.get_sub_part(AFTER_FORK_CLEANUP_PART_LOGGER)
        cleanup_logger = cleanup_logger.get_sub_part(f'pid:{os.getpid()}', register=False)
        
        # We are in a forked process, modules_manager does no more exist here.
        # * Do not use self.instances_lock as
        #   1 - it is useless here
        #   2 - its state might be incorrect
        # * No items() here as we will delete items in for loop
        for name, inst in list(self.instances.items()):
            module_cleanup_logger = cleanup_logger.get_sub_part(name)
            if inst == after_fork_new_top_instance:
                module_cleanup_logger.debug('skipping my own instance cleanup')
                continue
            inst_cleanup = getattr(inst, AFTER_FORK_CLEANUP_METHOD, None)
            if callable(inst_cleanup):
                module_cleanup_logger.debug(f'cleanup of module 〖 {name} 〗 is starting')
                try:
                    inst_cleanup()
                except Exception as e:
                    module_cleanup_logger.error(
                        'On linux system, the forking mechanism (process creation) is fast but create a copy of the father process. So we have to release unnecessary resources inherited from father. The cleaning has been performed but we encountered an error:')
                    module_cleanup_logger.error(f'Cleanup of data from module 〖 {name} 〗 raised error 〖 {e} 〗')
                    module_cleanup_logger.error('Some memory may have not been freed. Shinken will still run if enough memory remains available.')
                    module_cleanup_logger.error('You can report this message to support in order to optimize Shinken memory consumption')
                    module_cleanup_logger.print_stack()
                else:
                    module_cleanup_logger.debug(f'cleanup of module 〖 {name} 〗 has ended')
            elif inst_cleanup:
                module_cleanup_logger.error(
                    'On linux system, the forking mechanism (process creation) is fast but create a copy of the father process. So we have to release unnecessary resources inherited from father. The cleaning has been performed but we encountered an error:')
                module_cleanup_logger.error(f'〖 {AFTER_FORK_CLEANUP_METHOD} 〗 is not callable, can not run cleanup process')
                module_cleanup_logger.error('Some memory may have not been freed. Shinken will still run if enough memory remains available.')
                module_cleanup_logger.error('You can report this message to support in order to optimize Shinken memory consumption')
            module_cleanup_logger.debug(f'removing module 〖 {name} 〗 from process memory')
            del self.instances[name]
        del self.instances


def _instance_logger_subpart(logger: 'PartLogger', instance: 'BaseModule') -> 'PartLogger':
    return _instance_logger_subpart_from_name_and_type(logger, instance.get_name(), instance.get_module_type())


def _instance_logger_subpart_from_name_and_type(logger: 'PartLogger', module_name: str, module_type: str) -> 'PartLogger':
    logger = logger.get_sub_part(module_name, register=False)
    if module_type:
        logger = logger.get_sub_part(f'module-type={module_type}', register=False)
    return logger
