#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2021:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.state_id import STATE_ID

if TYPE_CHECKING:
    from shinken.misc.type_hint import Number, Union
    
    UnixTimestamp = Number
    FullStatus = Union[int, None]
    StateId = int


# Full status definition (cf. build_full_status() in SchedulingItem) :
#
# len(self.get_active_downtime_uuids()) * 10000000000 + \
#               self.inherited_flapping * 1000000000 + \
#   self.is_in_inherited_acknowledged() * 100000000 + \
#       self.is_in_inherited_downtime() * 10000000 + \
#              self.is_partial_flapping * 1000000 + \
#          self.is_partial_acknowledged * 100000 + \
#              self.in_partial_downtime * 10000 + \
#                      self.is_flapping * 1000 + \
#    self.problem_has_been_acknowledged * 100 + \
#            self.in_scheduled_downtime * 10 + \
#                              state_id

class FullStatusManipulator:
    
    @staticmethod
    def get_state_from_full_status(full_status, full_status_change_time):
        # type: (FullStatus, UnixTimestamp) -> StateId
        if full_status is None or not full_status_change_time:
            return STATE_ID.UNKNOWN  # No full status known yet, return UNKNOWN
        return full_status % 10
    
    
    @staticmethod
    def change_state_in_full_status(full_status, new_state):
        # type: (FullStatus, StateId) -> FullStatus
        if full_status is None:
            return full_status
        return (full_status // 10) * 10 + new_state
    
    
    @staticmethod
    def set_missing_data_state_in_full_status(current_full_status):
        # type: (FullStatus) -> FullStatus
        return FullStatusManipulator.change_state_in_full_status(current_full_status, STATE_ID.MISSING_DATA)
