#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import SOURCE_STATE
from shinkensolutions.api.synchronizer import get_class_from_type
from shinkensolutions.api.synchronizer.source.abstract_module.source_module import SourceModule
from shinkensolutions.api.synchronizer.source.source_configuration_value import SERVICE_MODE

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, List
    from shinkensolutions.api.synchronizer import ValidationState
    from shinkensolutions.api.synchronizer.source.item.item_container import ItemsContainers


class CollectorModule(SourceModule):
    
    def load_configuration(self, configuration):
        raise NotImplementedError()
    
    
    def import_source_items(self):
        raise NotImplementedError()
    
    
    def prepare_finale_source_result(self, items_containers, warning=None, errors=None):
        # type: (ItemsContainers, List[str], List[str]) -> Dict
        start_time_prepare_finale_source_result = time.time()
        
        objects = {}
        output = []
        state = SOURCE_STATE.OK
        if warning:
            state = SOURCE_STATE.WARNING
        if errors:
            state = SOURCE_STATE.CRITICAL
        
        errors = errors or []
        warning = warning or []
        
        if errors or warning:
            output = errors + warning
        
        self.source_configuration.validation_state.warnings = []
        self.source_configuration.proxy_source_info.validation_state.warnings = []
        
        if SERVICE_MODE.is_enable(self._state_of_host_template_binding_rule):
            state = self._check_warning(output, state, self.host_template_binding_rule_manager.validation_state, warning)
        
        if SERVICE_MODE.is_enable(self._state_of_mapping_origin_to_source):
            state = self._check_warning(output, state, self.mapper_origin_to_source.validation_state, warning)
        
        if SERVICE_MODE.is_enable(self._state_of_origin_item_properties_description):
            state = self._check_warning(output, state, self.origin_item_description.validation_state, warning)
        
        if SERVICE_MODE.is_enable(self._state_of_api_item_properties):
            state = self._check_warning(output, state, self.api_item_properties.validation_state, warning)
        self.logger.log_perf(start_time_prepare_finale_source_result, 'SourceVMWarePerformances', '1.5.1 - [collector_module.py] [prepare_finale_source_result] [check source state]', 0, 1000000, 0)
        
        start_time_apply_host_template_binding_rules = time.time()
        items_containers.apply_host_template_binding_rules()
        self.logger.log_perf(start_time_apply_host_template_binding_rules, 'SourceVMWarePerformances', '1.5.2 - [collector_module.py] [prepare_finale_source_result] [apply_host_template_binding_rules]', 0, 1000000, 0)
        
        start_time_iteritems = time.time()
        for item_type, items in items_containers.get_items().items():
            for source_item in items:
                if item_type not in self.type_to_import and self.type_to_import != ['']:
                    continue
                item_class = get_class_from_type(item_type)
                conf_item = self._mapper_source_to_conf.map(source_item)
                if item_class not in objects:
                    objects[item_class] = []
                if 'tpl' in item_type:
                    conf_item['register'] = '0'
                objects[item_class].append(conf_item)
        self.logger.log_perf(start_time_iteritems, 'SourceVMWarePerformances', '1.5.3 - [collector_module.py] [prepare_finale_source_result] [iteritems]', 0, 1000000, 0)
        
        start_time_result_creation = time.time()
        result = {
            'state'  : state,
            'output' : '<br>'.join(output) or self.translator.translate('source.import_message_when_ok') % self.source_name,
            'objects': objects,
            'errors' : errors,
            'warning': warning
        }
        self.logger.log_perf(start_time_result_creation, 'SourceVMWarePerformances', '1.5.4 - [collector_module.py] [prepare_finale_source_result] [result creation]', 0, 1000000, 0)
        
        self.logger.info('prepare return to synchronizer')
        self.logger.log_perf(start_time_prepare_finale_source_result, 'SourceVMWarePerformances', '1.5 - [collector_module.py] [prepare_finale_source_result] [Total]', 0, 1000000, 0)
        return result
    
    
    @staticmethod
    def _check_warning(output, state, validation, warning):
        # type: (List[str], str, ValidationState, List[str]) -> str
        if validation.has_warning():
            state = SOURCE_STATE.WARNING
            for validation_warning in validation.get_warnings():
                if validation_warning not in output:
                    output.append(validation_warning)
            for validation_warning in validation.get_warnings():
                if validation_warning not in warning:
                    warning.append(validation_warning)
        return state
