#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.


import re
from typing import Tuple, Any

from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.api.synchronizer.source.check_property import CHECK_PROPERTY

if TYPE_CHECKING:
    from shinken.misc.type_hint import Union, Tuple, Callable, Dict, Any

logger = LoggerFactory.get_logger('MAPPING RULES')

escaped_dollar_sequence = '%$'
replacement_in_string_of_escaped_dollar_sequence = 'SKN_ESCAPED_DOLLAR_SEQUENCE_SKN'


class ListOperations:
    ERRORS_MESSAGES = {
        'typo': 'TYPO_IN_PROPERTY'
    }
    
    MAPPING_RULES_DEFINITIONS = {
        'get_by_filter_in_list': (re.compile(r'>VALUES\(=[^)]+\)'), re.compile(r'VALUES\(=([^)]+)\)')),  # VALUES(=TOTO)
        'get_by_index_in_list' : (re.compile(r'>VALUES\((-?\d+|LAST)\)'), re.compile(r'VALUES\((-?\d+|LAST)\)')),  # VALUES(0)
        'concat_values'        : (re.compile(r'>CONCAT\(.+\)'), re.compile(r'CONCAT\((.+)\)')),  # CONCAT(...)
        'concat_on_all'        : (re.compile(r'>CONCAT_ON_ALL\(.+\)'), re.compile(r'CONCAT_ON_ALL\((.+)\)')),  # CONCAT_ON_ALL(...)
        'transform_values'     : (re.compile(r'>TRANSFORM\(.+\)'), re.compile(r'TRANSFORM\((.+)\)')),  # TRANSFORM(...)
        'transform_on_all'     : (re.compile(r'>TRANSFORM_ON_ALL\(.+\)'), re.compile(r'TRANSFORM_ON_ALL\((.+)\)')),  # TRANSFORM(...)
    }
    
    TRANSFORM_OPERATOR_REPLACEMENT_STRING = 'SNK__TRANSFORM__OPERATOR__SNK'
    ON_ALL_ITEM_VALUE_ANCHOR = '$CURRENT$'  # The value in the SOMETHING_ON_ALL args which represent the current item value
    
    
    @staticmethod
    def concat_values(_item, getter_args, mapper_function, origin_item):
        # type: (Union[Dict, object], Tuple[str], Callable, Union[Dict, object] ) -> Any
        
        tag_to_replace_with_value = {}
        raw_concat = getter_args[0]
        
        if escaped_dollar_sequence in raw_concat:
            raw_concat = raw_concat.replace(escaped_dollar_sequence, replacement_in_string_of_escaped_dollar_sequence)
        
        tag_to_replace_with_properties_name = ListOperations._parse_concat_arg(raw_concat)
        for tag_to_replace, property_name in tag_to_replace_with_properties_name.items():
            value = mapper_function(origin_item, property_name, origin_item)
            value = ','.join(value) if isinstance(value, list) else value
            tag_to_replace_with_value[tag_to_replace] = value
        
        for tag_to_replace, value in tag_to_replace_with_value.items():
            raw_concat = raw_concat.replace(tag_to_replace, value)
        
        if replacement_in_string_of_escaped_dollar_sequence in raw_concat:
            raw_concat = raw_concat.replace(replacement_in_string_of_escaped_dollar_sequence, '$')
        return raw_concat
    
    
    @staticmethod
    def concat_on_all(_item, getter_args, mapper_function, origin_item):
        # type: (Union[Dict, object], Tuple[str], Callable, Union[Dict, object] ) -> Any
        # """
        # Apply a concat operation for every item inside a list
        # @param _item: string containing the value to work on
        # @param getter_args: Function args, here the CONCAT instruction
        # @param mapper_function: the function used to extract values from the dictionary
        # @param origin_item: origin item provided by the source
        # @return:
        # """
        if not isinstance(_item, list):
            return ''
        
        list_to_return = list()
        for item in _item:
            concat_args = getter_args[0]
            if ListOperations.ON_ALL_ITEM_VALUE_ANCHOR in concat_args:
                concat_args = concat_args.replace(ListOperations.ON_ALL_ITEM_VALUE_ANCHOR, item)
            list_to_return.append(ListOperations.concat_values(_item, (concat_args,), mapper_function, origin_item))
        
        return list_to_return
    
    
    @staticmethod
    def transform_values(_item, getter_args, mapper_function, origin_item):
        # type: (Union[str, list], Tuple[str], Callable, Union[Dict, object] ) -> Any
        # """
        # transform substring into another
        # @param _item: string containing the value to work on
        # @param getter_args: Function args, here the instruction of transforming substring (ex: "BORDEAUX=>BDX")
        # @param mapper_function: the function used to extract values from the dictionary
        # @param origin_item: origin item provided by the source
        # @return:
        # """
        
        if not isinstance(_item, str) and not isinstance(_item, list):
            return ''
        
        if len(_item) == 0:
            return ''
        
        if isinstance(_item, list):
            list_to_return = list()
            for item in _item:
                transform_args = getter_args[0]
                
                if ListOperations.ON_ALL_ITEM_VALUE_ANCHOR in transform_args:
                    transform_args = transform_args.replace(ListOperations.ON_ALL_ITEM_VALUE_ANCHOR, item)
                list_to_return.append(ListOperations.transform_values(item, (transform_args,), mapper_function, origin_item))
            return list_to_return
        
        # Prepare
        function_args = getter_args[0]
        string_to_change = _item
        
        if ListOperations.TRANSFORM_OPERATOR_REPLACEMENT_STRING in function_args:
            function_args = function_args.replace(ListOperations.TRANSFORM_OPERATOR_REPLACEMENT_STRING, '=>')
        else:
            return _item
        
        # ACT
        all_transform_instructions = function_args.split(',')
        for transform_instruction in all_transform_instructions:
            substring_to_replace, substring_to_replace_with = transform_instruction.split('=>')
            if substring_to_replace != '':
                string_to_change = string_to_change.replace(substring_to_replace, substring_to_replace_with) if substring_to_replace in string_to_change else string_to_change
        
        # RETURN
        return string_to_change
    
    
    @staticmethod
    def _parse_concat_arg(raw_concat_arg):
        # type: (str) -> Dict[str, str]
        
        if escaped_dollar_sequence in raw_concat_arg:
            raw_concat_arg = raw_concat_arg.replace(escaped_dollar_sequence, replacement_in_string_of_escaped_dollar_sequence)
        
        all_to_replace = re.findall(r'\$([^$]+)\$', raw_concat_arg)
        
        tag_to_replace_with_properties_name = {}
        for properties_name in all_to_replace:
            tag_to_replace_with_properties_name['$%s$' % properties_name] = properties_name
        return tag_to_replace_with_properties_name
    
    
    @staticmethod
    def get_by_filter_in_list(item, getter_args, _mapper, _origin_item):
        # type: (Union[Dict, object], Tuple[str], Callable, Union[Dict, object] ) -> Any
        
        _filter = getter_args[0]
        not_equal = _filter.startswith('!')
        if not_equal:
            _filter = _filter.replace(_filter[0], '')
        must_start = _filter.startswith('^')
        must_end = _filter.endswith('$')
        if must_start and must_end:
            _filter = _filter[1:-1]
            if isinstance(item, str) or isinstance(item, str):
                return item if (item.upper() == _filter.upper() and not_equal is False) or (item.upper() != _filter.upper() and not_equal is True) else ''
            return [i for i in item if (i.upper() == _filter.upper() and not_equal is False) or (i.upper() != _filter.upper() and not_equal is True)]
        elif must_start:
            _filter = _filter[1:]
            if isinstance(item, str) or isinstance(item, str):
                return item if (item.upper().startswith(_filter.upper()) and not_equal is False) or (not item.upper().startswith(_filter.upper()) and not_equal is True) else ''
            return [i for i in item if (i.upper().startswith(_filter.upper()) and not_equal is False) or (not i.upper().startswith(_filter.upper()) and not_equal is True)]
        elif must_end:
            _filter = _filter[:-1]
            if isinstance(item, str) or isinstance(item, str):
                return item if (item.upper().endswith(_filter.upper()) and not_equal is False) or (not item.upper().endswith(_filter.upper()) and not_equal is True) else ''
            return [i for i in item if (i.upper().endswith(_filter.upper()) and not_equal is False) or (not i.upper().endswith(_filter.upper()) and not_equal is True)]
        else:
            if isinstance(item, str) or isinstance(item, str):
                return item if (_filter.upper() in item.upper() and not_equal is False) or (not _filter.upper() in item.upper() and not_equal is True) else ''
            return [i for i in item if (_filter.upper() in i.upper() and not_equal is False) or (not _filter.upper() in i.upper() and not_equal is True)]
    
    
    @staticmethod
    def get_by_index_in_list(item, getter_args, _mapper, _origin_item):
        # type: (Union[Dict, object], Tuple[str], Callable, Union[Dict, object] ) -> Any
        if len(item) == 0:
            return ''
        
        if isinstance(item, str) or isinstance(item, str) or hasattr(item, 'get'):
            return item
        
        index = int(getter_args[0]) if getter_args[0] != 'LAST' else 'LAST'
        
        if isinstance(index, int) and -1 < index < len(item):
            return [item[index]] if isinstance(item[index], str) else item[index]
        elif index == -1:
            return [item[len(item) - 1] if isinstance(item[len(item) - 1], str) else item[len(item) - 1]]
        return ''
    
    
    @staticmethod
    def rule_has_typo_on_concat(item_property_part):
        # type: (str) -> bool
        # """
        # As we need to know if the source user made a typo on the CONCAT rule, we run this function
        # @param item_property_part:
        # @return:
        # """
        if item_property_part.startswith('VALUES') or item_property_part.startswith('TRANSFORM'):
            return False
        concat_typo_mapping_regex = re.compile(r'^[a-zA-Z]+\([^\n\t\r]+\)$')
        
        match = re.match(concat_typo_mapping_regex, item_property_part)
        if match:
            return True
        return False
    
    
    @staticmethod
    def is_mapping_rules(item_property_part):
        # type: (str) -> tuple[bool|str, tuple[str, ...]]
        for mapping_rule_getter_name, tag_to_find in ListOperations.MAPPING_RULES_DEFINITIONS.items():
            match = re.match(tag_to_find[1], item_property_part)
            if match:
                if ListOperations.rule_has_typo_on_concat(item_property_part) and mapping_rule_getter_name != 'concat_values':
                    return False, tuple()
                return mapping_rule_getter_name, match.groups()
        return False, tuple()
    
    
    @staticmethod
    def get_value(item, mapping_rule_getter_name, getter_args, mapper, origin_item):
        # type: (Union[Dict, object], str,Tuple[str], Callable, Union[Dict, object] ) -> Any
        
        return getattr(ListOperations, mapping_rule_getter_name)(item, getter_args, mapper, origin_item)
    
    
    @staticmethod
    def check_property_mapping_value(item_type, property_name, prop_checker):
        # type: (str, str, Callable) -> str
        
        concat_values_rule = ListOperations.MAPPING_RULES_DEFINITIONS['concat_values']
        match = re.match(concat_values_rule[1], property_name)
        if match:
            raw_concat = match.group(1)
            for prop_reference in ListOperations._parse_concat_arg(raw_concat).values():
                check_prop_resul = prop_checker(item_type, prop_reference)
                if check_prop_resul != CHECK_PROPERTY.IS_ALLOW:
                    return check_prop_resul
            
            return CHECK_PROPERTY.IS_ALLOW
        
        return 'NO_RULE_FOUND'
    
    
    @staticmethod
    def remove_mapping_rule_from_property_mapping_value(raw_property_mapping_value):
        # type: (str) -> Union[str, bool]
        
        # """
        # On synchronizer startup, the synchronizer loads the sources and verify if the mapping field matches with the
        # list of collected fields.
        # This function gets a property extracted from the mapping file and returns it without the shinken functions.
        # ex:
        # tags_by_categories.V-NOM_SI.VALUES(=TUTU) -- becomes --> tags_by_categories.V-NOM_SI
        #
        # @param raw_property_mapping_value: str containing the raw property
        # @return: str containing the raw property
        # """
        
        for pattern in ListOperations.MAPPING_RULES_DEFINITIONS.values():
            raw_property_mapping_value = re.sub(pattern[0], '', raw_property_mapping_value)
        
        return raw_property_mapping_value
