#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.basesubprocess import EventHandler
from shinken.log import PartLogger
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.api.synchronizer import ITEM_TYPE, ItemType, SourceTranslatePart, RulesComponent, ValidationState
from shinkensolutions.api.synchronizer.http_lib_external.v01_00 import DefineBy
from shinkensolutions.api.synchronizer.source.api_item_properties import ApiItemProperties
from shinkensolutions.api.synchronizer.source.file_loader import FileLoader
from shinkensolutions.api.synchronizer.source.item import ITEM_TYPE_TO_SOURCE_ITEM
from shinkensolutions.api.synchronizer.source.item.origin_item import OriginItem
from shinkensolutions.api.synchronizer.source.item.source_item import SourceItem
from shinkensolutions.api.synchronizer.source.list_operations import ListOperations
from shinkensolutions.api.synchronizer.source.mapping.origin_to_source.mapper_element import MappingElement
from shinkensolutions.api.synchronizer.source.origin_item_description import OriginItemDescription
from shinkensolutions.api.synchronizer.source.source_configuration_value import SERVICE_MODE
from shinkensolutions.api.synchronizer.source.source_exception import SourceException

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Any, Union, Type, Optional, List
    from shinkensolutions.api.synchronizer.source.file_loader import File
    
    MapperName = str
    PropName = str
    Mapping = Dict[PropName, MappingElement]

UNMAPPED_PROPERTIES = 'unmapped_properties'
UNMAPPED_PROPERTY = 'unmapped_property'
DEFAULT_MAPPER = 'default_mapper'


class FILE_STATUS:
    OK = 'ok'
    ERROR = 'error'
    NOT_EXISTING = 'not_existing'


class ProxyDisableMapper:
    def map(self, origin_item_type, origin_item, origin_raw_data=None, mapper_name=None):
        # type: (ItemType, OriginItem, Any, Optional[List[MapperName]]) -> SourceItem
        raise SourceException('The mapper service is disable. You can activate it with SourceConfiguration.set_service_mapping_origin_to_source.')


class ReloadMapperEvent(EventHandler):
    
    def __init__(self, mapper_origin_to_source):
        # type: (MapperOriginToSource) -> None
        super(ReloadMapperEvent, self).__init__('reload-mapper')
        self.mapper_origin_to_source = mapper_origin_to_source
    
    
    def callback(self):
        self.mapper_origin_to_source.reload_mapper()


class AbstractMapperOriginToSource:
    
    def __init__(self, item_type, mapping=None):
        # type: (ItemType, Mapping) -> None
        self.item_type = item_type
        self.mapping = mapping
    
    
    def map(self, origin_item, raw_data):
        # type: (OriginItem, Any) -> SourceItem
        source_item = self._get_source_item_instance()
        for origin_prop_name, mapping_rule in self.mapping.items():
            if mapping_rule.is_valid and mapping_rule.source_prop:
                origin_item_value = self.get_item_property(origin_item, origin_prop_name, origin_item)
                if origin_item_value:
                    setattr(source_item, mapping_rule.source_prop, origin_item_value)
        
        source_item.raw_data = raw_data
        return source_item
    

    def get_separator(self, path):
        # type: (str) -> str
        # Take a path in input (ex: "shinken.tags_by_category.OS>VALUES(=Linux)") then returns the next separator, here: '.'
        # @param path: string containing the path
        # @return: string containing the separator
        separator = None
        if '.' in path and '>' in path:
            separator = '.' if path.index('.') < path.index('>') else '>'
        elif '.' in path:
            separator = '.'
        elif '>' in path:
            separator = '>'
        return separator
    
    
    def get_item_property(self, item, item_property, origin_item):
        # type: (Union[Dict, object], str, object) -> str
        
        # This function receives an item and a property.
        # The properties are a string possibly containing dots.
        # ex:
        #     this_item_property_is_another_item.another_item_inside_the_item.real_value
        #
        # in that case we have to get through all inner objects to obtain the value.
        #
        # This function is recursive.
        # first call will test if the property 'this_item_property_is_another_item' is an object inside 'item'.
        # if so, the function will call itself recursively to test another_item_inside_the_item the same way.
        #
        # If the object is a dict or a list, returns the dict or the list
        #
        # @param item: the object
        # @param item_property: the property that's going to get tested
        # @return: the property value

        if not item_property:
            return item
        
        if '=>' in item_property:
            item_property = item_property.replace('=>', ListOperations.TRANSFORM_OPERATOR_REPLACEMENT_STRING)
        
        mapping_rule_name, mapping_args = ListOperations.is_mapping_rules(item_property)
        if mapping_rule_name:
            return ListOperations.get_value(item, mapping_rule_name, mapping_args, self.get_item_property, origin_item)
        
        if ListOperations.rule_has_typo_on_concat(item_property):
            return ''
        
        separator = self.get_separator(item_property)
        if separator:
            key, path = item_property.split(separator, 1)
        
        if separator == '.':
            item = self._value_getter(item, key)
            if item:
                return self.get_item_property(item, path, origin_item)
            else:
                return ''
        
        elif separator == '>':
            next_separator = self.get_separator(path)
            if next_separator:
                _, next_path = path.split(next_separator, 1)
            else:
                next_path = None
            mapping_rule_name, mapping_args = ListOperations.is_mapping_rules(path)
            if mapping_rule_name:
                values = self._value_getter(item, key)
                if values:
                    return self.get_item_property(ListOperations.get_value(values, mapping_rule_name, mapping_args, self.get_item_property, item), next_path, origin_item)
                return ''
            else:
                return ''
        else:
            return self._value_getter(item, item_property)
        return ''
    
    
    def _get_source_item_instance(self):
        # type: () -> SourceItem
        raise NotImplementedError()
    
    
    def _value_getter(self, item, item_property):
        # type: (Any, str) -> Any
        raise NotImplementedError()


class GenericDictMapperOriginToSource(AbstractMapperOriginToSource):
    def _get_source_item_instance(self):
        # type: () -> SourceItem
        return ITEM_TYPE_TO_SOURCE_ITEM[self.item_type]()
    
    
    def _value_getter(self, item, item_property):
        # type: (Any, str) -> Any
        return item.get(item_property, '') if hasattr(item, 'get') else ''


class GenericObjectMapperOriginToSource(AbstractMapperOriginToSource):
    def _get_source_item_instance(self):
        # type: () -> SourceItem
        return ITEM_TYPE_TO_SOURCE_ITEM[self.item_type]()
    
    
    def _value_getter(self, item, item_property):
        # type: (Any, str) -> Any
        
        return getattr(item, item_property, '')


class MapperOriginToSource:
    
    def __init__(self, logger, translator, rules_component, file_loader, service_mode, mapper_class, origin_item_description, api_item_properties, mappers_name=None):
        # type: (PartLogger, SourceTranslatePart, RulesComponent,FileLoader, str, Type[AbstractMapperOriginToSource], OriginItemDescription, ApiItemProperties, Optional[List[MapperName]]) -> None
        self._service_mode = service_mode
        self._mapping = {}  # type:Dict[MapperName, Dict[ItemType, Mapping]]
        self._mappers = {}  # type: Dict[MapperName, Dict[ItemType, AbstractMapperOriginToSource]]
        self.logger = logger
        self._rules_component = rules_component
        self._translator = translator
        self.service_mode = service_mode
        self.mappers_name = mappers_name or [DEFAULT_MAPPER]
        self.file_loader = file_loader
        self.mapper_class = mapper_class
        self.user_mapping_files = []  # type: List[Dict]
        self.validation_state = ValidationState(self.logger, self._translator)  # type: ValidationState
        self.origin_item_description = origin_item_description
        self.api_item_properties = api_item_properties
        self.load()
    
    
    def reload_mapper(self):
        self.origin_item_description.reload()
        self.api_item_properties.reload()
        self.load()
    
    
    def load(self):
        # type: () -> None
        self.logger.info('reload mapper')
        new_mapping = {UNMAPPED_PROPERTIES: {}}  # type: Dict[MapperName, Dict[ItemType, Mapping]]
        validation_state = ValidationState(self.logger, self._translator)
        user_mapping_files = []
        
        if self.origin_item_description.validation_state.has_error():
            user_mapping_files.append({'path': self.origin_item_description.user_file.path, 'status': FILE_STATUS.ERROR})
        elif self.origin_item_description.user_file.exist():
            user_mapping_files.append({'path': self.origin_item_description.user_file.path, 'status': FILE_STATUS.OK})
        else:
            user_mapping_files.append({'path': self.origin_item_description.user_file.path, 'status': FILE_STATUS.NOT_EXISTING})
        validation_state.update_from_validation_state(self.origin_item_description.validation_state)
        
        if self.api_item_properties.validation_state.has_error():
            user_mapping_files.append({'path': self.api_item_properties.user_file.path, 'status': FILE_STATUS.ERROR})
        elif self.api_item_properties.user_file.exist():
            user_mapping_files.append({'path': self.api_item_properties.user_file.path, 'status': FILE_STATUS.OK})
        else:
            user_mapping_files.append({'path': self.api_item_properties.user_file.path, 'status': FILE_STATUS.NOT_EXISTING})
        validation_state.update_from_validation_state(self.api_item_properties.validation_state)
        
        for mapper_name in self.mappers_name:
            new_mapping_by_type = {}  # type:Dict[ItemType, Mapping]
            new_mapping[mapper_name] = new_mapping_by_type
            files = self.file_loader.get_named_mapping_rule(mapper_name)
            file_default_mapping_rule = files.default_file
            file_user_mapping_rule = files.user_file
            
            try:
                default_mapping_rules = file_default_mapping_rule.load()
            except Exception as e:
                validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_default_mapping_rule.path, str(e)))
                self.validation_state = validation_state
                return
            if not isinstance(default_mapping_rules, dict):
                validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_default_mapping_rule.path, self._translator.translate('error.fail_to_load_dict')))
                self.validation_state = validation_state
                return
            
            user_mapping_rules = {}
            if self.service_mode != SERVICE_MODE.NOT_OVERLOAD_BY_USER:
                if file_user_mapping_rule.exist():
                    status = FILE_STATUS.OK
                    try:
                        user_mapping_rules = file_user_mapping_rule.load()
                    except Exception as e:
                        validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_user_mapping_rule.path, str(e)))
                        self.validation_state = validation_state
                        status = FILE_STATUS.ERROR
                    status = self._validate_user_mapping_format(user_mapping_rules, validation_state, file_user_mapping_rule, status)
                else:
                    status = FILE_STATUS.NOT_EXISTING
                    validation_state.add_warning(translate_key='warning.missing_user_file', params=(file_user_mapping_rule.path,))
                user_mapping_files.append({'path': file_user_mapping_rule.path, 'status': status})
            
            self._build_mapping(mapper_name, new_mapping_by_type, default_mapping_rules, user_mapping_rules)
        
        self._add_unmapped_properties(new_mapping)
        self._set_description(new_mapping)
        self._validate(new_mapping, validation_state)
        
        for mapper_name in self.mappers_name:
            if mapper_name not in self._mappers:
                self._mappers[mapper_name] = {}
            for item_type in ITEM_TYPE.ALL_TYPES:
                self._mappers[mapper_name][item_type] = self.mapper_class(item_type, new_mapping.get(mapper_name, {}).get(item_type, {}))
        
        self._mapping = new_mapping
        self.validation_state = validation_state
        self.user_mapping_files = user_mapping_files
        return
    
    
    def _validate_user_mapping_format(self, user_mapping_rules, validation_state, file_user_mapping_rule, status):
        # type: (Dict, ValidationState, File, str) -> str
        if isinstance(user_mapping_rules, dict):
            for mapping in user_mapping_rules.values():
                if not isinstance(mapping, dict):
                    validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_user_mapping_rule.path, self._translator.translate('error.fail_to_load_dict')))
                    self.validation_state = validation_state
                    return FILE_STATUS.ERROR
        else:
            validation_state.add_error(translate_key='error.fail_to_load_file', params=(file_user_mapping_rule.path, self._translator.translate('error.fail_to_load_dict')))
            self.validation_state = validation_state
            status = FILE_STATUS.ERROR
        return status
    
    
    def _add_unmapped_properties(self, new_mapping):
        unmapped_properties = new_mapping[UNMAPPED_PROPERTIES]
        for item_type, known_properties in self.api_item_properties.properties_allows.items():
            for known_property in known_properties:
                mappings = self._search_mapping(new_mapping, item_type, known_property)
                if not mappings:
                    if item_type not in unmapped_properties:
                        unmapped_properties[item_type] = {}
                    default_mapper_by_type = unmapped_properties[item_type]
                    default_mapper_by_type[known_property] = MappingElement(item_type, UNMAPPED_PROPERTY, known_property, translator=self._translator, is_mapped=False, api_item_properties=self.api_item_properties)
    
    
    def _set_description(self, new_mapping):
        for item_type, item_description in self.origin_item_description.origin_item_properties_description.items():
            for item_property, property_description in item_description.items():
                mappings = self._search_mapping(new_mapping, item_type, item_property)
                if mappings:
                    for mapping in mappings:
                        mapping.description = property_description
    
    
    @staticmethod
    def _search_mapping(new_mapping, item_type, item_property):
        # type: (Dict[MapperName, Dict[ItemType, Mapping]], ItemType, PropName) -> Optional[List[MappingElement]]
        mappings = []
        for mapping_by_type in new_mapping.values():
            mapping = mapping_by_type.get(item_type, {}).get(item_property, None)
            if mapping:
                mappings.append(mapping)
        return mappings if mappings else None
    
    
    @staticmethod
    def _validate(new_mapping, validation_state):
        found_invalid_key = False
        for mappings in new_mapping.values():
            for mapping in mappings.values():
                for m in mapping.values():
                    if not m.is_valid:
                        validation_state.extra_counter_warning_number += 1
                        found_invalid_key = True
        if found_invalid_key:
            validation_state.add_warning(translate_key='warning.invalid_key_in_mapping', params=(validation_state.extra_counter_warning_number,))
    
    
    def _build_mapping(self, mapper_name, new_mapping, default_mapping_rules, user_mapping_rules):
        # type: (MapperName, Dict[ItemType, Mapping], Dict, Dict) -> None
        
        for item_type, mapping in default_mapping_rules.items():
            by_type_mapping = new_mapping.get(item_type, None)
            if by_type_mapping is None:
                new_mapping[item_type] = by_type_mapping = {}
            for origin_prop, source_prop in mapping.items():
                mapping = MappingElement(item_type, source_prop, origin_prop, is_set_by_user=DefineBy.BY_DEFAULT, mapper_name=mapper_name, translator=self._translator, api_item_properties=self.api_item_properties)
                by_type_mapping[origin_prop] = mapping
        if not isinstance(user_mapping_rules, dict):
            return
        for item_type, mapping in user_mapping_rules.items():
            if not isinstance(mapping, dict):
                continue
            by_type_mapping = new_mapping.get(item_type, None)
            if by_type_mapping is None:
                new_mapping[item_type] = by_type_mapping = {}
            for origin_prop, source_prop in mapping.items():
                mapping = MappingElement(item_type, source_prop, origin_prop, is_set_by_user=DefineBy.BY_USER, mapper_name=mapper_name, translator=self._translator, api_item_properties=self.api_item_properties)
                by_type_mapping[origin_prop] = mapping
    
    
    def have_multi_mapper(self):
        return len(self.mappers_name) != 1
    
    
    def get_mapping(self):
        # type: () -> Dict[MapperName, Dict[ItemType, Mapping]]
        return self._mapping
    
    
    def map(self, origin_item_type, origin_item, origin_raw_data=None, mapper_name=DEFAULT_MAPPER):
        # type: (ItemType, OriginItem, Any, MapperName) -> SourceItem
        if self.validation_state.has_error():
            raise SourceException(self._translator.translate('error.mapper_error', ','.join(self.validation_state.get_errors())))
        source_item_factory = self._mappers[mapper_name][origin_item_type]
        if not origin_raw_data:
            origin_raw_data = origin_item
        return source_item_factory.map(origin_item, origin_raw_data)
