#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.


import getpass

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.ssh_mongodb.mongo_client import MongoClient
from shinkensolutions.ssh_mongodb.mongo_conf import MongoConf
from shinkensolutions.ssh_mongodb.mongo_error import ShinkenMongoException

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Optional


class CommandMongoConf(MongoConf):
    def __init__(
            self,
            mongo_database: 'Optional[str]' = None,
            mongo_host: 'Optional[str]' = None,
            mongo_port: 'Optional[str|int|None]' = None,
            mongo_use_ssh: 'Optional[str]' = None,
            mongo_ssh_key: 'Optional[str]' = None,
            mongo_ssh_user: 'Optional[str]' = None,
            mongo_username: 'Optional[str]' = None,
            mongo_password: 'Optional[str]' = None,
            mongo_ssl: 'Optional[str]' = None,
            mongo_ssl_ca_file: 'Optional[str]' = None,
            mongo_ssl_pem_key_file: 'Optional[str]' = None,
            mongo_ssl_pem_key_password: 'Optional[str]' = None,
            mongo_ssl_crl_file: 'Optional[str]' = None,
            mongo_ssl_allow_invalid_hostnames: 'Optional[str]' = None,
            mongo_ssl_allow_invalid_certificates: 'Optional[str]' = None,
            logger: 'Optional[PartLogger]' = None
    ):
        ssl_parameters = f'''{'?tls=true' if mongo_ssl else ''}{f'&tlsAllowInvalidCertificates=true' if mongo_ssl_allow_invalid_certificates else ''}{'&tlsAllowInvalidHostnames=true' if mongo_ssl_allow_invalid_hostnames else ''}{f'&tlsCAFile={mongo_ssl_ca_file}' if mongo_ssl_ca_file else ''}{f'&tlsCertificateKeyFile={mongo_ssl_pem_key_file}' if mongo_ssl_pem_key_file else ''}{f'&tlsCertificateKeyFilePassword={mongo_ssl_pem_key_password}' if mongo_ssl_pem_key_password else ''}{f'&tlsCRLFile={mongo_ssl_crl_file}' if mongo_ssl_crl_file else ''}'''
        conf = ShinkenModuleDefinition(
            {
                'mongodb_uri'                             : f'''mongodb://{mongo_host if mongo_host else 'localhost'}{'' if mongo_port is None else ':%s' % mongo_port}/{ssl_parameters if mongo_ssl else ''}''',
                'mongodb_username'                        : mongo_username if mongo_username else '',
                'mongodb_password'                        : mongo_password if mongo_password else '',
                'mongodb_replica_set'                     : '',
                'mongodb_use_ssh_tunnel'                  : '1' if mongo_use_ssh else '0',
                'mongodb_ssh_keyfile'                     : mongo_ssh_key if mongo_ssh_key else '~shinken/.ssh/id_rsa',
                'mongodb_ssh_user'                        : mongo_ssh_user if mongo_ssh_user else 'shinken',
                'mongodb_use_ssh_retry_failure'           : '0',
                'mongodb_ssh_tunnel_timeout'              : '0',
                'mongodb_auto_reconnect_max_try'          : '0',
                'mongodb_auto_reconnect_sleep_between_try': '0',
            }
        )
        super(CommandMongoConf, self).__init__(conf, logger=logger, default_database=mongo_database)


def get_mongo_connection(
        command_name: str,
        mongo_database: 'Optional[str]' = 'shinken',
        mongo_host: 'Optional[str]' = None,
        mongo_port: 'Optional[str, int]' = None,
        mongo_use_ssh: 'Optional[str]' = None,
        mongo_ssh_key: 'Optional[str]' = None,
        mongo_ssh_user: 'Optional[str]' = None,
        mongo_username: 'Optional[str]' = None,
        mongo_password: 'Optional[str]' = None,
        mongo_ssl: 'Optional[str]' = None,
        mongo_ssl_ca_file: 'Optional[str]' = None,
        mongo_ssl_pem_key_file: 'Optional[str]' = None,
        mongo_ssl_pem_key_password: 'Optional[str]' = None,
        mongo_ssl_crl_file: 'Optional[str]' = None,
        mongo_ssl_allow_invalid_hostnames: 'Optional[str]' = None,
        mongo_ssl_allow_invalid_certificates: 'Optional[str]' = None,
        logger: 'PartLogger|None' = None
) -> tuple['MongoClient|None', 'CommandMongoConf|None']:
    
    try:
        mongo_conf = CommandMongoConf(
            mongo_database=mongo_database,
            mongo_host=mongo_host,
            mongo_port=mongo_port,
            mongo_use_ssh=mongo_use_ssh,
            mongo_ssh_key=mongo_ssh_key,
            mongo_ssh_user=mongo_ssh_user,
            mongo_username=mongo_username,
            mongo_password=mongo_password,
            mongo_ssl=mongo_ssl,
            mongo_ssl_ca_file=mongo_ssl_ca_file,
            mongo_ssl_pem_key_file=mongo_ssl_pem_key_file,
            mongo_ssl_pem_key_password=mongo_ssl_pem_key_password,
            mongo_ssl_crl_file=mongo_ssl_crl_file,
            mongo_ssl_allow_invalid_hostnames=mongo_ssl_allow_invalid_hostnames,
            mongo_ssl_allow_invalid_certificates=mongo_ssl_allow_invalid_certificates,
            logger=logger
        )
    except Exception as e:
        raise e
    
    try:
        mongodb_con = MongoClient(mongo_conf, log_database_parameters=False, for_command_line=True)
        mongodb_con.init(command_name)
        mongodb_con.get_server_status()
    except ShinkenMongoException as e:
        raise e
    
    return mongodb_con, mongo_conf


def assert_mongo_credentials(mongo_username: 'Optional[str]' = None, mongo_password: 'Optional[str]' = None) -> tuple['str|None', 'str|None']:
    
    if mongo_username and not mongo_password:
        mongo_password = getpass.getpass(prompt='Enter your mongo password: ')
        return mongo_username, mongo_password
    
    elif not mongo_username and not mongo_password:
        return None, None
    
    else:
        return mongo_username, mongo_password


def setup_mongo_connection(command_name: str,
                           mongo_database: 'Optional[str]' = 'shinken',
                           mongo_host: 'Optional[str]' = None,
                           mongo_port: 'Optional[str, int]' = None,
                           mongo_use_ssh: 'Optional[str]' = None,
                           mongo_ssh_key: 'Optional[str]' = None,
                           mongo_ssh_user: 'Optional[str]' = None,
                           mongo_username: 'Optional[str]' = None,
                           mongo_password: 'Optional[str]' = None,
                           mongo_ssl: 'Optional[str]' = None,
                           mongo_ssl_ca_file: 'Optional[str]' = None,
                           mongo_ssl_pem_key_file: 'Optional[str]' = None,
                           mongo_ssl_pem_key_password: 'Optional[str]' = None,
                           mongo_ssl_crl_file: 'Optional[str]' = None,
                           mongo_ssl_allow_invalid_hostnames: 'Optional[str]' = None,
                           mongo_ssl_allow_invalid_certificates: 'Optional[str]' = None) -> tuple['MongoClient|None', 'CommandMongoConf|None']:
    
    mongo_username, mongo_password = assert_mongo_credentials(mongo_username, mongo_password)
    
    mongodb_con, mongo_conf = get_mongo_connection(
        command_name,
        mongo_database=mongo_database,
        mongo_host=mongo_host,
        mongo_port=mongo_port,
        mongo_use_ssh=mongo_use_ssh,
        mongo_ssh_key=mongo_ssh_key,
        mongo_ssh_user=mongo_ssh_user,
        mongo_username=mongo_username,
        mongo_password=mongo_password,
        mongo_ssl=mongo_ssl,
        mongo_ssl_ca_file=mongo_ssl_ca_file,
        mongo_ssl_pem_key_file=mongo_ssl_pem_key_file,
        mongo_ssl_pem_key_password=mongo_ssl_pem_key_password,
        mongo_ssl_crl_file=mongo_ssl_crl_file,
        mongo_ssl_allow_invalid_hostnames=mongo_ssl_allow_invalid_hostnames,
        mongo_ssl_allow_invalid_certificates=mongo_ssl_allow_invalid_certificates
    )
    return mongodb_con, mongo_conf
