#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.


import datetime
from collections import namedtuple

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.component.abstract_component import AbstractComponent
from shinkensolutions.date_helper import TZINFO_UTC, get_datetime_now_as_utc

EXPIRE_FORMAT_IN_COOKIE_DATA = '%x%X'

if TYPE_CHECKING:
    from typing import Protocol, Tuple
    from shinken.misc.type_hint import Final, Optional
    from shinken.webui.bottlewebui import Response, Request
    
    
    class AbstractConfigurationComponentWithAuthSecret(Protocol):
        auth_secret = ''
        app_type = ''
        daemon_name = ''

ParsedCookie = namedtuple('ParsedCookie', ['has_an_error', 'user_auth_duration', 'expires', 'user_id'])

AUTHENTICATION_COOKIE_NAME = 'user'  # type: Final[str]


class CookieComponent(AbstractComponent):
    
    def __init__(self, configuration_component):
        # type: (AbstractConfigurationComponentWithAuthSecret) -> None
        self.configuration_component = configuration_component
        self.auth_secret = ''
        self.cookie_session_name = ''
    
    
    def init(self):
        # self.auth_secret = self.configuration_component.auth_secret.encode('utf-8')
        self.auth_secret = self.configuration_component.auth_secret
        self.cookie_session_name = 'session_%s_%s' % (self.configuration_component.app_type, self.configuration_component.daemon_name)
    
    
    def delete_all_cookies(self, response):
        # type:(Response) -> None
        
        response.delete_cookie(AUTHENTICATION_COOKIE_NAME, secret=self.auth_secret, path='/')
        response.delete_cookie(self.cookie_session_name, secret=self.auth_secret, path='/')
    
    
    @staticmethod
    def make_user_cookie_data(user_id, user_auth_duration):
        # type: (str, int) -> Tuple[str, datetime.datetime]
        
        expires = get_datetime_now_as_utc() + datetime.timedelta(0, user_auth_duration * 60)
        return '%s~%s~%s' % (user_auth_duration, expires.strftime(EXPIRE_FORMAT_IN_COOKIE_DATA), user_id), expires
    
    
    def set_user_cookie(self, user_id, user_auth_duration, response, is_webui=False):
        # type:(str, int, Response, bool) -> None
        
        cookie_data, expires = self.make_user_cookie_data(user_id, user_auth_duration)
        if is_webui:
            expires = expires.strftime("%a, %d %b %Y %H:%M:%S %Z")
        
        response.set_cookie(AUTHENTICATION_COOKIE_NAME, cookie_data, secret=self.auth_secret, path='/', expires=expires, httponly=True)
    
    
    def set_session_cookie(self, user_id, response):
        # type:(str, Response) -> None
        
        response.set_cookie(self.cookie_session_name, user_id, secret=self.auth_secret, path='/', httponly=True)
    
    
    def set_cookies(self, user_id, user_auth_duration, response, is_webui=False):
        # type:(str, int, Response, bool) -> None
        
        self.set_user_cookie(user_id, user_auth_duration, response, is_webui)
        self.set_session_cookie(user_id, response)
    
    
    def get_session_cookie(self, request):
        # type: (Request) -> ParsedCookie
        return request.get_cookie(self.cookie_session_name, secret=self.auth_secret)
    
    
    def get_and_parse_user_cookie(self, request):
        # type: (Request) -> ParsedCookie
        cookie_data = request.get_cookie(AUTHENTICATION_COOKIE_NAME, secret=self.auth_secret)
        return self.parse_user_cookie(cookie_data)
    
    
    @staticmethod
    def parse_user_cookie(cookie_data=None):
        # type: (Optional[str]) -> ParsedCookie
        has_an_error = False
        user_auth_duration, expires, user_id = None, None, None
        
        if cookie_data is not None:
            try:
                user_auth_duration, expires, user_id = cookie_data.split('~', 2)
                if datetime.datetime.strptime(expires, EXPIRE_FORMAT_IN_COOKIE_DATA).replace(tzinfo=TZINFO_UTC) < get_datetime_now_as_utc():
                    has_an_error = True
            except ValueError:
                has_an_error = True
        
        return ParsedCookie(has_an_error, user_auth_duration, expires, user_id)
