#!/usr/bin/env python3-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2025
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.lib_checks.common import Result, EXIT_STATUS, COLOR, BREAK_LINE, HTMLTag, HTMLTable, Utils

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any


class BrokerModuleChecks:
    
    @staticmethod
    def get_last_unavailability_in_long_output(result: 'Result', broker_module_stats: 'dict[str, Any]', nb_line_to_show: int) -> None:
        module_status = EXIT_STATUS.OK
        output = ''
        last_unavailability_duration = broker_module_stats.get('last_unavailability_time_when_new_configuration', {})
        
        if not last_unavailability_duration:
            result.add_check(EXIT_STATUS.OK, long_output='No data for last unavailability time on new configuration')
            return
        
        headers = [
            'Configuration creation date',
            'Arbiter name',
            'Architecture name',
            f'''Configuration uuid{BREAK_LINE}{HTMLTag.gray_note('( For log analysis )')}''',
            'Unavailability start time',
            'Received monitoring configuration parts',
            'Report builder unavailability duration',
        ]
        lines = []
        
        data_list = sorted(last_unavailability_duration.values(), key=lambda c: c.get('unavailability_start_time', -1), reverse=True)
        
        nb_to_show = min(len(data_list), nb_line_to_show)
        for data in data_list[0:nb_to_show]:
            if data.get('nb_conf_to_handle'):
                part_handle = data.get('nb_conf_handle')
                part_to_handle = data.get('nb_conf_to_handle')
                extra_part_message = ''
                if part_handle < part_to_handle:
                    extra_part_message = '( UNCOMPLETED )' if lines else '( ONGOING )'
                part_msg = f'{part_handle} / {part_to_handle} {extra_part_message}'.strip()
            else:
                part_msg = f'''This configuration was not received from Arbiter, but we process {data.get('nb_conf_handle')} part.'''
            lines.append([
                Utils.print_time(data.get('creation_date', 0)),
                data.get('author', ''),
                data.get('architecture_name', ''),
                HTMLTag.color_text(data.get('uuid', ''), COLOR.GRAY, bold=False, italic=True),
                Utils.print_time(data.get('unavailability_start_time', 0)),
                part_msg,
                f'''{float(data.get('unavailability_time_when_new_configuration', 0.0)):.3f}s'''
            ])
        
        _maximum_nb_lines_label = HTMLTag.gray_note(f'(maximum {nb_line_to_show})')
        result.add_check(module_status, output=output, long_output=HTMLTable.table(headers, lines, title=f'This is the last configurations received {_maximum_nb_lines_label} and the unavailability time the receipt cause for the module'))
    
    
    @staticmethod
    def add_last_unavailability_metrics(result: 'Result', broker_module_stats: 'dict[str, Any]') -> None:
        item_counter = broker_module_stats.get('item_counter', {})
        result.add_perf_data('unavailability_time_per_minute', f'''{broker_module_stats.get('unavailability_duration', 0.0):.3f}''')
        result.add_perf_data('nb_hosts', item_counter.get('hosts', 0))
        result.add_perf_data('nb_clusters', item_counter.get('clusters', 0))
        result.add_perf_data('nb_checks', item_counter.get('checks', 0))
        result.add_perf_data('nb_contacts', item_counter.get('contacts', 0))
