#!/usr/bin/env python3-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2025
# This file is part of Shinken Enterprise, all rights reserved.

from itertools import islice
from typing import TYPE_CHECKING

from shinkensolutions.date_helper import get_datetime_with_local_time_zone
from shinkensolutions.lib_checks.common import Result, Utils, HTMLList, HTMLTable, HTMLTag, BREAK_LINE, COLOR
from shinkensolutions.shinken_time_helper import print_human_readable_period as _original_print_human_readable_period, print_human_readable_date_time, DisplayFormat

if TYPE_CHECKING:
    from typing import Any, TypedDict
    
    
    class ReportGenerationStats(TypedDict):
        number: int
        percentage: int
    
    
    class PendingReportStatistics(TypedDict):
        uuid: str
        user: str
        format: str
        data_type: str
        search: str
        date_begin: str
        date_end: str
        request_start_timestamp: int
    
    
    class RunningReportStatistics(PendingReportStatistics, total=False):
        number_of_elements: int
        generation_start_timestamp: int
        data_fetch_time: float
    
    
    class DoneReportStatistics(RunningReportStatistics):
        generation_end_timestamp: int
        total_generation_time: float

REPORT_DATA_TYPE_LABEL = {
    'history'    : 'History',
    'sla'        : 'SLA',
    'history_sla': 'History &amp; SLA',
}

_TIME_DISPLAY_FORMAT = DisplayFormat('', '%.2fs', '%sm', '%sh', '%s days')
_TIME_DISPLAY_FORMAT_NO_MS = _TIME_DISPLAY_FORMAT._replace(s='%.0fs')


def print_human_readable_period(time_period: 'int|float', display_format=_TIME_DISPLAY_FORMAT) -> str:
    return _original_print_human_readable_period(time_period, time_format='h', display_format=display_format) or '0s'


class ReportModuleChecks:
    IN_LAST_MINUTE = HTMLTag.gray_note('(in last minute)')
    IN_LAST_24H = HTMLTag.gray_note('(in last 24h)')
    
    
    @staticmethod
    def parse_report_builder_name(full_name: str) -> 'tuple[str, str]':
        daemon_name, _, module_name = full_name.partition('::')
        daemon_name = Utils.escape_XSS(daemon_name)
        module_name = Utils.escape_XSS(module_name)
        return daemon_name, module_name
    
    
    @classmethod
    def add_in_progress_reports(cls, result: 'Result', report_builder_stats: 'dict[str,Any]') -> None:
        running_reports: 'list[RunningReportStatistics]|None' = report_builder_stats.get('running_reports')
        if not running_reports:
            return
        
        headers = (
            'User',
            'Format',
            'Type',
            'Search',
            'Date Range',
            f'''Report ID{BREAK_LINE}{HTMLTag.gray_note('( For log analysis )')}''',
            'Number of elements',
            'Requested at',
        )
        lines = [
            (
                r['user'],
                r['format'],
                REPORT_DATA_TYPE_LABEL.get((data_type := r['data_type']), data_type),
                r['search'],
                f'''{r['date_begin']} to{BREAK_LINE}{r['date_end']}''',
                HTMLTag.color_text(r['uuid'], COLOR.GRAY, bold=False, italic=True),
                r['number_of_elements'],
                print_human_readable_date_time(get_datetime_with_local_time_zone(r['request_start_timestamp'])),
            )
            for r in running_reports
        ]
        result.add_check(output=HTMLTable.table(headers, lines, f'''Report{'s' if len(running_reports) > 1 else ''} in progress'''))
    
    
    @classmethod
    def check_module_performance(cls, result: 'Result', report_builder_stats: 'dict[str,Any]', work_load_on_type: str) -> None:
        pending_reports: 'list[PendingReportStatistics]' = report_builder_stats.get('pending_reports', [])
        running_reports: 'list[RunningReportStatistics]' = report_builder_stats.get('running_reports', [])
        next_pending_report_wait_time: int = report_builder_stats.get('next_pending_report_wait_time', 0)
        nb_done_reports_in_last_24h: int = report_builder_stats.get('nb_done_reports_in_last_24h', 0)
        nb_done_reports_in_last_minute: int = report_builder_stats.get('nb_done_reports_in_last_minute', 0)
        average_report_generation_time_in_last_24h: float = report_builder_stats.get('average_report_generation_time_in_last_24h', 0.0)
        average_report_pending_time_in_last_24h: float = report_builder_stats.get('average_report_pending_time_in_last_24h', 0.0)
        report_generation_top: 'list[DoneReportStatistics]' = report_builder_stats.get('report_generation_top', [])
        report_generation_ranges: 'dict[tuple[int, int|None], ReportGenerationStats]' = cls._compute_report_generation_ranges({int(r): v for r, v in report_builder_stats.get('report_generation_stats', {}).items()})
        work_load_percentage_in_last_minute: float = report_builder_stats.get('work_load_percentage_in_last_minute', 0.0)
        
        # Should already be sorted, but it is for the consistency
        report_generation_top.sort(key=lambda r: r.get('total_generation_time', 0.0), reverse=True)
        
        # -*- Output -*-
        cls._add_reports_stats_summary(
            result,
            pending_reports=pending_reports,
            next_pending_report_wait_time=next_pending_report_wait_time,
            nb_done_reports_in_last_24h=nb_done_reports_in_last_24h,
            average_report_pending_time_in_last_24h=average_report_pending_time_in_last_24h,
            average_report_generation_time_in_last_24h=average_report_generation_time_in_last_24h,
            work_load_percentage_in_last_minute=work_load_percentage_in_last_minute
        )
        
        # -*- Long Output -*-
        cls.add_generation_top_in_long_output(result, report_generation_top)
        cls._add_generation_stats_in_long_output(result, report_generation_ranges)
        
        # -*- Metrics -*-
        cls._add_performance_metrics(
            result,
            pending_reports=pending_reports,
            next_pending_report_wait_time=next_pending_report_wait_time,
            running_reports=running_reports,
            report_generation_ranges=report_generation_ranges,
            nb_done_reports_in_last_24h=nb_done_reports_in_last_24h,
            nb_done_reports_in_last_minute=nb_done_reports_in_last_minute,
            average_report_pending_time_in_last_24h=average_report_pending_time_in_last_24h,
            average_report_generation_time_in_last_24h=average_report_generation_time_in_last_24h,
            work_load_percentage_in_last_minute=work_load_percentage_in_last_minute,
            work_load_on_type=work_load_on_type,
        )
    
    
    @staticmethod
    def _compute_report_generation_ranges(report_generation_stats: 'dict[int, ReportGenerationStats]') -> 'dict[tuple[int, int|None], ReportGenerationStats]':
        report_generation_ranges: 'dict[tuple[int, int|None], ReportGenerationStats]' = {}
        if report_generation_stats:
            exec_stats_range = sorted(v for v in report_generation_stats if v > 0)
            start_range = 0
            for end_range in exec_stats_range:
                report_generation_ranges[start_range, end_range] = report_generation_stats[end_range]
                start_range = end_range
            report_generation_ranges[start_range, None] = report_generation_stats[-1]
        return report_generation_ranges
    
    
    @classmethod
    def _add_reports_stats_summary(
            cls,
            result: 'Result',
            *,
            pending_reports: 'list[PendingReportStatistics]',
            next_pending_report_wait_time: int,
            nb_done_reports_in_last_24h: int,
            average_report_pending_time_in_last_24h: float,
            average_report_generation_time_in_last_24h: float,
            work_load_percentage_in_last_minute: float,
    ) -> None:
        nb_pending_reports: int = len(pending_reports)
        report_stats: 'list[str]' = [
            HTMLList.header_list(f'''Work load: {work_load_percentage_in_last_minute:.2f}%''', [
                HTMLTag.gray_note('(Time spent generating a report in last 60 seconds in %)')
            ]),
            HTMLList.header_list(f'Number of pending reports: {nb_pending_reports}', [
                f'Next pending report is waiting since: {print_human_readable_period(next_pending_report_wait_time)}' if nb_pending_reports else '',
                HTMLList.header_list(f'Average pending time {cls.IN_LAST_24H} : {print_human_readable_period(average_report_pending_time_in_last_24h)}', [
                    HTMLTag.gray_note(f'''(Delay between {HTMLTag.bold_text("request start")} and {HTMLTag.bold_text("generation start")}.)''')
                ]),
            ], filter_empty=True),
            HTMLList.header_list(f'Number of done reports {cls.IN_LAST_24H} : {nb_done_reports_in_last_24h}', [
                f'Average report generation time: {print_human_readable_period(average_report_generation_time_in_last_24h)}',
            ]),
        ]
        result.add_check(output=HTMLTable.table([], [[HTMLList.simple_list(report_stats)]], left_headers=['Report stats summary'], all_col_same_width=False))
    
    
    @classmethod
    def add_generation_top_in_long_output(
            cls,
            result: 'Result',
            report_generation_top: 'list[DoneReportStatistics]',
    ) -> None:
        headers = ('User', 'Format', 'Type', 'Search', 'Date Range', 'Number of elements', 'Pending time', 'Generation time', 'Generation date')
        lines = [
            (
                r['user'],
                r['format'],
                REPORT_DATA_TYPE_LABEL.get((data_type := r['data_type']), data_type),
                r['search'],
                f'''{r['date_begin']} to{BREAK_LINE}{r['date_end']}''',
                r['number_of_elements'],
                print_human_readable_period(r['generation_start_timestamp'] - r['request_start_timestamp'], _TIME_DISPLAY_FORMAT_NO_MS),
                HTMLList.simple_list([
                    f'''Data fetch time: {print_human_readable_period(r['data_fetch_time'])}''',
                    f'''{HTMLTag.bold_text('Total')}: {print_human_readable_period(r['total_generation_time'])}'''
                ]),
                print_human_readable_date_time(get_datetime_with_local_time_zone(r['generation_end_timestamp'])),
            )
            for r in islice(report_generation_top, 5)
        ]
        
        if lines:
            result.add_check(long_output=HTMLTable.table(headers, lines, f'Top 5 report generation {cls.IN_LAST_24H}'))
    
    
    @classmethod
    def _add_generation_stats_in_long_output(
            cls,
            result: 'Result',
            report_generation_ranges: 'dict[tuple[int, int|None], ReportGenerationStats]',
    ) -> None:
        headers: 'list[str]' = []
        line: 'list[str]' = []
        
        for (start_range, end_range), time_range_stats in report_generation_ranges.items():
            if end_range is None:
                headers.append(f'+ {start_range}s')
            else:
                headers.append(f'{start_range}-{end_range}s')
            line.append(f'''{time_range_stats['number']}{BREAK_LINE}( {time_range_stats['percentage']}% )''')
        
        if line:
            result.add_check(long_output=HTMLTable.table(headers, [line], f'Number of reports per generation time {cls.IN_LAST_24H}'))
    
    
    @classmethod
    def _add_performance_metrics(
            cls,
            result: 'Result',
            *,
            pending_reports: 'list[PendingReportStatistics]',
            next_pending_report_wait_time: int,
            running_reports: 'list[RunningReportStatistics]',
            report_generation_ranges: 'dict[tuple[int, int|None], ReportGenerationStats]',
            nb_done_reports_in_last_24h: int,
            nb_done_reports_in_last_minute: int,
            average_report_pending_time_in_last_24h: float,
            average_report_generation_time_in_last_24h: float,
            work_load_percentage_in_last_minute: float,
            work_load_on_type: str,
    ) -> None:
        result.add_perf_data('nb_pending_reports', len(pending_reports))
        result.add_perf_data('nb_running_reports', len(running_reports))
        result.add_perf_data('nb_done_reports_in_last_24h', nb_done_reports_in_last_24h)
        result.add_perf_data('nb_done_reports_in_last_minute', nb_done_reports_in_last_minute)
        result.add_perf_data('next_pending_report_wait_time', f'{next_pending_report_wait_time}s')
        result.add_perf_data('average_report_pending_time_in_last_24h', f'{average_report_pending_time_in_last_24h:.2f}s')
        result.add_perf_data('average_report_generation_time_in_last_24h', f'{average_report_generation_time_in_last_24h:.2f}s')
        result.add_perf_data(f'work_load_on_{work_load_on_type}_in_last_minute', f'{work_load_percentage_in_last_minute:.2f}%')
        for (start_range, end_range), time_range_stats in report_generation_ranges.items():
            nb_entries = time_range_stats['number']
            if end_range is None:
                result.add_perf_data(f'reports_per_generation_time_{start_range}_s_and_more', nb_entries)
            else:
                result.add_perf_data(f'reports_per_generation_time_{start_range}_{end_range}_s', nb_entries)
