#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import atexit
import configparser
import json
import os
import shutil
import socket
import subprocess
import sys
import time
import traceback
import uuid
from enum import StrEnum
from hashlib import md5
from io import StringIO
from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from typing import Any

import psutil

if os.name != 'nt':
    CONTEXT_PATH = '/var/lib/shinken/context.json'
else:  # specific path for windows
    CONTEXT_PATH = r'c:\shinken\var\context.json'
VERSION = '2.03.02'  # DEFAULT VALUE before we have a context file, so very old

WARNING_COLOR = 35
ERROR_COLOR = 31
OK_COLOR = 32

# What to print in a shell string to have colors.
# IMPORTANT: always set the RESET at the end of your string
WARNING_SHELL_TAG = '\033[%dm' % WARNING_COLOR
OK_SHELL_TAG = '\033[%dm' % OK_COLOR
ERROR_SHELL_TAG = '\033[%dm' % ERROR_COLOR
RESET_SHELL_TAG = '\033[0m'

if os.path.exists(CONTEXT_PATH):
    _context = json.loads(open(CONTEXT_PATH, 'r').read())
    VERSION = str(_context.get('current_version', VERSION).split('-')[0])
    VERSION = VERSION.replace('.fr', '').replace('.us', '')

# ENV variable to force unset at all cost
unset_variables = ['http_proxy', 'https_proxy', 'ftp_proxy']
for v in os.environ:
    if v.lower() in unset_variables:
        os.environ[v] = ''

stdout = sys.stdout

devnull = open(os.devnull, 'w')

atexit.register(devnull.close)


def protect_stdout():
    sys.stdout = devnull


def unprotect_stdout():
    sys.stdout = stdout


ADDITIONAL_DAEMONS = ['gatherer']
POSSIBLE_DAEMONS = sorted(['poller', 'synchronizer', 'scheduler', 'reactionner', 'receiver', 'broker', 'arbiter'])
# linker and provider are not ready yet

POSSIBLE_ADDONS = sorted(['nagvis', 'nagvis-shinken-architecture'])

POSSIBLE_DATA = ['configuration', 'sla', 'metrology', 'user', 'events', 'modules']

INSTALLATION_TYPES = ['INSTALLATION', 'UPDATE', 'FIX', 'RESTORE', 'SANITIZE', 'PATCH', 'ADDON', 'UNPATCH', 'REMOVE ADDON']


class SANITIZE_INFO_CONTEXT_FLAG(StrEnum):
    ADD_WEBUI_MODULE_REPORT_HANDLER = 'add_webui_module_report_handler'


def get_local_daemons():
    if not os.path.exists(CONTEXT_PATH):
        return POSSIBLE_DAEMONS
    with open(CONTEXT_PATH, 'r') as f:
        context = json.loads(f.read())
    local_daemons = [k for (k, installed) in context['daemons'].items() if installed]
    return local_daemons


def get_local_addons():
    ctx = __get_context()
    return ctx['addons']


def get_local_instances_for_type(daemon_type):
    context = __get_context()
    return list(context['local_instances'][daemon_type].items())


def get_shinken_current_version():
    # type: () -> str
    context = __get_context()
    return context['current_version'] if context['current_version'].startswith('V') else 'V%s' % context['current_version']


def get_shinken_current_patch():
    context = __get_context()
    last_install = context['installation_history'][-1]
    if last_install.get('type') == 'PATCH':
        return last_install.get('patch_name') if last_install.get('patch_name').startswith('V') else 'V%s' % last_install.get('patch_name')
    # There is no patch currently installed, returning None
    return None


def get_shinken_current_version_and_patch():
    current_version = get_shinken_current_version()
    current_patch = get_shinken_current_patch()
    if current_patch is not None:
        return 'Version : %s\nPatch   : %s' % (current_version, current_patch)
    return 'Version : %s' % current_version


def get_shinken_raw_current_version():
    context = __get_context()
    current_version = context['current_version']
    last_install = context['installation_history'][-1]
    return current_version, last_install.get('patch_name', current_version)


def get_shinken_current_short_version():
    # type: () -> str
    context = __get_context()
    version = context['current_version'] if context['current_version'].startswith('V') else 'V%s' % context['current_version']
    version_suffix = version[10:12]
    return version[0:13] if version_suffix.isdigit() else version[0:9]


# Return the next id for the daemon typ, but as a STRING because json does not manage
# keys as integers
def get_next_local_instance_number_available(daemon_type):
    context = __get_context()
    # Look if there is a hole in the ids currently set
    ids = sorted(context['local_instances'][daemon_type].keys())
    for i in range(len(ids)):
        if str(i) not in ids:
            return str(i)
    # All is filled? get a new id so
    return str(len(ids))


# AT INSTALL: create the context file is not exiting already
# will set install version inside, and only enabled daemons (maybe all)
# and all addons will be enabled (cannot choose at install)
def create_context_file(code_version, installation_path, enabled_daemons, installer_version):
    # If not root, get away moron
    check_root()
    
    if os.path.exists(CONTEXT_PATH):
        raise Exception('The context file %s already exists' % CONTEXT_PATH)
    now = int(time.time())
    context = {'creation_time'            : now,
               'install_path'             : installation_path,
               'daemons'                  : {},  # OLD structure entry, to clean when all the U01 versions are out of scope
               'local_instances'          : {},
               'addons'                   : {},
               'addons_history'           : {},
               'installed_version'        : code_version,  # VERY FIRST install version
               'current_version'          : code_version,  # CURRENT running version
               'current_installer_version': installer_version,  # LAST installer version used, can be "" if standard release
               'installation_history'     : [],
               'data_history'             : {},
               'sanitize_info'            : {},
               }
    for data_type in POSSIBLE_DATA:
        context['data_history'][data_type] = []
    
    # Enable daemons only if ask at creation (central role= all daemons are ON)
    for daemon_type in POSSIBLE_DAEMONS:
        is_enabled = (daemon_type in enabled_daemons) or 'central' in enabled_daemons
        # enabled (or not) the global type, AND the instance 0
        context['daemons'][daemon_type] = is_enabled
        context['local_instances'][daemon_type] = {'0': is_enabled}
    
    __add_installation_version(context, code_version, 'INSTALLATION', now, installer_version=installer_version)
    # At creation all addons are enabled, no choice
    for addon_name in POSSIBLE_ADDONS:
        context['addons'][addon_name] = True
    
    for data_type in POSSIBLE_DATA:
        __add_data_version(data_type, context, code_version, 'INSTALLATION', now)
    
    __set_default_sanitize_info_flags(context['sanitize_info'])
    
    __write_context(context)


def __set_default_sanitize_info_flags(sanitize_info: 'dict[str, int]') -> None:
    sanitize_info[SANITIZE_INFO_CONTEXT_FLAG.ADD_WEBUI_MODULE_REPORT_HANDLER] = 1


def __get_windows_server_identity():
    # type: () -> str
    macs = []
    for interface_name, interface_info in psutil.net_if_addrs().items():
        interface_info = interface_info[0]
        if interface_info.family == psutil.AF_LINK:
            macs.append(interface_info.address)
    to_hash = ''.join(sorted(macs)).encode('utf-8')
    mac_hash = md5(to_hash).hexdigest()
    return 'sk-%s-%s' % (mac_hash, uuid.uuid4())


def __get_linux_server_identity():
    # type: () -> str
    macs = []
    product_uuid_path = '/sys/class/dmi/id/product_uuid'
    
    if os.path.exists(product_uuid_path):
        with open(product_uuid_path) as fp:
            server_uuid = fp.readline().strip().lower()
    else:
        server_uuid_path = '/var/lib/shinken/server.uuid'
        if os.path.exists(server_uuid_path):
            with open(server_uuid_path) as fp:
                server_uuid = fp.readline().strip().lower()
        else:
            server_uuid = ''
        for iface in os.listdir('/sys/class/net/'):
            try:
                with open('/sys/class/net/%s/address' % iface) as fp:
                    macs.append(fp.readline().strip().lower())
            except IOError:
                # This interface doesn't have any HW address, just skip it
                pass
        mac_hash = md5(''.join(sorted(macs))).hexdigest()
        if server_uuid.count('-') == 0 or mac_hash != server_uuid.split('-')[1]:
            server_uuid = 'sk-%s-%s' % (mac_hash, uuid.uuid4())
    return server_uuid


# returns the server identity, so we know on which server it was installed before:
# server_name = hostname of the server
# server_uuid = bios uuid in lower, same as the arbiter one in the init.d script
#               if the bios uuid is not available, hash of MAC addresses and uuid
def __get_server_identity():
    try:
        server_name = socket.gethostname()
    except Exception:
        server_name = 'unknown'
    try:
        if sys.platform.startswith('win'):
            server_uuid = __get_windows_server_identity()
        else:
            server_uuid = __get_linux_server_identity()
    except Exception:
        traceback.print_exc()
        server_uuid = 'unknown'
    return server_name, server_uuid


# Add an installation entry in the context['installation_history']
# An entry looks like:
#  {'type': 'INSTALLATION', 'date':EPOCH, 'version':VERSION, 'server_name': NOM_SERVER, 'server_uuid': UUID_SERVER, 'installer_version': STR}
def __add_installation_version(_context, installation_version, installation_type, installation_epoch, installer_version):
    if installation_type not in INSTALLATION_TYPES:
        raise ValueError('The installation type %s is not in the allowed list' % installation_type)
    
    installation_histories = _context['installation_history']
    server_name, server_uuid = __get_server_identity()
    installation_entry = {'type'             : installation_type,
                          'date'             : int(installation_epoch),
                          'version'          : installation_version,
                          'server_name'      : server_name,
                          'server_uuid'      : server_uuid,
                          'installer_version': installer_version
                          }
    installation_histories.append(installation_entry)


# Add an installation entry in the context['data_history']
# An entry looks like:
#  {'type': 'INSTALLATION', 'date':EPOCH, 'version':VERSION, 'server_name': NOM_SERVER, 'server_uuid': UUID_SERVER}
def __add_data_version(data_type, _context, data_version, installation_type, installation_epoch, backup_path='', sanitize_name='', comment='', installer_version=''):
    if data_type not in POSSIBLE_DATA:
        raise ValueError(f'Invalid data type "{data_type}"')
    if installation_type not in INSTALLATION_TYPES:
        raise ValueError('The installation type %s is not in the allowed list' % installation_type)
    
    histories = _context['data_history'].get(data_type, [])
    if not histories:
        _context['data_history'][data_type] = histories
        installation_type = 'INSTALLATION'
    server_name, server_uuid = __get_server_identity()
    installation_entry = {'type'             : installation_type,
                          'date'             : int(installation_epoch),
                          'version'          : data_version,
                          'server_name'      : server_name,
                          'server_uuid'      : server_uuid,
                          'backup_path'      : backup_path,
                          'sanitize_name'    : sanitize_name,
                          'comment'          : comment,
                          'installer_version': installer_version,
                          }
    histories.append(installation_entry)


# Add an installation entry in the context['installation_history']
# An entry looks like:
#  {'type': 'INSTALLATION', 'date':EPOCH, 'version':VERSION, 'server_name': NOM_SERVER, 'server_uuid': UUID_SERVER}
def __add_patch_install(data_type, _context, data_version, installation_type, installation_epoch, backup_path='', patch_name=''):
    if installation_type not in INSTALLATION_TYPES:
        raise ValueError('The Patch type %s is not in the allowed list' % installation_type)
    
    if data_type == 'installation':
        histories = _context['installation_history']
    else:
        histories = _context['data_history'].get(data_type, [])
    server_name, server_uuid = __get_server_identity()
    installation_entry = {'type'       : installation_type,
                          'date'       : int(installation_epoch),
                          'version'    : data_version,
                          'server_name': server_name,
                          'server_uuid': server_uuid,
                          'backup_path': backup_path,
                          'patch_name' : patch_name
                          }
    histories.append(installation_entry)


def __add_addons_version(addon_type, _context, installation_type, installation_epoch, old_map_uuid, new_map_uuid):
    if installation_type not in INSTALLATION_TYPES:
        raise ValueError('The installation type %s is not in the allowed list' % installation_type)
    
    histories = _context['addons_history'][addon_type]
    server_name, server_uuid = __get_server_identity()
    installation_entry = {'type'        : installation_type,
                          'date'        : int(installation_epoch),
                          'server_name' : server_name,
                          'server_uuid' : server_uuid,
                          'old_map_uuid': old_map_uuid,
                          'new_map_uuid': new_map_uuid
                          }
    histories.append(installation_entry)


def __sanitize_context_data(_context: 'dict[str, Any]') -> None:
    # Fix missing daemon types (like after an update), by default disable them
    for dtype in POSSIBLE_DAEMONS:
        if dtype not in _context['daemons']:
            _context['daemons'][dtype] = False
    
    # For missing addons, enable them by default
    if _context.get("addons") is None:
        _context['addons'] = {}
    for addon_type in POSSIBLE_ADDONS:
        if addon_type not in _context['addons']:
            _context['addons'][addon_type] = True
    
    if 'addons_history' not in _context:
        _context['addons_history'] = {}
    for addon_type in POSSIBLE_ADDONS:
        if addon_type not in _context['addons_history']:
            _context['addons_history'][addon_type] = []
    
    # Maybe there was central = True, if so, fix it
    if _context['daemons'].get('central', False):
        for dtype in _context['daemons']:
            _context['daemons'][dtype] = True
        _context['daemons']['central'] = False
    
    # set local_instances dict if missing
    if 'local_instances' not in _context:
        _context['local_instances'] = {}
    for (dtype, enabled) in _context['daemons'].items():
        if dtype not in _context['local_instances']:
            _context['local_instances'][dtype] = {'0': enabled}
    
    if 'installation_history' not in _context:
        _context['installation_history'] = []
        # Stack the installed version, at the creation_date
        installed_version = _context['installed_version']
        __add_installation_version(_context, installed_version, 'INSTALLATION', int(_context['creation_time']), installer_version='')
        # Stack the current version if different
        current_version = _context['current_version']
        if current_version != installed_version:  # if there was an update in the past, set it, we just don't know when so fake the date
            __add_installation_version(_context, current_version, 'UPDATE', 0, installer_version='')
    
    installed_version = _context['installed_version']
    current_version = _context['current_version']
    if 'data_history' not in _context:
        _context['data_history'] = {}
    
    # This parts that already exist in old version but not in context
    _02_03_03_format_data_type = ['configuration', 'sla', 'metrology', 'user']
    for data_type in _02_03_03_format_data_type:
        if data_type not in _context['data_history']:
            _context['data_history'][data_type] = []
            # Stack the installed version, at the creation_date
            __add_data_version(data_type, _context, installed_version, 'INSTALLATION', int(_context['creation_time']))
            # Stack the current version if different
            if current_version != installed_version:  # if there was an update in the past, set it, we just don't know when so display UNKNOWN DATE
                __add_data_version(data_type, _context, current_version, 'UPDATE', 0)
    
    for data_type in list(_context['data_history'].keys()):
        update_me = False
        history_entries = []
        entry: dict
        for entry in _context['data_history'][data_type]:
            if entry.get('type') == 'SANATIZE':
                entry['type'] = 'SANITIZE'
                update_me = True
            if 'sanatize_name' in entry:
                entry['sanitize_name'] = entry.pop('sanatize_name')
                update_me = True
            history_entries.append(entry)
        if update_me:
            _context['data_history'][data_type] = history_entries
    
    # sanitize flags used by sanitize_data.py and install.sh
    # to check if some migration had been done.
    _context.setdefault('sanitize_info', {})


def get_installation_history():
    __assert_context()
    context = __get_context()
    return context['installation_history']


def get_data_history(data_type: str) -> 'list[dict[str, Any]]':
    __assert_context()
    if data_type not in POSSIBLE_DATA:
        raise ValueError(f'Invalid data type "{data_type}"')
    context = __get_context()
    return context['data_history'].get(data_type, [])


def sanitize_context_file() -> None:
    check_root()
    __assert_context()
    context = __get_context()  # already sanitize it
    # so just rewrite it, and we are clean
    __write_context(context)


def __get_context() -> 'dict[str, Any]':
    with open(CONTEXT_PATH, 'r') as f:
        context = json.load(f)
    
    # When we load the context file, always be sure that it was sanitized from old versions structure
    __sanitize_context_data(context)
    
    return context


def __write_context(context: 'dict[str, Any]') -> None:
    with open('/var/lib/shinken/context.json.tmp', 'w') as f:
        json.dump(context, f, indent=4)
    shutil.move('/var/lib/shinken/context.json.tmp', '/var/lib/shinken/context.json')


# If context file is missing, exit
def __assert_context() -> None:
    if not os.path.exists('/var/lib/shinken/context.json'):
        print("ERROR: Missing context file, you cannot use this command currently")
        sys.exit(2)


# For some modules operation and cache, we need to have the full file hash, so if something did change
# like at install, or even at patch install, to know if the cache need to be reset (like for webui)
def get_context_hash():
    with open(CONTEXT_PATH, 'rb') as f:
        context_hash = md5(f.read()).hexdigest()
        return context_hash


def set_local_daemon(daemon, b):
    __assert_context()
    context = __get_context()
    # Now really set the value for our daemon type
    context['daemons'][daemon] = b
    __write_context(context)


def set_local_addon(addon, b):
    __assert_context()
    context = __get_context()
    context['addons'][addon] = b
    __write_context(context)


def set_local_daemon_instance(daemon_type, daemon_id, enabled):
    __assert_context()
    context = __get_context()
    # Now really set the value for our daemon type
    context['local_instances'][daemon_type][str(daemon_id)] = enabled
    
    __write_context(context)


def remove_local_daemon_instance(daemon_type, daemon_id):
    __assert_context()
    context = __get_context()
    # Now really set the value for our daemon type
    try:
        del context['local_instances'][daemon_type][str(daemon_id)]
    except Exception:
        print("WARNING: the instance %d was already removed from the context file")
        return
    
    __write_context(context)


def check_root():
    if os.getuid() != 0:
        print("You must be logged as root to launch this tool")
        sys.exit(2)


def is_local_daemon_instance_exists(daemon_type, daemon_id):
    __assert_context()
    context = __get_context()
    return str(daemon_id) in context['local_instances'][daemon_type]


def get_local_daemon_configuration_file_path(daemon_type, daemon_id):
    ini_suffix = ''
    if daemon_id != '0':
        ini_suffix = '-%s' % daemon_id
    cfg_p = '/etc/shinken/daemons/%sd%s.ini' % (daemon_type, ini_suffix)
    # now exceptions...
    if daemon_type == 'arbiter':
        cfg_p = '/etc/shinken/shinken.cfg'
    if daemon_type == 'synchronizer':
        cfg_p = '/etc/shinken/synchronizer.cfg'
    
    return cfg_p


def parse_ini_file(ini_path):
    config = configparser.ConfigParser()
    
    # Beware: ini file do not like space before comments and such things
    data = StringIO('\n'.join(line.strip() for line in open(ini_path)))
    
    config.read_file(data)
    if config._sections == {}:
        return None
    
    r = {}
    try:
        for (key, value) in config.items('daemon'):
            r[key] = value
    except configparser.InterpolationMissingOptionError:
        return None
    return r


def get_instance_name(daemon_type, daemon_id):
    if daemon_type in ADDITIONAL_DAEMONS:
        return daemon_type
    
    # Grok the pid file path inside the configuration file
    cfg_p = get_local_daemon_configuration_file_path(daemon_type, daemon_id)
    # HOOK: for arbiter and synchronizer, it's somewhere else
    if daemon_type == 'arbiter':
        cfg_p = '/etc/shinken/daemons/arbiterd.ini'
    if daemon_type == 'synchronizer':
        cfg_p = '/etc/shinken/daemons/synchronizerd.ini'
    
    if cfg_p.endswith('.ini'):  # look at the name parameter
        if not os.path.exists(cfg_p):
            return 'unnamed-%s' % daemon_type
        r = parse_ini_file(cfg_p)
        if r is None or 'name' not in r:
            return 'unnamed-%s' % daemon_type
        return r['name']
    if cfg_p.endswith('.cfg'):  # cfg files are not managed currently, and arbiter+synchronizer are manually managed, so how did you get there??
        return daemon_type
    
    return '(unknown-type)'


def _actions_on_daemon(action, daemon_type, daemon_id=None):
    id_param = ""
    if daemon_id:
        id_param = " --id %s" % daemon_id
    cmd = "/etc/init.d/shinken-%s %s %s" % (daemon_type, id_param, action)
    p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
    out = p.stdout.read()
    err = p.stderr.read()
    if err:
        print(' \033[%dm ERROR (%s): %s\033[0m' % (ERROR_COLOR, daemon_type, err))


def stop_daemon(daemon_type, daemon_id=None):
    _actions_on_daemon('stop', daemon_type, daemon_id)


def start_daemon(daemon_type, daemon_id=None):
    _actions_on_daemon('start', daemon_type, daemon_id)


def add_update_version_to_context(update_version, installer_version):
    check_root()
    __assert_context()
    context = __get_context()
    # Now really set the value for our daemon type
    context['current_version'] = update_version
    context['current_installer_version'] = installer_version
    __add_installation_version(context, update_version, 'UPDATE', int(time.time()), installer_version)
    for data_type in POSSIBLE_DATA:
        __add_data_version(data_type, context, update_version, 'UPDATE', int(time.time()), installer_version=installer_version)
    __write_context(context)


def add_patch_to_context(installation_type, patch_types, patch_name):
    check_root()
    __assert_context()
    context = __get_context()
    update_version = context.get('current_version')
    
    list_of_patch_types = [patch_type.strip() for patch_type in patch_types.split(',')]
    
    for patch_type in list_of_patch_types:
        __add_patch_install(patch_type, context, update_version, installation_type, int(time.time()), patch_name=patch_name)
    
    __write_context(context)


# SHINKEN-RESTORE --configuration: we are importing a full new configuration, so we must also load former context associated with configuration files
# * Backup from >= 02.08.02-RC018 will overwrite : "daemons", "local_instances", "addons", "addons_history", "sanitize_info"
def do_restore_context_from_file(restore_context_from_file: str) -> None:
    if not os.path.exists(restore_context_from_file):
        print(f'ERROR: while merging {restore_context_from_file} into context.json: The context file {restore_context_from_file} does not exist.', file=sys.stderr)
        sys.exit(2)
    
    # Load the distant context file
    try:
        with open(restore_context_from_file, 'r') as f:
            restored_context = json.loads(f.read())
    except ValueError:
        print(f'ERROR: while merging {restore_context_from_file} into context.json: The file {restore_context_from_file} does not contain valid JSON', file=sys.stderr)
        traceback.print_exc(file=sys.stderr)
        sys.exit(2)
    
    __assert_context()
    context = __get_context()
    
    # EXHAUSTIVE LIST of keys to retrieve from the old context file:
    keys_to_restore: 'tuple[str, ...]' = (
        # Restore enabled daemons
        'daemons',
        # Restore local daemon instances states
        'local_instances',
        # Restore enabled addons
        'addons',
        'addons_history',
        # Restore sanitize flags
        'sanitize_info',
    )
    
    # By default, if the key is not in the restored context.json, do nothing.
    # Set here the keys to overwrite if missing.
    key_to_overwrite_if_missing: 'dict[str, Any]' = {
        # If there were no flags ( backup pre-02.08.02-RC018 ), remove ALL saved flags.
        'sanitize_info': {},
    }
    
    for key in keys_to_restore:
        if key not in context:
            # Do not use sys.exit(), "keys_to_restore" MUST be up-to-date.
            raise KeyError(f'ERROR: Unknown key "{key}" in context.json file.')
        
        if key in restored_context:
            context[key] = restored_context[key]
        elif key in key_to_overwrite_if_missing:
            context[key] = key_to_overwrite_if_missing[key]
    
    # Ensure the restore context is valid.
    __sanitize_context_data(context)
    
    # We update it, write it down
    __write_context(context)


# SHINKEN-RESTORE --configuration: we are importing a full new configuration, so we must also load its data_history->configuration
# in this function, we have a context file in the backup, so >= 2.5.0
# * Backup from >= 2.5.0 => data_history->configuration is used, just erase our own entry with it
def do_import_data_history_from_file(data_type, import_data_history_from_file, backup_path='', comment=''):
    if not os.path.exists(import_data_history_from_file):
        raise FileNotFoundError(f'The context file {import_data_history_from_file} does not exist.')
    
    if data_type not in POSSIBLE_DATA:
        raise ValueError(f'''The available data types are {', '.join(POSSIBLE_DATA)}''')
    
    # Load the distant context file
    with open(import_data_history_from_file, 'r') as f:
        imported_context = json.loads(f.read())
    
    __assert_context()
    context = __get_context()
    
    # Erase our own, data are FULL new, so drop the old values
    context['data_history'][data_type] = imported_context['data_history'].get(data_type, [])
    
    # Add an entry, so we did RESTORE it in our version
    __add_data_version(data_type, context, context['current_version'], 'RESTORE', int(time.time()), backup_path=backup_path, comment=comment)
    
    # We did update, write it down
    __write_context(context)


# SHINKEN-RESTORE --configuration: we are importing a full new configuration, so we must also load its data_history->configuration
# in this function, we DO NOT have a context file in the backup, so < 2.5.0, but we have a version in the backup (so > 2.03.03-U01-RC240)
# we must fake our data with this version
def do_import_data_history_from_version(import_data_history_from_version, backup_path=''):
    __assert_context()
    context = __get_context()
    
    # Erase our own, data are FULL new, so drop the old values
    for data_type in POSSIBLE_DATA:
        context['data_history'][data_type] = []
        # Add an entry from the version we have
        __add_data_version(data_type, context, import_data_history_from_version, 'INSTALLATION', int(_context['creation_time']))
        # and the current RESTORE action
        __add_data_version(data_type, context, context['current_version'], 'RESTORE', int(time.time()), backup_path=backup_path)
    
    # We did update, write it down
    __write_context(context)


# SHINKEN-RESTORE --configuration: we are importing a full new configuration, so we must also load its data_history->configuration
# ok this time we are bad, we do not even have a backup version AT ALL (2.03.03-U01). Do not try to guess, sorry
def do_import_data_history_no_version(backup_path=''):
    __assert_context()
    context = __get_context()
    # Erase our own, data are FULL new, so drop the old values
    for data_type in POSSIBLE_DATA:
        context['data_history'][data_type] = []
        # just add the current RESTORE action
        __add_data_version(data_type, context, context['current_version'], 'RESTORE', int(time.time()), backup_path=backup_path)
    # We did update, write it down
    __write_context(context)


# SHINKEN-RESTORE & sanitize-data  we save the executed sanitize pass
def do_import_data_history_sanitize(sanitize_name, data_type='configuration'):
    __assert_context()
    context = __get_context()
    
    # add the entry with sanitize name
    __add_data_version(data_type, context, context['current_version'], 'SANITIZE', int(time.time()), sanitize_name=sanitize_name)
    
    # We did update, write it down
    __write_context(context)


# See the default flag values in set_default_sanitize_info_flags()
def check_sanitize_info_flag_in_context(sanitize_flag: str, expected_value: int) -> bool:
    __assert_context()
    context = __get_context()
    
    sanitize_info: 'dict[str, int]' = context.get('sanitize_info', {})
    if sanitize_info:
        return sanitize_info.get(sanitize_flag, 0) == expected_value
    return False


# sanitize-data : we save sanitize flag
# Remember to set the default flag value in set_default_sanitize_info_flags()
def update_sanitize_info_flag_in_context(sanitize_flag: str, value: int) -> None:
    __assert_context()
    context = __get_context()
    
    # Check if the given flag is valid.
    sanitize_flag = SANITIZE_INFO_CONTEXT_FLAG(sanitize_flag)
    
    # Set the flag in sanitize info.
    sanitize_info: 'dict[str, int]' = context.setdefault('sanitize_info', {})
    sanitize_info[sanitize_flag] = value
    
    __write_context(context)


def merge_from_key(key, file):
    __assert_context()
    context = __get_context()
    if key not in context:
        print("ERROR: while merging %s into context.json, the key %s does not exist in context.json" % (file, key))
        sys.exit(2)
    
    if not os.path.exists(file):
        print("ERROR: while merging %s into context.json, the file %s does not exist in context.json" % (file, file))
        sys.exit(2)
    try:
        with open(file) as f:
            data_to_merge = json.load(f)
        context[key].update(data_to_merge)
        __write_context(context)
    except ValueError:
        print("ERROR: while merging %s into context.json, the file %s does not contain valid JSON" % (file, file))
        sys.exit(2)


def disable_addons():
    __assert_context()
    context = __get_context()
    for addon in context['addons']:
        context['addons'][addon] = False
    __write_context(context)


def nagvis_update_uuid(old_uuid, new_uuid):
    __assert_context()
    
    if old_uuid == new_uuid:
        return
    
    context = __get_context()
    try:
        basedir = "/etc/shinken/external/nagvis/etc/conf.d/"
        os.unlink("%s/rotation_shinken_%s.ini.php" % (basedir, old_uuid))
        for file in os.listdir(basedir):
            if old_uuid in file:
                os.unlink("%s/%s" % (basedir, file))
        os.unlink("%s/shinken_architecture-%s.cfg" % (basedir, old_uuid))
        os.unlink("%s/shinken_global-%s.cfg" % (basedir, old_uuid))
    except OSError:
        pass
    
    try:
        with open("/var/lib/shinken/architecture_export_received.json") as fp:
            exports = json.load(fp)
    except:
        exports = {}
    if old_uuid in exports:
        with open("/var/lib/shinken/architecture_export_received.json", "w") as fp:
            del exports[old_uuid]
            json.dump(exports, fp)
    
    for addon in ['nagvis', 'nagvis-shinken-architecture']:
        __add_addons_version(addon, context, 'UPDATE', int(time.time()), old_uuid, new_uuid)
    __write_context(context)
