#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import PART_INITIALISATION, LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.lib_modules.configuration_reader_mixin import SeparatorFormat, ConfigurationFormat, TypeConfiguration, ConfigurationReaderMixin, HIDDEN_LOG_CONTENT

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from shinken.misc.type_hint import Optional, Iterable


class MongoConf(ConfigurationReaderMixin):
    def __init__(self, _conf: 'ShinkenModuleDefinition', logger: 'PartLogger' = None, default_database: str = 'shinken', prefixes_module_property: 'Iterable[str]|None' = None) -> None:
        self.name_database = None  # type: Optional[str]
        self.uri = None  # type: Optional[str]
        self.replica_set = None  # type: Optional[str]
        self.ssh_keyfile = None  # type: Optional[str]
        self.ssh_user = None  # type: Optional[str]
        self.mongodb_username = None  # type: Optional[str]
        self.mongodb_password = None  # type: Optional[str]
        self.use_ssh_retry_failure = None  # type: Optional[bool]
        self.ssh_tunnel_timeout = None  # type: Optional[int]
        self.use_ssh_tunnel = None  # type: Optional[bool]
        self.auto_reconnect_max_try = None  # type: Optional[int]
        self.auto_reconnect_sleep_between_try = None  # type: Optional[int]
        if prefixes_module_property is None:
            prefixes_module_property = []
        
        if logger:
            self.logger = logger  # type: PartLogger
        else:
            self.logger = LoggerFactory.get_logger()  # type: PartLogger
        
        self.logger_init = self.logger.get_sub_part(PART_INITIALISATION).get_sub_part('MONGO')  # type: PartLogger
        
        configuration_format = self.get_configuration_format(prefixes_module_property, default_database)
        ConfigurationReaderMixin.__init__(self, configuration_format, _conf, self.logger_init)
        self.read_configuration()
    
    
    def get_configuration_format(self, prefixes_module_property: 'list[Optional[str]]', default_database='shinken'):
        
        configuration_format = [
            SeparatorFormat('Connection parameters'),
            ConfigurationFormat(self.get_configuration_keys(['database', 'name_database', 'mongodb_database'], ['database__name'], prefixes_module_property), default_database, TypeConfiguration.STRING, 'name_database'),
            ConfigurationFormat(self.get_configuration_keys(['uri', 'mongodb_uri'], ['database__uri'], prefixes_module_property), 'mongodb://localhost/?w=1&fsync=false', TypeConfiguration.STRING, 'uri'),
            ConfigurationFormat(self.get_configuration_keys(['replica_set', 'mongodb_replica_set'], ['database__replica_set'], prefixes_module_property), '', TypeConfiguration.STRING, 'replica_set'),
            ConfigurationFormat(self.get_configuration_keys(['use_ssh_tunnel', 'mongodb_use_ssh_tunnel'], ['use_ssh_tunnel', 'database__use_ssh_tunnel'], prefixes_module_property), 0, TypeConfiguration.INT, 'use_ssh_tunnel'),
            ConfigurationFormat(self.get_configuration_keys(['use_ssh_retry_failure', 'mongodb_use_ssh_retry_failure'], ['use_ssh_retry_failure', 'database__use_ssh_retry_failure'], prefixes_module_property), 1, TypeConfiguration.INT,
                                'use_ssh_retry_failure'),
            ConfigurationFormat(self.get_configuration_keys(['ssh_user', 'mongodb_ssh_user'], ['database__ssh_user'], prefixes_module_property), 'shinken', TypeConfiguration.STRING, 'ssh_user'),
            ConfigurationFormat(self.get_configuration_keys(['mongodb_username', 'database__username'], ['database__username'], prefixes_module_property), '', TypeConfiguration.STRING, 'mongodb_username'),
            ConfigurationFormat(self.get_configuration_keys(['mongodb_password', 'database__password'], ['database__password'], prefixes_module_property), '', TypeConfiguration.HIDDEN_STRING, 'mongodb_password'),
            ConfigurationFormat(self.get_configuration_keys(['ssh_keyfile', 'mongodb_ssh_keyfile'], ['ssh_keyfile', 'database__ssh_keyfile'], prefixes_module_property), '~shinken/.ssh/id_rsa', TypeConfiguration.STRING, 'ssh_keyfile'),
            ConfigurationFormat(self.get_configuration_keys(['ssh_tunnel_timeout', 'mongodb_ssh_tunnel_timeout', 'mongodb_retry_timeout'], ['ssh_tunnel_timeout', 'database__ssh_tunnel_timeout'], prefixes_module_property), 10,
                                TypeConfiguration.INT, 'ssh_tunnel_timeout'),
            
            SeparatorFormat('AutoReconnect Management'),
            ConfigurationFormat(self.get_configuration_keys(['auto_reconnect_max_try', 'mongodb_auto_reconnect_max_try'], ['database__retry_connection_X_times_before_considering_an_error'], prefixes_module_property), 5, TypeConfiguration.INT,
                                'auto_reconnect_max_try'),
            ConfigurationFormat(self.get_configuration_keys(['auto_reconnect_sleep_between_try', 'mongodb_auto_reconnect_sleep_between_try'], ['database__wait_X_seconds_before_reconnect'], prefixes_module_property), 5, TypeConfiguration.INT,
                                'auto_reconnect_sleep_between_try'),
        
        ]
        return configuration_format
    
    
    @staticmethod
    def get_configuration_keys(keys_without_prefix: 'list[Optional[str]]', keys_with_prefix: list[str], prefixes_module_property: 'list[Optional[str]]') -> list[str]:
        for prefix_module_property in prefixes_module_property:
            for key_with_prefix in keys_with_prefix:
                
                keys_without_prefix.append(f'{prefix_module_property}__{key_with_prefix}')
        return keys_without_prefix