#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import errno
import os
import shutil
import socket
import struct
import subprocess
from stat import S_ISREG, S_IRUSR, S_IWUSR, S_IXUSR, S_IRGRP, S_IWGRP, S_IXGRP, S_IROTH, S_IWOTH, S_IXOTH

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.os_helper import get_user_id_from_name, get_cur_user_id, get_cur_group_id, get_group_id_from_name, set_ownership

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List, Union
    from shinken.log import PartLogger

try:
    import fcntl
except ImportError:
    # noinspection SpellCheckingInspection
    fcntl = None


class FILE_RIGHT_MODE:
    READ = 'read'
    WRITE = 'write'
    EXECUTE = 'execute'


class FILE_RIGHT_TARGET:
    USER = 'user'
    GROUP = 'group'
    OTHER = 'other'


class PATH_TYPE:
    DIRECTORY = 'directory'
    FILE = 'file'


# This dict use the unix mask to compute the rights
unix_rights = {
    FILE_RIGHT_TARGET.USER : {
        FILE_RIGHT_MODE.READ   : S_IRUSR,
        FILE_RIGHT_MODE.WRITE  : S_IWUSR,
        FILE_RIGHT_MODE.EXECUTE: S_IXUSR
    },
    FILE_RIGHT_TARGET.GROUP: {
        FILE_RIGHT_MODE.READ   : S_IRGRP,
        FILE_RIGHT_MODE.WRITE  : S_IWGRP,
        FILE_RIGHT_MODE.EXECUTE: S_IXGRP
        
    },
    FILE_RIGHT_TARGET.OTHER: {
        FILE_RIGHT_MODE.READ   : S_IROTH,
        FILE_RIGHT_MODE.WRITE  : S_IWOTH,
        FILE_RIGHT_MODE.EXECUTE: S_IXOTH
    }
}


def check_right_on_path(path, mode=FILE_RIGHT_MODE.READ, path_type=PATH_TYPE.DIRECTORY, user=None, group=None):
    # type: (str, str, str, Optional[str], Optional[str]) -> bool
    # Use this function to check the rights on folder and file.
    # path: the path to check
    # mode: which mode you want for your path : 'read', 'write', 'execute'
    # path_type: is your path is a directory or a file
    # user: on which user check right ? if None, it will use the current process’s effective user id
    # group: on which group check right ? if None, it will use the current process’s effective group id
    # return: True if rights are OK. Raise if you can access to your file
    
    path = os.path.join(path)
    # First check if the path exists
    if not os.path.exists(path):
        raise OSError(errno.ENOENT, '''The path doesn't exists''', path)
    
    # Is the target is the good one ?
    if (path_type == PATH_TYPE.DIRECTORY and not os.path.isdir(path)) or \
            (path_type == PATH_TYPE.FILE and not os.path.isfile(path)):
        raise OSError(errno.ENOENT, 'This is not a %s  : ' % path_type, path)
    
    # Will check if we use the user, the group or other
    user_uid = get_user_id_from_name(user) if user else get_cur_user_id()
    
    if user_uid == 0:
        # You're root ? ... ok you can go
        return True
    
    group_gid = get_group_id_from_name(group) if group else get_cur_group_id()
    file_stat = os.stat(path)
    
    if user_uid == file_stat.st_uid:
        key_used = FILE_RIGHT_TARGET.USER
    elif group_gid == file_stat.st_gid:
        key_used = FILE_RIGHT_TARGET.GROUP
    else:
        key_used = FILE_RIGHT_TARGET.OTHER
    
    # Now will check the rights with the good mask
    if bool(file_stat.st_mode & unix_rights[key_used][mode]):
        return True
    else:
        raise OSError(errno.EACCES, 'User %s can not access to the %s with the %s mode ' % (user, path_type, mode), path)


def check_right_on_file(path, mode=FILE_RIGHT_MODE.READ, user=None, group=None):
    # type: (str, str, Optional[str], Optional[str]) -> bool
    # Use this function to check the rights on file.
    # path: the path to check
    # mode: which mode you want for your path : 'read', 'write', 'execute'
    # user: on which user check right ? if None, it will use the current process’s effective user id
    # group: on which group check right ? if None, it will use the current process’s effective group id
    # return: True if rights are OK. Raise if you can access to your file
    return check_right_on_path(path, mode, PATH_TYPE.FILE, user, group)


def check_right_on_directory(path, mode=FILE_RIGHT_MODE.READ, user=None, group=None):
    # type: (str, str, Optional[str], Optional[str]) -> bool
    # Use this function to check the rights on folder.
    # path: the path to check
    # mode: which mode you want for your path : 'read', 'write', 'execute'
    # user: on which user check right ? if None, it will use the current process’s effective user id
    # group: on which group check right ? if None, it will use the current process’s effective group id
    # return: True if rights are OK. Raise if you can access to your folder
    return check_right_on_path(path, mode, PATH_TYPE.DIRECTORY, user, group)


def create_tree(path, user='shinken', group='shinken', mode=0o755):
    # type: (str, str, str, int) -> None
    # Warning! the mode will be overwritten by os.umask if set
    drive, path = os.path.splitdrive(path)
    full_path = [p for p in path.split(os.sep) if p]
    # to manage relative path or not
    parents = ('%s%s' % (drive, os.sep)) if os.path.isabs(path) else ''
    for folder_name in full_path:
        current_path = os.path.join(parents, folder_name)
        
        if not os.path.exists(current_path):
            try:
                check_right_on_directory(parents, FILE_RIGHT_MODE.WRITE)
            except OSError:
                raise OSError(errno.EACCES, 'Cannot create the full tree. Permission denied ', parents)
            os.mkdir(current_path, mode)
            set_ownership(current_path, user, group)
        
        try:
            check_right_on_directory(current_path, FILE_RIGHT_MODE.READ, user, group)
            check_right_on_directory(current_path, FILE_RIGHT_MODE.EXECUTE, user, group)
        except OSError:
            raise OSError(errno.EACCES, 'The path already exists but cannot jump in. Permission denied ', current_path)
        
        # Now, next level
        parents = current_path


def run_command(command):
    # type: (str) -> str
    _, stdout, stderr = run_command_with_return_code(command)
    return '%s%s' % (stdout, stderr)


def run_command_with_return_code(command: 'Union[List[str], str]', shell: bool = True, env: 'dict[str, str]|None' = None) -> tuple[int, str, str]:
    if env is None:
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell, encoding='utf-8')
    else:
        _process = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=shell, env=env, encoding='utf-8')
    stdout, stderr = _process.communicate()
    return _process.returncode, stdout, stderr


def is_regular_file(path):
    # type: (str) -> bool
    if os.path.exists(path):
        mode = os.stat(path).st_mode
        return S_ISREG(mode)
    return False


def get_linux_local_addresses(_logger):
    # type: (PartLogger) -> List[str]
    
    try:
        rc, stdout, stderr = run_command_with_return_code('hostname -I')
    except Exception as exp:
        _logger.info('Cannot use the hostname -I call for linux (%s), trying to guess local addresses' % exp)
        stdout = ''
    buf = stdout.strip()
    res = [s.strip() for s in buf.split(' ') if s.strip()]
    
    # Some system like in alpine linux that don't have hostname -I call so try to guess
    if len(res) == 0:
        _logger.debug('Cannot use the hostname -I call for linux, trying to guess local addresses')
        for interface_prefix in ('bond', 'eth', 'venet', 'wlan'):
            for interface_counter in range(0, 10):
                interface_name = '%s%d' % (interface_prefix, interface_counter)
                try:
                    addr = _get_ip_address(interface_name)
                    res.append(addr)
                except IOError:  # no such interface
                    pass
    res = sorted(res, key=_key_local_address)
    return res


def _get_ip_address(interface_name):
    # type: (str) -> str
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    # noinspection SpellCheckingInspection
    return socket.inet_ntoa(fcntl.ioctl(
        s.fileno(),
        0x8915,  # SIOCGIFADDR
        struct.pack('256s', interface_name[:15])
    )[20:24])


def _key_local_address(address):
    addr_is_192 = address.startswith('192.')
    addr_is_10 = address.startswith('10.')
    addr_is_172 = address.startswith('172.')
    addr_is_127 = address.startswith('127.')
    addr_order = 4
    if addr_is_192:
        addr_order = 1
    elif addr_is_172:
        addr_order = 2
    elif addr_is_10:
        addr_order = 3
    if addr_is_127:
        addr_order = 5
    return addr_order


def remove_file(file_name: 'str', raise_error: 'bool' = False, logger: 'PartLogger|None' = None) -> bool:
    if not os.path.exists(file_name):
        if logger and logger.is_debug():
            logger.debug(f'No file {file_name} to remove')
        if raise_error:
            raise FileNotFoundError(file_name)
        return False
    try:
        os.unlink(file_name)
        if logger and logger.is_debug():
            logger.debug(f'Removed {file_name}')
        return True
    except Exception as e:
        if logger:
            logger.warning(f'Could not remove file {file_name} with error {str(e)}')
        if raise_error:
            raise e
    return False


def move_file(source_file: 'str', destination_file: 'str', overwrite_more_recent: 'bool' = False, raise_error: 'bool' = False, logger: 'PartLogger|None' = None, user: 'str|None' = None, group: 'str|None' = None) -> 'bool':
    if os.path.exists(source_file):
        if os.path.exists(destination_file):
            if overwrite_more_recent and os.stat(destination_file).st_mtime < os.stat(source_file).st_mtime:
                remove_file(destination_file, raise_error=raise_error, logger=logger)
            else:
                if logger and logger.is_debug():
                    logger.debug(f'{destination_file} already exists and is more recent than {source_file}, keeping it')
                remove_file(source_file, raise_error=raise_error, logger=logger)
                return True
        
        copy_done = True
        try:
            # First try, just create a new name for the file (must be on same filesystem)
            os.link(source_file, destination_file)
        except:
            try:
                # Second try, copy whole data to new file (will work across filesystems)
                shutil.copy2(source_file, destination_file)
                if logger and logger.is_debug():
                    logger.debug(f'shutil.copy2 {source_file} -> {destination_file}')
                set_ownership_params = {}
                if user:
                    set_ownership_params.update({'user': user})
                if group:
                    set_ownership_params.update({'group': group})
                if set_ownership_params:
                    try:
                        set_ownership(destination_file, **set_ownership_params)
                    except Exception as e:
                        if logger:
                            logger.warning(f'Failed to change ownership of {destination_file} to {set_ownership_params} with error {str(e)}')
                        if raise_error:
                            raise e
            except Exception as e:
                if logger:
                    logger.warning(f'Could not copy {source_file} to {destination_file} with error {str(e)}')
                if raise_error:
                    raise e
                copy_done = False
        if copy_done:
            if logger and logger.is_debug():
                logger.debug(f'Successfully copied file {source_file} to {destination_file}')
            remove_file(source_file, raise_error=raise_error, logger=logger)
        return copy_done
    elif logger:
        logger.debug(f'Cannot move {source_file} as it does not exists')
        if raise_error:
            raise FileNotFoundError(source_file)
        return False
