#!/usr/bin/env python3-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import optparse
import os.path
import re
import sys
import warnings

from shinkensolutions.localinstall import get_local_instances_for_type, VERSION

warnings.filterwarnings("ignore", category=DeprecationWarning)
ALL_FIX = []
NUMBER_INDENT_SPACES = 0
DEFAULT_LOG_FILENAME = os.path.abspath(os.path.join(os.sep, 'shinken', 'var', 'versions_installations', 'sanitize.log'))


def add_doc(doc_):
    def decorated(func):
        func.doc = doc_
        return func
    
    
    return decorated


def add_context_daemons(lst):
    # protect again dev that don't read the doc...
    if isinstance(lst, str):
        lst = [lst]
    
    
    def decorated(func):
        func.context_daemons = lst
        return func
    
    
    return decorated


def add_version(version):
    def decorated(func):
        func.version = version
        return func
    
    
    return decorated


def auto_launch(value):
    def decorated(func):
        func.auto_launch = value
        return func
    
    
    return decorated


def need_shinken_stop(value):
    def decorated(func):
        func.need_shinken_stop = value
        return func
    
    
    return decorated


def add_fix(func):
    global NUMBER_INDENT_SPACES
    ALL_FIX.append(func)
    name_length = len(func.__name__)
    if name_length > NUMBER_INDENT_SPACES:
        NUMBER_INDENT_SPACES = name_length
    return func


def natural_key(string_):
    _natural_key = []
    for s in re.split(r'(\d+)', string_):
        if s.isdigit():
            _natural_key.append(int(s))
        else:
            _natural_key.append(s.lower())
    return _natural_key


def natural_version(f):
    return natural_key(f.version)


def do_match_daemons(f):
    needed_daemons = f.context_daemons
    if not needed_daemons:
        return True
    
    for daemon_type in needed_daemons:
        local_instances = [d for d in get_local_instances_for_type(daemon_type) if d[1] is True]
        if local_instances:
            return True
    return False


@add_fix
@add_doc('This sanitize will cleanup previously recorded configuration of daemons')
@add_context_daemons(['broker', 'poller', 'reactionner', 'receiver', 'scheduler'])
@add_version('02.08.02')
@auto_launch(True)
@need_shinken_stop(True)
def last_configuration_recorded_remove():
    import last_configuration_recorder.module as last_conf_rec_py
    
    g_did_run = False
    
    sanitize_on_update = last_conf_rec_py.LastConfigurationRecorder.sanitize_on_update
    
    daemon_type = 'poller'
    daemon_id = 0
    
    daemon_logger = PartLogger(f'{daemon_type}d-{daemon_id}', register=False)
    try:
        if sanitize_on_update(daemon_type, daemon_id, daemon_logger):
            g_did_run = True
    except Exception as err:
        daemon_logger.error(f'Configuration / retention removal raised error: {str(err)}')
        daemon_logger.print_stack('')
        g_did_run = 'Could not cleanup some retention / configuration files'
    
    return g_did_run


if __name__ == '__main__':
    
    os.system('color')
    shinken_path = os.path.dirname(os.path.dirname(os.path.relpath(__file__)))
    modules_path = os.path.join(shinken_path, 'modules')
    
    # allow import of shinken.*
    if shinken_path not in sys.path:
        sys.path.insert(1, shinken_path)
    
    # allow import of modules
    if modules_path not in sys.path:
        sys.path.insert(1, modules_path)
    
    import shinken.log
    from shinken.log import logger, PartLogger
    from shinkensolutions.system_tools import create_tree
    
    parser = optparse.OptionParser("%prog ", version="%prog: " + VERSION, description='This tool is used to check the state of your Shinken Enterprise installation and configuration')
    parser.add_option('-l', '--list', dest='list_only', action='store_true', help="List available fixes")
    parser.add_option('-L', '--log_file', dest='log_file', default=DEFAULT_LOG_FILENAME, help="The log file path")
    opts, args = parser.parse_args()
    
    # monkey patch : Remove ugly colors
    shinken.log.cprint = lambda text, *arguments: print(text)
    
    # create log file
    log_file = opts.log_file
    log_dir = os.path.dirname(log_file)
    create_tree(log_dir)
    logger.register_local_log(log_file)
    logger.set_name('Sanitize', False)
    
    message_template = "   \033[%%dm%%-%ds\033[0m:  \033[%%dm %%s \033[0m" % (NUMBER_INDENT_SPACES + 2)
    ALL_FIX = sorted(ALL_FIX, key=natural_version)
    
    if opts.list_only:
        cur_version = ''
        for sanitize_function in ALL_FIX:
            if not hasattr(sanitize_function, 'auto_launch'):
                raise Exception('Please specify auto launch parameter on fix [ %s ].' % sanitize_function.__name__)
            v = sanitize_function.version
            if v != cur_version:
                cur_version = v
                logger.info("################ Version \033[35m%s\033[0m ##########" % v)
            logger.info("-------------------")
            logger.info(" \033[32m%-40s\033[0m:" % sanitize_function.__name__)
            logger.info("\tDoc: %s" % sanitize_function.doc)
        sys.exit(0)
    
    exit_code = 0
    
    for sanitize_function in ALL_FIX:
        sanitize_function_name = sanitize_function.__name__
        logger.set_name(sanitize_function_name, False)
        skipped = False
        if do_match_daemons(sanitize_function):
            logger.info('Execution is starting')
            try:
                ret = sanitize_function()
            except Exception as e:
                ret = str(e)
                logger.print_stack('')
            
            if isinstance(ret, str):
                has_error = True
                logger.error('\n\033[31m ERROR: %s => %s\033[0m\n' % (sanitize_function_name, ret))
                exit_code = 1
            elif isinstance(ret, bool) and not ret:
                has_error = False
                skipped = True
            else:
                has_error = False
            
            if has_error:
                logger.error(message_template % (35, sanitize_function_name, 31, "executed [Failure]"))
            elif skipped:
                logger.info(message_template % (35, sanitize_function_name, 36, "skip (unnecessary)"))
            else:
                logger.info(message_template % (35, sanitize_function_name, 32, "executed [OK]"))
        else:
            if len(sanitize_function.context_daemons) > 1:
                state = 'skip (this sanitize is only needed for these daemons : %s)' % ', '.join(sanitize_function.context_daemons)
            else:
                state = 'skip (this sanitize is only needed for this daemon : %s)' % ', '.join(sanitize_function.context_daemons)
            logger.info(message_template % (35, sanitize_function_name, 36, state))
    
    sys.exit(exit_code)
