#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import io
import os.path
import pickle
import shutil
import sys
import threading
import time

from shinken.log import PART_INITIALISATION
from shinken.misc.os_utils import safe_write_binary_file_and_force_mtime
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.autoloaded_module import AutoLoadedModule
from shinken.modules.base_module.basemodule import ModuleState
from shinken.safepickle import SafeUnpickler
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, ConfigurationFormat, TypeConfiguration
from shinkensolutions.system_tools import create_tree, move_file, remove_file

if TYPE_CHECKING:
    from shinken.configuration_incarnation import ConfigurationIncarnation
    from shinken.daemon import Daemon
    from shinken.daemons.receiverdaemon import Receiver
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Optional
    from shinken.objects.inventory import ElementsInventory
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from shinken.satellite import BaseSatellite
    from shinken.withinventorysatellite import WithInventorySatellite

app: 'Optional[Daemon]' = None
properties: dict

if sys.platform.startswith('win'):
    MODULE_CONFIGURATION_DIR = 'c:\\shinken\\etc\\daemons'
    DEFAULT_RETENTION_DIR = 'c:\\shinken\\var\\last_configuration_recorder\\daemons'
else:
    MODULE_CONFIGURATION_DIR = '/etc/shinken/daemons'
    DEFAULT_RETENTION_DIR = '/var/lib/shinken/persistent_data/last_configuration_recorder/daemons'


def get_autoload_class() -> 'type[LastConfigurationRecorder]':
    return LastConfigurationRecorder


def get_instance(module_configuration: 'ShinkenModuleDefinition') -> 'LastConfigurationRecorder':
    return LastConfigurationRecorder(module_configuration)


class LastConfigurationRecorder(AutoLoadedModule, ConfigurationReaderMixin):
    # noinspection SpellCheckingInspection
    satellite_types = ['arbiters', 'pollers', 'reactionners', 'receivers', 'schedulers']
    # noinspection SpellCheckingInspection
    satellite_unpickable_parameters = ('wait_homerun_lock', 'con')
    
    
    @staticmethod
    def _get_configuration_dir(daemon_type: 'int|str') -> 'str':
        return os.path.join(MODULE_CONFIGURATION_DIR, 'last_configuration_recorder', f'{daemon_type}')
    
    
    @classmethod
    def _get_configuration_file(cls, daemon_type: 'str', daemon_id: 'int|str') -> 'str':
        return os.path.join(cls._get_configuration_dir(daemon_type), f'{daemon_id}')
    
    
    @classmethod
    def _get_configuration_file_invalid(cls, daemon_type: 'str', daemon_id: 'int|str') -> 'str':
        return f'{cls._get_configuration_file(daemon_type, daemon_id)}.invalid'
    
    
    @classmethod
    def get_autoload_configuration(cls, daemon_type: 'str', daemon_id: 'int|str', daemon_name: 'str', module_type: 'str', logger: 'PartLogger' = None) -> 'ShinkenModuleDefinition|None':
        module_configuration_file = cls._get_configuration_file(daemon_type, daemon_id)
        if not os.path.exists(module_configuration_file) or os.path.getsize(module_configuration_file) == 0:
            return super(LastConfigurationRecorder, cls).get_autoload_configuration(daemon_type, daemon_id, daemon_name, module_type, logger=None)
        try:
            if logger and logger.is_debug():
                logger.debug(f'Reading previous saved module configuration from file {module_configuration_file}')
            with open(module_configuration_file, 'rb') as default_configuration_fd:
                default_configuration = SafeUnpickler.load(io.BytesIO(default_configuration_fd.read()), f'{daemon_type}-{daemon_id}')
        except Exception as e:
            error_message = f'Failed to read module configuration from file {module_configuration_file} with error : {str(e)}'
            raise Exception(error_message) from e
        return default_configuration
    
    
    @staticmethod
    def _get_configuration_parameters_name(daemon_type: 'str') -> dict[str, str]:
        return {
            'daemon_id'    : f'{daemon_type}__module_last_configuration_recorder__daemon_id',
            'daemon_type'  : f'{daemon_type}__module_last_configuration_recorder__daemon_type',
            'retention_dir': f'{daemon_type}__module_last_configuration_recorder__directory',
            'enabled'      : f'{daemon_type}__module_last_configuration_recorder__enable'
        }
    
    
    @staticmethod
    def _get_configuration_format(parameters_name: 'dict[str,str]') -> 'list':
        return [
            ConfigurationFormat([parameters_name['daemon_id']], 0, TypeConfiguration.INT, 'daemon_id'),
            ConfigurationFormat([parameters_name['daemon_type']], '', TypeConfiguration.STRING, 'daemon_type'),
            ConfigurationFormat([parameters_name['retention_dir']], DEFAULT_RETENTION_DIR, TypeConfiguration.STRING, 'configuration_retention_dir'),
            ConfigurationFormat([parameters_name['enabled']], True, TypeConfiguration.BOOL, 'enabled'),
        ]
    
    
    @classmethod
    def sanitize_on_update(cls, daemon_type: 'str', daemon_id: 'int', logger: 'PartLogger|None' = None) -> 'bool':
        module_configuration_dir = cls._get_configuration_dir(daemon_type)
        module_configuration_file = cls._get_configuration_file(daemon_type, daemon_id)
        module_invalid_configuration_file = cls._get_configuration_file_invalid(daemon_type, daemon_id)
        configuration_data = cls.get_autoload_configuration(daemon_type, daemon_id, 'sanitize', 'in_sanitize')
        configuration_format = cls._get_configuration_format(cls._get_configuration_parameters_name(daemon_type))
        configuration_reader = ConfigurationReaderMixin(configuration_format, configuration_data)
        configuration_reader.read_configuration()
        retention_dir = cls._get_retention_dir_name(getattr(configuration_reader, 'configuration_retention_dir'), daemon_type, daemon_id)
        if not os.path.exists(retention_dir) and not os.path.exists(module_configuration_file) and not os.path.exists(module_invalid_configuration_file):
            return False
        if os.path.exists(retention_dir):
            if logger:
                logger.info(f'Removing retention directory {retention_dir}')
            cls._remove_retention_dir(retention_dir)
        if os.path.exists(module_invalid_configuration_file):
            if logger:
                logger.info(f'Removing invalid configuration file {module_invalid_configuration_file}')
            try:
                os.unlink(module_invalid_configuration_file)
            except:
                pass
        if os.path.exists(module_configuration_file):
            if logger:
                logger.info(f'Removing configuration file {module_configuration_file}')
            os.unlink(module_configuration_file)
        cls._remove_module_configuration_dir(module_configuration_dir)
        
        return True
    
    
    def __init__(self, module_configuration: 'ShinkenModuleDefinition'):
        self.configuration_retention_dir: 'Optional[str]' = None
        self.daemon_id: 'Optional[int]' = None
        self.daemon_type: 'Optional[str]' = None
        self.enabled: bool = True
        self.errors = {}
        self.retention_dir: 'str|None' = None
        self.configuration_file: 'str|None' = None
        self.inventories_file: 'str|None' = None
        self.host_mapping_file: 'str|None' = None
        self._is_ready: 'bool' = False
        
        AutoLoadedModule.__init__(self, module_configuration)
        
        self.lock = threading.RLock()
        self.module_configuration_dir = self._get_configuration_dir(self.properties['daemon_type'])
        self.module_configuration_file = self._get_configuration_file(self.properties['daemon_type'], f'''{self.properties['daemon_id']}''')
        self.module_invalid_configuration_file = self._get_configuration_file_invalid(self.properties['daemon_type'], f'''{self.properties['daemon_id']}''')
        self.logger_configuration = self.logger.get_sub_part('CONFIGURATION')
        self.logger_loader = self.logger_configuration.get_sub_part('LOADER')
        self.logger_recorder = self.logger_configuration.get_sub_part('RECORDER')
    
    
    def _save_module_configuration(self, module_configuration: 'ShinkenModuleDefinition'):
        if not os.path.exists(self.module_configuration_dir):
            try:
                create_tree(self.module_configuration_dir)
            except Exception as e:
                error_message = self.errors['default_create_dir'] = f'Failed to create module configuration directory {self.module_configuration_dir} with error {str(e)}'
                raise Exception(error_message) from e
        
        if 'default_create_dir' in self.errors:
            del self.errors['default_create_dir']
        
        self.logger_recorder.info(f'Saving module configuration to [ {self.module_configuration_file} ]')
        try:
            with open(self.module_configuration_file, 'wb') as default_conf_fd:
                pickle.dump(module_configuration, default_conf_fd)
        except Exception as e:
            error_message = self.errors['default_save'] = f'Failed to write module configuration to file {self.module_configuration_file} with error : {str(e)}'
            raise Exception(error_message) from e
        
        if 'default_save' in self.errors:
            del self.errors['default_save']
    
    
    @staticmethod
    def _remove_module_configuration_dir(module_configuration_dir):
        try:
            os.rmdir(module_configuration_dir)
            # Remove module directory if empty
            os.rmdir(os.path.dirname(module_configuration_dir))
        except:
            pass
    
    
    def _remove_module_configuration(self):
        if os.path.exists(self.module_invalid_configuration_file):
            try:
                os.unlink(self.module_invalid_configuration_file)
            except:
                pass
        
        if not os.path.exists(self.module_configuration_file):
            return
        self.logger_recorder.debug(f'Removing previous module configuration file [ {self.module_configuration_file} ] (configuration has been reset to default)')
        try:
            os.unlink(self.module_configuration_file)
        except Exception as e:
            error_message = self.errors['remove_conf_file'] = f'Failed to remove module configuration file {self.module_configuration_file} with error {str(e)}'
            raise Exception(error_message) from e
        
        if 'remove_conf_file' in self.errors:
            del self.errors['remove_conf_file']
        
        self._remove_module_configuration_dir(self.module_configuration_dir)
    
    
    def _move_all_retention_data_when_directory_has_changed(self, previous_retention_dir):
        if previous_retention_dir != self.configuration_retention_dir:
            old_retention_dir = self._get_retention_dir_name(previous_retention_dir, self.daemon_type, self.daemon_id)
            if os.path.exists(old_retention_dir):
                try:
                    self.logger_recorder.info(f'Moving previous retention data from directory [ {old_retention_dir} ] to [ {self.retention_dir} ]')
                    with self.lock:
                        self._move_retention_data(os.path.join(old_retention_dir, 'configuration'), self.configuration_file)
                        self._move_retention_data(os.path.join(old_retention_dir, 'inventories'), self.inventories_file)
                        self._move_retention_data(os.path.join(old_retention_dir, 'host-mapping'), self.host_mapping_file)
                    self._remove_retention_dir(old_retention_dir)
                except Exception as e:
                    self.logger_recorder.error(f'Failed to move previous retention data from directory [ {old_retention_dir} ] with error {str(e)}')
    
    
    def init(self):
        module_configuration = self.myconf
        daemon_type = self.properties['daemon_type']
        daemon_id = self.properties['daemon_id']
        
        try:
            previous_configuration = self.get_autoload_configuration(daemon_type, daemon_id, self.daemon_display_name, self.module_type, self.logger_loader)
        except Exception as conf_exception:
            if hasattr(module_configuration, 'auto_configured'):
                raise
            previous_module_invalid_configuration_file = self.module_invalid_configuration_file
            previous_module_configuration_file = self.module_configuration_file
            self.logger_configuration.warning(f'{str(conf_exception)}')
            self.logger_configuration.info(f'A new configuration has been received, backing up previous invalid configuration to {previous_module_invalid_configuration_file}')
            
            if os.path.exists(previous_module_invalid_configuration_file):
                try:
                    os.unlink(previous_module_invalid_configuration_file)
                except Exception as e:
                    self.logger_configuration.error(f'Failed to remove already existing invalid configuration backup file {previous_module_invalid_configuration_file} with error : {str(e)}')
            try:
                os.rename(previous_module_configuration_file, previous_module_invalid_configuration_file)
            except Exception as e:
                self.logger_configuration.error(f'Failed to rename previous configuration file {previous_module_configuration_file} to {previous_module_invalid_configuration_file} with error : {str(e)}')
                try:
                    os.unlink(previous_module_configuration_file)
                except Exception as e:
                    self.logger_configuration.error(f'Failed to remove previous configuration file {previous_module_configuration_file}  with error : {str(e)}')
            
            previous_configuration = self.get_autoload_auto_configuration('dummy', daemon_type)
        
        # Module's parameters are named based on daemon type. As this module can be set on several daemons, parameters' names must be computed at runtime
        parameters_name = self._get_configuration_parameters_name(daemon_type)
        daemon_id_key = parameters_name['daemon_id']
        daemon_type_key = parameters_name['daemon_type']
        retention_dir_key = parameters_name['retention_dir']
        # enabled_key = parameters_name['enabled']
        
        # Attach data provided by ModuleManager to configuration to:
        # - allow nice logging of configuration parameters
        # - overload any data coming from cfg file with real runtime values
        setattr(module_configuration, daemon_id_key, daemon_id)
        setattr(module_configuration, daemon_type_key, daemon_type)
        
        # Define default values based on previous recorded configuration, to allow detection of configuration changes
        previous_retention_dir = getattr(previous_configuration, retention_dir_key, DEFAULT_RETENTION_DIR)
        
        # As default values may change each time a new conf is received, __init__ of ConfigurationReaderMixin cannot be called in __init__ of this class
        configuration_format = self._get_configuration_format(parameters_name)
        
        ConfigurationReaderMixin.__init__(self, configuration_format, module_configuration, logger=self.logger.get_sub_part(PART_INITIALISATION))
        self.read_configuration()
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
        self.retention_dir = self._get_retention_dir_name(self.configuration_retention_dir, self.daemon_type, self.daemon_id)
        
        if hasattr(module_configuration, 'auto_configured'):
            self._remove_module_configuration()
        # Save own configuration if parameters have changed from Arbiter
        elif getattr(previous_configuration, 'hash', '0') != module_configuration.hash:
            self._save_module_configuration(module_configuration)
        
        self.configuration_file = os.path.join(self.retention_dir, 'configuration')
        self.inventories_file = os.path.join(self.retention_dir, 'inventories')
        self.host_mapping_file = os.path.join(self.retention_dir, 'host-mapping')
        self._create_retention_dir()
        
        # Try to keep previous data (inventories or host mapping) when retention directory changes
        self._move_all_retention_data_when_directory_has_changed(previous_retention_dir)
        self._is_ready = True
    
    
    def _move_retention_data(self, old_file, new_file):
        if os.path.exists(old_file):
            if not move_file(old_file, new_file, logger=self.logger_recorder):
                remove_file(old_file, logger=self.logger_recorder)
    
    
    def get_state(self, module_wanted=None):
        output = []
        if os.path.exists(self.module_invalid_configuration_file):
            output.append(f'{self.name}: invalid configuration file back up found in {self.module_invalid_configuration_file}')
        if self.errors:
            return {'status': ModuleState.CRITICAL, 'output': [f'{self.name}: {error_message}' for error_message in self.errors.values()] + output}
        if not self._is_ready:
            return {'status': ModuleState.CRITICAL, 'output': [f'{self.name}: module has not been initialised'] + output}
        if output:
            return {'status': ModuleState.WARNING, 'output': output}
        
        return {'status': ModuleState.OK, 'output': 'OK'}
    
    
    @staticmethod
    def _get_retention_dir_name(base_directory: 'str', daemon_type: 'str', daemon_id: 'int'):
        return os.path.join(base_directory, daemon_type, f'{daemon_id}')
    
    
    @staticmethod
    def _remove_retention_dir(retention_dir):
        shutil.rmtree(retention_dir, ignore_errors=True)
        try:
            daemon_dir = os.path.dirname(retention_dir)
            os.rmdir(daemon_dir)
            os.rmdir(os.path.dirname(daemon_dir))
        except (OSError, FileNotFoundError):
            pass
    
    
    def _create_retention_dir(self):
        if not self.enabled:
            if os.path.exists(self.retention_dir):
                self.logger.info(f'Module is disabled, removing retention directory [ {self.retention_dir} ]')
                try:
                    with self.lock:
                        self._remove_retention_dir(self.retention_dir)
                except Exception as e:
                    error_message = f'Failed to remove retention directory [ {self.retention_dir} ] with error {str(e)}'
                    # self.logger.error(error_message)
                    raise Exception(error_message) from e
            return
        
        with self.lock:
            if not os.path.exists(self.retention_dir):
                try:
                    create_tree(self.retention_dir)
                    if 'conf_create_dir' in self.errors:
                        del self.errors['conf_create_dir']
                except Exception as e:
                    error_message = self.errors['conf_create_dir'] = f'Failed to create daemon configuration directory {self.retention_dir} with error {str(e)}'
                    raise Exception(error_message) from e
            elif 'conf_create_dir' in self.errors:
                del self.errors['conf_create_dir']
    
    
    def retention_file_save(self, file_name: 'str', short_name: 'str', retention_data: 'dict', daemon_name: 'str'):
        error_tag = f'{short_name}_save'
        try:
            with self.lock:
                self._create_retention_dir()
                safe_write_binary_file_and_force_mtime(file_name, pickle.dumps(retention_data, pickle.HIGHEST_PROTOCOL))
        except Exception as e:
            error_msg = self.errors[error_tag] = f'''Could not save {short_name} to file '{file_name}' with error {str(e)}'''
            self.logger_recorder.error(error_msg)
            self.logger_recorder.print_stack()
            return
        
        self.logger_recorder.debug(f'Saved {short_name} to file [ {file_name} ] for daemon {daemon_name}')
        if error_tag in self.errors:
            del self.errors[error_tag]
    
    
    @staticmethod
    def _retention_file_remove(file_to_remove: 'str', logger: 'PartLogger'):
        if os.path.exists(file_to_remove):
            try:
                os.unlink(file_to_remove)
                logger.info(f'Successfully removed file {file_to_remove}')
            except Exception as e:
                logger.error(f'Failed to remove file {file_to_remove} with error {str(e)}')
    
    
    def retention_files_remove(self, logger: 'PartLogger'):
        with self.lock:
            self._retention_file_remove(self.configuration_file, logger)
            self._retention_file_remove(self.inventories_file, logger)
            self._retention_file_remove(self.host_mapping_file, logger)
            if os.path.exists(self.retention_dir):
                self._remove_retention_dir(self.retention_dir)
    
    
    def hook_configuration_retention_save(self, daemon: 'BaseSatellite', conf: dict) -> None:
        if not self.enabled:
            return
        
        if conf:
            self.logger_recorder.info(f'Saving configuration to file [ {self.configuration_file} ] for daemon {daemon.daemon_display_name}')
            # Horrible hack 1 : Daemon changes items saved conf received from Arbiter (satellites) adding unpickable objects, we must remove them
            whole_conf = {
                'conf'   : {},
                'version': daemon.get_version()
            }
            
            retention_conf = whole_conf['conf']
            with daemon.satellite_lock:
                # Clean satellites definition in conf
                for parameter_name, parameter_value in conf.items():
                    if parameter_name in self.satellite_types:
                        satellites_conf = {}
                        for satellite_id, satellite_info in parameter_value.items():
                            this_satellite_conf = {}
                            for satellite_parameter, satellite_value in satellite_info.items():
                                # Horrible hack 1 : remove unpickable parameters from definition
                                if satellite_parameter not in self.satellite_unpickable_parameters:
                                    this_satellite_conf.update({satellite_parameter: satellite_value})
                            satellites_conf.update({satellite_id: this_satellite_conf})
                        retention_conf.update({parameter_name: satellites_conf})
                    else:
                        retention_conf.update({parameter_name: parameter_value})
            
            self.retention_file_save(self.configuration_file, 'configuration', whole_conf, daemon.daemon_display_name)
    
    
    def hook_configuration_retention_load(self, daemon: 'BaseSatellite') -> None:
        if not self.enabled:
            return
        
        inventories = None
        host_mapping = None
        with self.lock:
            if not (os.path.exists(self.configuration_file) and os.path.getsize(self.configuration_file) > 0):
                self.logger_loader.debug(f'No configuration file [ {self.configuration_file} ] available for daemon {daemon.daemon_display_name}')
                return
            
            if os.path.exists(self.inventories_file) and os.path.getsize(self.inventories_file) > 0:
                self.logger_loader.info(f'loading inventories from file [ {self.inventories_file} ] for daemon {daemon.daemon_display_name}')
                try:
                    with open(self.inventories_file, 'rb') as inventories_file:
                        inventories = SafeUnpickler.load(io.BytesIO(inventories_file.read()), f'{self.name}')
                except Exception as e:
                    error_msg = self.errors['conf_load'] = f'''Could not load inventories from file '{self.inventories_file}' with error {str(e)}'''
                    self.logger_loader.error(error_msg)
                    self.logger_loader.print_stack()
                    return
                
                if not ('realms' in inventories and 'configuration_incarnation' in inventories):
                    self.logger_loader.error(f'''Could not find correct data in inventories file '{self.inventories_file}' ''')
                    return
            
            if os.path.exists(self.host_mapping_file) and os.path.getsize(self.host_mapping_file) > 0:
                self.logger_loader.info(f'loading host mapping from file [ {self.host_mapping_file} ] for daemon {daemon.daemon_display_name}')
                try:
                    with open(self.host_mapping_file, 'rb') as host_mapping_file:
                        host_mapping = SafeUnpickler.load(io.BytesIO(host_mapping_file.read()), f'{self.name}')
                except Exception as e:
                    error_msg = self.errors['conf_load'] = f'''Could not load host mapping from file '{self.host_mapping_file}' with error {str(e)}'''
                    self.logger_loader.error(error_msg)
                    self.logger_loader.print_stack()
                    return
                
                if not ('host_mapping' in host_mapping and 'configuration_incarnation' in host_mapping):
                    self.logger_loader.error(f'''Could not find correct data in host mapping file '{self.host_mapping_file}' ''')
                    return
            
            self.logger_loader.info(f'loading configuration from file [ {self.configuration_file} ] for daemon {daemon.daemon_display_name}')
            try:
                with open(self.configuration_file, 'rb') as configuration_file:
                    whole_conf = SafeUnpickler.load(io.BytesIO(configuration_file.read()), f'{self.name}')
            except Exception as e:
                error_msg = self.errors['conf_load'] = f'''Could not load configuration from configuration file '{self.configuration_file}' with error {str(e)}'''
                self.logger_loader.error(error_msg)
                self.logger_loader.print_stack()
                return
            
            retention_version = whole_conf.get('version', '')
            daemon_version = daemon.get_version()
            if retention_version != daemon_version:
                error_msg = self.errors['conf_load'] = f'''Loaded configuration version {retention_version} does not match daemon version {daemon_version}, discarding and removing it'''
                self.logger_loader.error(error_msg)
                self.retention_files_remove(self.logger_loader)
                return
            
            if 'conf_load' in self.errors:
                del self.errors['conf_load']
        
        conf = whole_conf['conf']
        
        conf['arbiter_trace']['arbiter_time'] = time.time()
        
        if inventories:
            daemon: 'WithInventorySatellite'
            daemon.push_inventories(inventories['realms'], inventories['configuration_incarnation'], received_from_arbiter=False)
            self.logger_loader.debug(f'loaded inventories from file [ {self.inventories_file} ] for daemon {daemon.daemon_display_name}')
        
        if host_mapping:
            daemon: Receiver
            daemon.push_host_mappings(host_mapping['configuration_incarnation'], host_mapping['host_mapping'], received_from_arbiter=False)
            self.logger_loader.debug(f'loaded host mapping from file [ {self.host_mapping_file} ] for daemon {daemon.daemon_display_name}')
        
        daemon.put_conf(conf, received_from_arbiter=False)
        self.logger_loader.debug(f'loaded configuration from file [ {self.configuration_file} ] for daemon {daemon.daemon_display_name}')
    
    
    def hook_inventories_save(self, daemon: 'Daemon', configuration_incarnation: 'ConfigurationIncarnation', all_inventories: 'dict[str, ElementsInventory]') -> None:
        if not self.enabled:
            return
        
        if all_inventories and configuration_incarnation:
            self.logger_recorder.info(f'Saving inventories to file [ {self.inventories_file} ] for daemon {daemon.daemon_display_name}')
            inventories = {'configuration_incarnation': configuration_incarnation, 'realms': all_inventories}
            self.retention_file_save(self.inventories_file, 'inventories', inventories, daemon.daemon_display_name)
    
    
    def hook_host_mappings_save(self, daemon: 'Receiver', configuration_incarnation: 'str', host_mapping: 'dict') -> None:
        if not self.enabled:
            return
        
        if host_mapping and configuration_incarnation:
            self.logger_recorder.info(f'Saving host mapping to file [ {self.host_mapping_file} ] for daemon {daemon.daemon_display_name}')
            retention_data = {'configuration_incarnation': configuration_incarnation, 'host_mapping': host_mapping}
            self.retention_file_save(self.host_mapping_file, 'host_mapping', retention_data, daemon.daemon_display_name)
    
    
    def hook_configuration_retention_remove(self, daemon: 'Daemon'):
        self.logger_recorder.info(f'{daemon.daemon_display_name} has been disabled by Arbiter, removing possibly recorded configuration')
        self.retention_files_remove(self.logger_recorder)
        self._remove_module_configuration()
