#!/usr/bin/env python
# -*- mode: python ; coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import logging
import re

from shinken.util import to_float, to_split, to_char, to_int, unique_value

# Suggestion
# Is this useful? see above
__author__ = "Hartmut Goebel <h.goebel@goebel-consult.de>"
__copyright__ = "Copyright 2010-2011 by Hartmut Goebel <h.goebel@goebel-consult.de>"
__licence__ = "GNU Affero General Public License version 3 (AGPL v3)"

FULL_STATUS = 'full_status'
CHECK_RESULT = 'check_result'

_boolean_states = {'1': True, 'yes': True, 'true': True, 'on': True,
                   '0': False, 'no': False, 'false': False, 'off': False}

BOOLEAN_STATES = _boolean_states

none_object = object()


class Property:
    need_pythonize = True
    """Baseclass of all properties.

    Same semantic for all subclasses (except UnusedProp): The property
    is required if, and only if, the default value is `None`.


    """
    
    
    def __init__(self,
                 default=none_object,
                 class_inherit=None,
                 unmanaged=False,
                 help='',
                 no_slots=False,
                 fill_brok=None,
                 conf_send_preparation=None,
                 brok_transformation=None,
                 retention=False,
                 retention_preparation=None,
                 to_send=False,
                 override=False,
                 managed=True,
                 split_on_coma=True,
                 merging='uniq',
                 separator=',',
                 fill_raw_data=None,
                 raw_data_transformation=None,
                 inherit=True,
                 handle_additive_inheritance=False,
                 useless_in_configuration=False,
                 is_difference_visible_by_user=True,
                 ):
        """
        `default`: default value to be used if this property is not set.
                   If default is None, this property is required.

        `class_inherit`: List of 2-tuples, (Service, 'blabla'): must
                   set this property to the Service class with name
                   blabla. if (Service, None): must set this property
                   to the Service class with same name
        `unmanaged`: ....
        `help`: usage text
        `no_slots`: do not take this property for __slots__

        `fill_brok`: if set, send to broker. There are two categories:
                     FULL_STATUS for initial and update status,
                     CHECK_RESULT for check results
        `retention`: if set, we will save this property in the retention files
        `retention_preparation`: function, if set, will go this function before
                     being save to the retention data
        `split_on_coma`: indicates that list property value should not be
                     split on coma delimiter (values contain comas that
                     we want to keep).

        Only for the initial call:

        conf_send_preparation: if set, will pass the property to this
                     function. It's used to 'flatten' some dangerous
                     properties like realms that are too 'linked' to
                     be send like that.

        brok_transformation: if set, will call the function with the
                     value of the property when flattening
                     data is necessary (like realm_name instead of
                     the realm object).

        override: for scheduler, if the property must override the
                     value of the configuration we send it

        managed: property that is managed in Nagios but not in Shinken

        merging: for merging properties, should we take only one or we can link with ,
        
        separator: For a property value containing several items, the separator between items
        TODO: Implement properly someday in ListProperty
        
        inherit: if False the property will not inherit
        
        handle_additive_inheritance: if True the property can handle additive inheritance (The +)
        
        useless_in_configuration: if True, won't be generated by the arbiter to gain time, only the scheduler will recreate them
        """
        
        self.default = default
        self.has_default = (default is not none_object)
        self.required = not self.has_default
        self.class_inherit = class_inherit or []
        self.help = help or ''
        self.unmanaged = unmanaged
        self.no_slots = no_slots
        self.fill_brok = fill_brok or []
        self.fill_raw_data = fill_raw_data or []
        self.conf_send_preparation = conf_send_preparation
        self.brok_transformation = brok_transformation
        self.raw_data_transformation = raw_data_transformation or brok_transformation
        self.retention = retention
        self.retention_preparation = retention_preparation
        self.to_send = to_send
        self.override = override
        self.managed = managed
        self.unused = False
        self.merging = merging
        self.separator = separator
        self.split_on_coma = split_on_coma
        self.inherit = inherit
        self.handle_additive_inheritance = handle_additive_inheritance
        self.useless_in_configuration = useless_in_configuration
        self.is_difference_visible_by_user = is_difference_visible_by_user
    
    
    def set_default(self, value):
        self.default = value
        self.has_default = True
        self.required = False
    
    
    def pythonize(self, value):
        return value


class UnusedProp(Property):
    """A unused Property. These are typically used by Nagios but
    no longer useful/used by Shinken.

    This is just to warn the user that the option he uses is no more used
    in Shinken.

    """
    
    need_pythonize = False
    
    
    # Since this property is not used, there is no use for other
    # parameters than 'text'.
    # 'text' a some usage text if present, will print it to explain
    # why it's no more useful
    def __init__(self, text=None):
        if text is None:
            text = 'This parameter is no longer useful in the Shinken architecture.'
        self.text = text
        self.has_default = False
        self.class_inherit = []
        self.unused = True
        self.managed = True


class BoolProp(Property):
    """A Boolean Property.

    Boolean values are currently case insensitively defined as 0,
    false, no, off for False, and 1, true, yes, on for True).
    """
    
    
    # @staticmethod
    def pythonize(self, val):
        val = unique_value(val)
        if isinstance(val, str):
            _val = _boolean_states.get(val.lower(), None)
            if _val is None:
                raise ValueError()
            return _val
        else:
            return val
    
    
    @staticmethod
    def unpythonize(val):
        if isinstance(val, str):
            val = _boolean_states.get(val.lower(), False)
        return '1' if val else '0'


class AclShareProp(Property):
    """A Property for ALC it pythonize value.
     * none     -> 00000
     * read     -> 00001
     * create   -> 00010
     * organize -> 00100
     * modify   -> 01000
     * delete   -> 10000
     * all      -> 11111
     
     mix of read, create, organize, modify, delete is possible
    """
    
    KEY_WORD = ['read', 'create', 'organize', 'modify', 'delete']
    
    
    # @staticmethod
    def pythonize(self, val):
        val = unique_value(val)
        
        # TODO Replace __DEFAULT_NO_TEMPLATE__ by a constant
        if val == '__DEFAULT_NO_TEMPLATE__':
            return val
        
        if not val:
            return ''
        
        # test if the property is already format
        if re.match(r'[0-1]{5}', val):
            return val
        elif val == 'null':
            return 'null'
        elif val == 'none':
            return '00000'
        elif val == 'all':
            return '11111'
        else:
            _val = ['0', '0', '0', '0', '0']
            for key_word in val.split(','):
                if key_word == 'read':
                    _val[4] = '1'
                if key_word == 'create':
                    _val[3] = '1'
                if key_word == 'organize':
                    _val[2] = '1'
                if key_word == 'modify':
                    _val[1] = '1'
                if key_word == 'delete':
                    _val[0] = '1'
            return ''.join(_val)
    
    
    @staticmethod
    def unpythonize(val):
        val = unique_value(val)
        if not val:
            return ''
        
        if re.match(r'[0-1]{5}', val):
            if val == '00000':
                return 'none'
            elif val == '11111':
                return 'all'
            else:
                _val = []
                if val[4] == '1':
                    _val.append('read')
                if val[3] == '1':
                    _val.append('create')
                if val[2] == '1':
                    _val.append('organize')
                if val[1] == '1':
                    _val.append('modify')
                if val[0] == '1':
                    _val.append('delete')
                return ','.join(_val)
        else:
            return val


class IntegerProp(Property):
    """Please Add a Docstring to describe the class here"""
    
    
    # @staticmethod
    def pythonize(self, val):
        val = unique_value(val)
        return to_int(val)


class EpochProp(Property):
    pass


class FloatProp(Property):
    """Please Add a Docstring to describe the class here"""
    
    
    # @staticmethod
    def pythonize(self, val):
        val = unique_value(val)
        return to_float(val)


class CharProp(Property):
    """Please Add a Docstring to describe the class here"""
    
    
    # @staticmethod
    def pythonize(self, val):
        val = unique_value(val)
        return to_char(val)


class StringProp(Property):
    """Please Add a Docstring to describe the class here"""
    
    
    # @staticmethod
    def pythonize(self, val):
        val = unique_value(val)
        return val


class LinkToAnotherObjectProp(Property):
    need_pythonize = False
    
    
    # @staticmethod
    def pythonize(self, val):
        return val.strip()


class LinkToOthersObjectsProp(Property):
    # @staticmethod
    def pythonize(self, val):
        return to_split(val, self.split_on_coma)


class RawProp(Property):
    need_pythonize = False


class VirtualProp(Property):
    need_pythonize = False


class PathProp(StringProp):
    """ A string property representing a "running" (== VAR) file path """


class ConfigPathProp(StringProp):
    """ A string property representing a config file path """


class ListProp(Property):
    """Please Add a Docstring to describe the class here"""
    
    
    # @staticmethod
    def pythonize(self, val):
        return to_split(val, self.split_on_coma)


class EditableListProp(Property):
    """Please Add a Docstring to describe the class here"""
    
    
    # @staticmethod
    def pythonize(self, val):
        return list(to_split(val, self.split_on_coma))  # list because this one we want to modify it


class LogLevelProp(StringProp):
    """ A string property representing a logging level """
    
    
    def pythonize(self, val):
        val = unique_value(val)
        return logging.getLevelName(val)


class DictProp(Property):
    def __init__(self, elts_prop=None, *args, **kwargs):
        """Dictionary of values.
             If elts_prop is not None, must be a Property subclass
             All dict values will be casted as elts_prop values when pythonized

            elts_prop = Property of dict members
        """
        super(DictProp, self).__init__(*args, **kwargs)
        
        if elts_prop is not None and not issubclass(elts_prop, Property):
            raise TypeError("DictProp constructor only accept Property sub-classes as elts_prop parameter")
        self.elts_prop = elts_prop()
    
    
    def pythonize(self, val):
        val = unique_value(val)
        
        
        def split(kv):
            m = re.match(r"^\s*([^\s]+)\s*=\s*([^\s]+)\s*$", kv)
            if m is None:
                raise ValueError
            
            return (
                m.group(1),
                # >2.4 only. we keep it for later. m.group(2) if self.elts_prop is None else self.elts_prop.pythonize(m.group(2))
                (self.elts_prop.pythonize(m.group(2)), m.group(2))[self.elts_prop is None]
            )
        
        
        if val is None:
            return dict()
        
        # val is in the form "key1=addr:[port],key2=addr:[port],..."
        return dict([split(kv) for kv in to_split(val)])


class AddrProp(Property):
    """Address property (host + port)"""
    
    
    def pythonize(self, val):
        """
            i.e: val = "192.168.10.24:445"
            NOTE: port is optional
        """
        val = unique_value(val)
        m = re.match(r"^([^:]*)(?::(\d+))?$", val)
        if m is None:
            raise ValueError
        
        addr = {'address': m.group(1)}
        if m.group(2) is not None:
            addr['port'] = int(m.group(2))
        
        return addr
