#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import argparse
import signal
import sys
import traceback

from shinken.basesubprocess import LookAtMyFatherThread
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.runtime_stats.memory_stats import MemoryStats
from shinken.runtime_stats.threads_dumper import thread_dumper
from shinken.util import set_process_name, start_malloc_trim_thread
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory
from shinken.log import logger, LoggerFactory

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Iterable, Dict, Optional
    
    from types import FrameType

ON_LINUX = sys.platform.startswith('linux')


class ShinkenProcessAbstract:
    
    def __init__(self):
        # type: () -> None
        
        self.name = None
        self.father_pid = None
        self.father_name = None
        
        self.logger = None  # type: Optional[PartLogger]
        self.args = None
        self.data_hub_exchange_with_father = None
        self.process_configuration = None
        self.look_at_father_thread = None
        self.interrupted = False
        self.configuration_id = ''
        self.result_id = ''
    
    
    def read_configuration(self, parsed_args):
        # type: (Dict[str, str]) -> None
        
        self.data_hub_exchange_with_father = DataHubFactory.build_in_memory_exchange_json_data_hub(
            logger=self.logger,
            data_type=parsed_args['data_type'],
            data_id_key_name=parsed_args['data_id_key_name'],
            daemon_name=parsed_args['daemon_name'],
            module_name=parsed_args['module_name'],
        )
        self.configuration_id = parsed_args['configuration_id']
        self.result_id = parsed_args['result_id']
        
        self.process_configuration = self.data_hub_exchange_with_father.get_data(self.configuration_id)
    
    
    def get_process_name(self):
        # type: () -> str
        return '%s [ %s ]' % (self.father_name, self.name)
    
    
    def manage_signal(self, sig, _frame):
        # type: (int, FrameType) -> None
        if not ON_LINUX:
            return
        if sig == signal.SIGUSR1:  # if USR1, ask a memory dump
            MemoryStats.dump_memory_full_memory_dump(self.name)
            MemoryStats.print_memory_stats()
        elif sig == signal.SIGPWR:  # SIGPWR (old signal not used) dump all threads stacks
            thread_dumper.dump_all_threads()
        else:
            self.interrupted = True
    
    
    def set_signal_handler(self, sigs=None):
        # type: (Iterable[int]) -> None
        if not ON_LINUX:
            return
        
        if sigs is None:
            sigs = (signal.SIGINT, signal.SIGTERM, signal.SIGUSR1, signal.SIGPWR)
        
        for sig in sigs:
            signal.signal(sig, self.manage_signal)
    
    
    def _start_sub_process_threads(self):
        # type: () -> None
        start_malloc_trim_thread()
        self.look_at_father_thread.start_thread()
    
    
    def setting_up_process_configuration(self):
        if not self.args.logger_file:
            raise Exception('The argument logger_file is required')
        if not self.args.process_name:
            raise Exception('The argument process_name is required')
        if not self.args.father_name:
            raise Exception('The argument father_name is required')
        if not self.args.father_pid:
            raise Exception('The argument father_pid is required')
        
        logger.register_local_log(self.args.logger_file)
        logger.set_name(self.args.daemon_name)
        self.name = self.args.process_name
        self.father_name = self.args.father_name
        self.father_pid = int(self.args.father_pid)
        
        self.logger = LoggerFactory.get_logger()
        for logger_name in self.args.logger_names:
            self.logger = self.logger.get_sub_part(logger_name)
        
        self.look_at_father_thread = LookAtMyFatherThread(self.father_pid, self.father_name, self.name, logger=self.logger)
        set_process_name(self.get_process_name())
        self.set_signal_handler()
    
    
    def parse_communication_args(self):
        # type: () -> Dict[str, str]
        return self._parse_args_data_hub_configuration()
    
    
    def parse_args(self):
        # type: () -> Dict[str, str]
        return self.parse_communication_args()
    
    
    def _parse_args_data_hub_configuration(self):
        # type: () -> Dict[str, str]
        data_hub_configuration = self.args.data_hub_configuration
        if not data_hub_configuration:
            raise Exception('The argument data_hub_configuration is required')
        data_hub_configuration = data_hub_configuration.split(';')
        if len(data_hub_configuration) != 6:
            raise Exception('The argument data_hub_configuration is invalid, use --help for more information')
        
        data_hub_configuration = {
            'data_type'       : data_hub_configuration[0],
            'data_id_key_name': data_hub_configuration[1],
            'daemon_name'     : data_hub_configuration[2],
            'module_name'     : data_hub_configuration[3],
            'configuration_id': data_hub_configuration[4],
            'result_id'       : data_hub_configuration[5],
        }
        return data_hub_configuration
    
    
    def update_configuration(self):
        # type: () -> None
        self.name = self.process_configuration['name']
        self.father_pid = self.process_configuration['father_pid']
        self.father_name = self.process_configuration['father_name']
    
    
    def init(self):
        # type: () -> None
        parser = self.build_arg_parser()
        self.args = parser.parse_args()
        
        self.setting_up_process_configuration()
        
        parsed_args = self.parse_args()
        self.read_configuration(parsed_args)
        self.update_configuration()
        
        # Our sub process need some utility threads, like:
        # * malloc trimming
        # * looking at father process (if dead, we die)
        # * others set by the subclass
        self._start_sub_process_threads()
    
    
    def _main(self):
        # type: () -> None
        try:
            self.main()
        except Exception as e:
            formatted_lines = traceback.format_exc().splitlines()
            first_line = formatted_lines[0]
            if first_line == 'None':
                for line in traceback.format_stack():
                    print(line)
            else:
                print(first_line)
                for line in formatted_lines[1:]:
                    print(line)
            
            print(e)
    
    
    def main(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    @staticmethod
    def build_arg_parser():
        # type: () -> argparse.ArgumentParser
        parser = argparse.ArgumentParser(description='Standard Arguments for talking with process')
        parser.add_argument('-i', '--ip', dest='ip', default='', help='Ip of the socket to communicate with the process.')
        parser.add_argument('-p', '--port', dest='port', default='', help='Port of the socket to communicate with the process')
        parser.add_argument('-d', '--data_hub_configuration', dest='data_hub_configuration', default='',
                            help='Configuration to create the Datahub to read the configuration of the process. We need "data_type, data_id_key_name, daemon_name, module_name, submodule_name, id of the data to get". Separated by a ;')
        parser.add_argument('-l', '--logger_file', dest='logger_file', help='Path to the log file')
        parser.add_argument('--logger_name', dest='logger_names', default=[], action='append', help='Name of the logger')
        parser.add_argument('-n', '--process_name', dest='process_name', help='Name of the process')
        parser.add_argument('-f', '--father_name', dest='father_name', help='Name of the father')
        parser.add_argument('-t', '--father_pid', dest='father_pid', help='Pid of the father')
        parser.add_argument('-e', '--daemon_name', dest='daemon_name', help='Name of the daemon father')
        
        return parser
    
    
    def start(self):
        # type: () -> None
        self._main()
