#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2024:

# This file is part of Shinken Enterprise, all rights reserved.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

from cheroot.server import get_ssl_adapter_class
from cheroot.wsgi import WSGIServer

import shinken.webui.bottlewebui as bottle
from shinken.log import LoggerFactory, WARNING

wsgi_logger = LoggerFactory.get_logger().get_sub_part('WSGI')


class ShinkenWSGI(WSGIServer):
    def error_log(self, msg='', level=20, traceback=False):
        wsgi_logger.warning(msg)
        if traceback:
            wsgi_logger.print_stack(level=WARNING)


# CherryPy is allowing us to have an HTTP 1.1 server, and so have a KeepAlive
class CherryPyServerHTTP(bottle.ServerAdapter):
    quiet = True
    
    
    def run(self, handler):  # pragma: no cover
        daemon_thread_pool_size = self.options.get('numthreads', 64)
        server = ShinkenWSGI((self.host, self.port), handler, numthreads=daemon_thread_pool_size, shutdown_timeout=1, timeout=300)
        wsgi_logger.info('Initializing a Cheroot backend with %d threads' % daemon_thread_pool_size)
        use_ssl = self.options['use_ssl']
        ssl_cert = self.options['ssl_cert']
        ssl_key = self.options['ssl_key']
        
        if use_ssl:
            server.ssl_adapter = get_ssl_adapter_class()(certificate=ssl_cert, private_key=ssl_key)
        
        return server
