#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.modules.base_module.basemodule import SOURCE_STATE
from shinken.log import PartLogger
from shinken.misc.type_hint import Optional, TYPE_CHECKING
from shinkensolutions.api.synchronizer import ComponentManagerSyncui
from shinkensolutions.api.synchronizer.http_lib_external.v01_00 import SELECTOR_CONST, instance_selector_distributor, HtmlDistributor, LabelHtml
from shinkensolutions.api.synchronizer.source.route.route import AbstractRoute
from shinkensolutions.api.synchronizer.source.tab.tab import AbstractTab

if TYPE_CHECKING:
    from shinkensolutions.api.synchronizer.source.rules_application_template import SourceRulesManager
    from shinkensolutions.api.synchronizer import SourceTranslatePart
    from shinken.basesubprocess import EventHandler


class TabHostTemplateBindingRules(AbstractTab):
    
    def __init__(self, logger, translator, source_name, host_template_binding_rule_manager):
        # type: (PartLogger, SourceTranslatePart, str, SourceRulesManager ) -> None
        super(TabHostTemplateBindingRules, self).__init__(logger, translator, source_name, 'host_template_binding_rules', translator.translate('host_template_binding_rules.tab_display_name'))
        self.host_template_binding_rule_manager = host_template_binding_rule_manager
        self.html_distributor = None  # type: Optional[HtmlDistributor]
    
    
    def controller(self):
        common_translator = ComponentManagerSyncui.get_translate_component().translator()
        for rule in self.host_template_binding_rule_manager.rules:
            rule.compute_validation_message_to_html()
        return {
            '__shinken_lang__'   : self.translator.get_lang(),
            'tab_info'           : self,
            'source_name'        : self.source_name,
            'translate'          : self.translator.translate,
            'common_translator'  : common_translator,
            'rules'              : self.host_template_binding_rule_manager.rules,
            'html_distributor'   : self.init_html_object(),
            'validation_state'   : self.host_template_binding_rule_manager.validation_state.as_html(),
            'files_configuration': self.host_template_binding_rule_manager.user_rules_files
        }
    
    
    def init_html_object(self):
        # type: ()-> HtmlDistributor
        if not self.html_distributor:
            _to_return = HtmlDistributor(self.translator)
            # BUTTONS
            _to_return.add_button('reload_rules', 'reload_rules', LabelHtml('''<span class="shinicon-undo"></span>'''))
            # FILTERS
            _to_return.add_list_filter('define_by', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_DEFINE_BY))
            _to_return.add_list_filter('index', placeholder='--')
            _to_return.add_list_filter('name', placeholder='--')
            _to_return.add_list_filter('conditions', placeholder='--')
            _to_return.add_list_filter('templates', placeholder='--')
            _to_return.add_list_filter('counters', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_COUNTERS))
            self.html_distributor = _to_return
        return self.html_distributor


class RouteReloadTemplateBindingRules(AbstractRoute):
    
    def __init__(self, logger, source_name, host_template_binding_rule_manager, host_template_binding_rule_event):
        # type: (PartLogger, str, SourceRulesManager, EventHandler) -> None
        super(RouteReloadTemplateBindingRules, self).__init__(logger, source_name, 'template_binding_rules/reload', have_http_template=False)
        self.host_template_binding_rule_manager = host_template_binding_rule_manager
        self.host_template_binding_rule_event = host_template_binding_rule_event
    
    
    def controller(self):
        # type: () -> str
        _source_info = ComponentManagerSyncui.get_source_info_component().get_source_info(self.source_name)
        if _source_info.is_import_in_progress():
            return SOURCE_STATE.RUNNING
        
        self.host_template_binding_rule_manager.set_translator(ComponentManagerSyncui.get_translate_component().source_translator(self.source_name))
        self.host_template_binding_rule_manager.reload_rule()
        self.host_template_binding_rule_event.send_event()
        return SOURCE_STATE.OK
